#' @title Coalition excesses
#' @description Given a game and an allocation, this function computes the excess of each coalition.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param x An allocation, as a vector.
#' @return The excesses of all coalitions, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Given a game \eqn{v\in G^N} and an allocation \eqn{x}, the excess of coalition \eqn{S \in 2^N}
#' with respect to \eqn{x} is defined as \eqn{e(x,S)=v(S)-x(S)}, where \eqn{x(S)=\sum_{i\in S} x_i}.
#' @examples
#' excesses(v=c(0,0,3,0,3,8,6,0,6,9,15,8,16,17,20), binary=TRUE, x=c(8,7,2,3))
#' excesses(v=c(1,5,10,6,11,15,16), x=c(1,5,10)) <= 0 # core allocation
#' @seealso \link{nucleolusvalue}, \link{nucleoluspcvalue}
#' @export

excesses <- function(v,binary=FALSE,x) {

  nC=length(v)
  n=log(nC+1)/log(2)

  ##################
  # comprobaciones #
  ##################

  if (n>floor(n))
    stop("'v' must have length 2^n-1 for some n.")

  if(length(x)!=n)
    stop("The dimension of 'x' must be the number of players in 'v'.")

  if(sum(x)!=v[nC])
    warning("'x' is not an efficient allocation.")

  ###################
  # cálculo excesos #
  ###################

  e<-c()

  if (binary == FALSE) { # Cálculos en lexicográfico.
    for (S in 1:nC) {
      Sjug <- getcoalition(codelex2bin(n=n,Nlex=S)) # calculamos quiénes están en S
      e[S] <- v[S]-sum(x[Sjug])
    }
  } else { # Cálculos en binario.
    for (S in 1:nC) {
      Sjug <- getcoalition(S)
      e[S] <- v[S]-sum(x[Sjug])
    }
  }

  ##########
  # salida #
  ##########

  return(e)

}
