% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/localPOET.R
\name{time_varying_cov}
\alias{time_varying_cov}
\title{Estimate Time-Varying Covariance Matrix Using Local PCA}
\usage{
time_varying_cov(
  obj,
  max_factors = 3,
  kernel_func = epanechnikov_kernel,
  M0 = 10,
  rho_grid = seq(0.005, 2, length.out = 30),
  floor_value = 1e-12,
  epsilon2 = 1e-06,
  full_output = FALSE
)
}
\arguments{
\item{obj}{An object of class TVMVP with the data.}

\item{max_factors}{The number of factors to use in local PCA.}

\item{kernel_func}{The kernel function to use (default is \code{\link{epanechnikov_kernel}}).}

\item{M0}{Integer. The number of observations to leave out between the two sub-samples in the adaptive thresholding procedure. Default is 10.}

\item{rho_grid}{A numeric vector of candidate shrinkage parameters \eqn{\rho} used in \code{\link{adaptive_poet_rho}}. Default is \code{seq(0.005, 2, length.out = 30)}.}

\item{floor_value}{A small positive number specifying the lower bound for eigenvalues in the final positive semidefinite repair. Default is \code{1e-12}.}

\item{epsilon2}{A small positive tuning parameter for the adaptive thresholding. Default is \code{1e-6}.}

\item{full_output}{Logical; if \code{TRUE}, returns all components of the estimation.}
}
\value{
By default, a covariance matrix. If \code{full_output = TRUE}, a list containing:
\itemize{
  \item \code{total_cov} – the estimated covariance matrix,
  \item \code{residual_cov} – the residual (idiosyncratic) covariance,
  \item \code{loadings} – estimated factor loadings,
  \item \code{best_rho} – optimal shrinkage parameter,
  \item \code{naive_resid_cov} – residual covariance before shrinkage
}
}
\description{
This function estimates a time-varying covariance matrix using local principal component
analysis and the soft thresholding for residual shrinkage. By default, only the total
covariance matrix is returned. Optionally, the user can retrieve all intermediate
components of the estimation process. The procedure is available either as a
stand-alone function or as a method in the `TVMVP` R6 class.
}
\details{
The function estimates a time-varying covariance matrix using Local PCA:
\deqn{\hat{\Sigma}_{r,t}=\hat{\Lambda}_t \hat{\Sigma}_F \hat{\Lambda}_t' + \tilde{\Sigma}_e}
Where \eqn{\hat{\Lambda}_t} is the factor loadings at time t, \eqn{\hat{\Sigma}_F} is the factor covariance matrix, and \eqn{\tilde{\Sigma}_e} is regularized covariance matrix of the idiosyncratic errors.

Two usage styles:

\preformatted{
# Function interface
tv <- TVMVP$new()
tv$set_data(returns)
cov <- time_varying_cov(tv, m = 5)

# R6 method interface
tv$determine_factors(max_m = 5)
cov <- tv$time_varying_cov()
}
}
\section{References}{
 
Lillrank, E. (2025). \ifelse{html}{
    \out{<a href='../doc/thesis.pdf'>A Time-Varying Factor Approach to Covariance Estimation</a>}
  }{Master’s thesis (PDF in inst/doc)}
  
Chen, J., Li, D., & Linton, O. (2019). A new semiparametric estimation approach for large dynamic covariance matrices with multiple conditioning variables. Journal of Econometrics, 212(1), 155–176.

Fan, Q., Wu, R., Yang, Y., & Zhong, W. (2024). Time-varying minimum variance portfolio. Journal of Econometrics, 239(2), 105339.
}

\examples{
\donttest{
set.seed(123)
returns <- matrix(rnorm(100 * 30), nrow = 100, ncol = 30)

# Initialize object
tv <- TVMVP$new()
tv$set_data(returns)

# Using function interface
time_varying_cov(obj = tv, m = 3)

# Or using R6 method
tv$time_varying_cov(m=3)
}


}
