% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netclu_leiden.R
\name{netclu_leiden}
\alias{netclu_leiden}
\title{Finding communities using the Leiden algorithm}
\usage{
netclu_leiden(
  net,
  weight = TRUE,
  cut_weight = 0,
  index = names(net)[3],
  seed = NULL,
  objective_function = "CPM",
  resolution_parameter = 1,
  beta = 0.01,
  n_iterations = 2,
  vertex_weights = NULL,
  bipartite = FALSE,
  site_col = 1,
  species_col = 2,
  return_node_type = "both",
  algorithm_in_output = TRUE
)
}
\arguments{
\item{net}{The output object from \code{\link[=similarity]{similarity()}} or
\code{\link[=dissimilarity_to_similarity]{dissimilarity_to_similarity()}}. If a \code{data.frame} is used, the first two
columns represent pairs of sites (or any pair of nodes), and the next
column(s) are the similarity indices.}

\item{weight}{A \code{boolean} indicating if the weights should be considered
if there are more than two columns.}

\item{cut_weight}{A minimal weight value. If \code{weight} is TRUE, the links
between sites with a weight strictly lower than this value will not be
considered (\code{0} by default).}

\item{index}{The name or number of the column to use as weight. By default,
the third column name of \code{net} is used.}

\item{seed}{The random number generator seed (NULL for random by default).}

\item{objective_function}{A string indicating the objective function to use,
either the Constant Potts Model ("CPM") or "modularity" ("CPM" by default).}

\item{resolution_parameter}{The resolution parameter to use. Higher
resolutions lead to smaller communities, while lower resolutions lead
to larger communities.}

\item{beta}{A parameter affecting the randomness in the Leiden algorithm. This
affects only the refinement step of the algorithm.}

\item{n_iterations}{The number of iterations for the Leiden
algorithm. Each iteration may further improve the partition.}

\item{vertex_weights}{The vertex weights used in the Leiden algorithm. If
not provided, they will be automatically determined based on the
objective_function. Please see the details of this function to understand how
to interpret the vertex weights.}

\item{bipartite}{A \code{boolean} indicating if the network is bipartite
(see Details).}

\item{site_col}{The name or number for the column of site nodes
(i.e., primary nodes).}

\item{species_col}{The name or number for the column of species nodes
(i.e., feature nodes).}

\item{return_node_type}{A \code{character} indicating what types of nodes
("site", "species", or "both") should be returned in the output
(\code{"both"} by default).}

\item{algorithm_in_output}{A \code{boolean} indicating if the original output
of \link[igraph:cluster_leiden]{cluster_leiden} should be returned in the
output (\code{TRUE} by default, see Value).}
}
\value{
A \code{list} of class \code{bioregion.clusters} with five slots:
\enumerate{
\item{\strong{name}: A \code{character} containing the name of the algorithm.}
\item{\strong{args}: A \code{list} of input arguments as provided by the user.}
\item{\strong{inputs}: A \code{list} of characteristics of the clustering process.}
\item{\strong{algorithm}: A \code{list} of all objects associated with the
clustering procedure, such as original cluster objects (only if
\code{algorithm_in_output = TRUE}).}
\item{\strong{clusters}: A \code{data.frame} containing the clustering results.}}

In the \code{algorithm} slot, if \code{algorithm_in_output = TRUE}, users can
find the output of \link[igraph:cluster_leiden]{cluster_leiden}.
}
\description{
This function finds communities in a (un)weighted undirected network based
on the Leiden algorithm of Traag, van Eck & Waltman.
}
\details{
This function is based on the Leiden algorithm
(Traag et al., 2019) as implemented in the
\href{https://cran.r-project.org/package=igraph}{igraph}
package (\link[igraph:cluster_leiden]{cluster_leiden}).
}
\note{
Although this algorithm was not primarily designed to deal with bipartite
networks, it is possible to consider the bipartite network as a unipartite
network (\code{bipartite = TRUE}).

Do not forget to indicate which of the first two columns is
dedicated to the site nodes (i.e., primary nodes) and species nodes (i.e.
feature nodes) using the arguments \code{site_col} and \code{species_col}.
The type of nodes returned in the output can be chosen with the argument
\code{return_node_type} equal to \code{"both"} to keep both types of nodes,
\code{"site"} to preserve only the site nodes, and \code{"species"} to
preserve only the species nodes.
}
\examples{
comat <- matrix(sample(1000, 50), 5, 10)
rownames(comat) <- paste0("Site", 1:5)
colnames(comat) <- paste0("Species", 1:10)

net <- similarity(comat, metric = "Simpson")
com <- netclu_leiden(net)

net_bip <- mat_to_net(comat, weight = TRUE)
clust2 <- netclu_leiden(net_bip, bipartite = TRUE)

}
\references{
Traag VA, Waltman L & Van Eck NJ (2019) From Louvain to Leiden: guaranteeing
well-connected communities. \emph{Scientific reports} 9, 5233.
}
\seealso{
For more details illustrated with a practical example,
see the vignette:
\url{https://biorgeo.github.io/bioregion/articles/a4_3_network_clustering.html}.

Associated functions:
\link{netclu_infomap} \link{netclu_louvain} \link{netclu_oslom}
}
\author{
Maxime Lenormand (\email{maxime.lenormand@inrae.fr}) \cr
Pierre Denelle (\email{pierre.denelle@gmail.com}) \cr
Boris Leroy (\email{leroy.boris@gmail.com})
}
