! DESCRIPTION: 
!
!    These functions are due to minor modifications from the glmnet package:
!
!    Jerome Friedman, Trevor Hastie, Robert Tibshirani (2010). 
!    Regularization Paths for Generalized Linear Models via Coordinate Descent. 
!    Journal of Statistical Software, 33(1), 1-22. 
!    URL http://www.jstatsoft.org/v33/i01/.
!
! --------------------------------------------------------------------------
! standard: An auxiliary function for standardize x matrix.
! --------------------------------------------------------------------------
!
! USAGE:
! 
! call standard (nobs,nvars,x,ju,isd,xmean,xnorm,maj)   
! 
! INPUT ARGUMENTS:
! 
!    nobs = number of observations
!    nvars = number of predictor variables
!    x(nobs, nvars) = matrix of predictors, of dimension N * p; each row is an observation vector.
!    ju(nvars) = flag of predictor variables
!                ju(j) = 0 => this predictor has zero variance
!                ju(j) = 1 => this predictor does not have zero variance
!    isd = standarization flag:
!          isd = 0 => do not standardize predictor variables
!          isd = 1 => standardize predictor variables
!          NOTE: no matter isd is 1 or 0, matrix x is always centered by column. That is, col.mean(x) = 0.
!    
! OUTPUT:
!
!    x(nobs, nvars) = standarized matrix x
!    xmean(nvars) = column mean of x matrix
!    xnorm(nvars) = column standard deviation of x matrix
!    maj(nvars) = column variance of x matrix
!
! --------------------------------------------------
SUBROUTINE standard(nobs,nvars,x,ju,isd,xmean,xnorm,maj)     
! --------------------------------------------------
    IMPLICIT NONE
    ! - - - arg types - - -
    INTEGER :: nobs
    INTEGER :: nvars
    INTEGER :: isd
    INTEGER :: ju(nvars)
    DOUBLE PRECISION :: x(nobs,nvars)
    DOUBLE PRECISION :: xmean(nvars)
    DOUBLE PRECISION :: xnorm(nvars)
    DOUBLE PRECISION :: maj(nvars)
    ! - - - local declarations - - -
    INTEGER:: j
! - - - begin - - -                                
    DO j = 1,nvars                                  
        IF (ju(j) == 1) THEN                         
            xmean(j) = sum(x(:,j)) / nobs     !mean                        
            x(:,j) = x(:,j) - xmean(j)    
            maj(j) = dot_product(x(:,j),x(:,j))/nobs                                              
              IF (isd == 1) THEN
                xnorm(j) = sqrt(maj(j))    !standard deviation               
                x(:,j) = x(:,j)/xnorm(j)
                maj(j) = 1.0D0
            ENDIF                                                        
        ENDIF                                     
    ENDDO                             
END SUBROUTINE standard
! --------------------------------------------------------------------------
! chkvars: An auxiliary function for variable check.
! --------------------------------------------------------------------------
!
! USAGE:
! 
! call chkvars (nobs, nvars, x, ju)
! 
! INPUT ARGUMENTS:
! 
!    nobs = number of observations
!    nvars = number of predictor variables
!    x(nobs, nvars) = matrix of predictors, of dimension N * p; each row is an observation vector.
!    y(no) = response variable. This argument should be a two-level factor {-1, 1} 
!            for classification.
!    
! OUTPUT:
!
!    ju(nvars) = flag of predictor variables
!                ju(j) = 0 => this predictor has zero variance
!                ju(j) = 1 => this predictor does not have zero variance
!
! --------------------------------------------------
SUBROUTINE chkvars (nobs, nvars, x, ju)
! --------------------------------------------------
      IMPLICIT NONE
    ! - - - arg types - - -
      INTEGER :: nobs
      INTEGER :: nvars
      INTEGER :: ju (nvars)
      DOUBLE PRECISION :: x (nobs, nvars)
    ! - - - local declarations - - -
      INTEGER :: i
      INTEGER :: j
      DOUBLE PRECISION :: t
! - - - begin - - -
      DO j = 1, nvars
         ju (j) = 0
         t = x (1, j)
         DO i = 2, nobs
            IF (x(i, j) /= t) THEN
               ju (j) = 1
               EXIT
            END IF
         END DO
      END DO
END SUBROUTINE chkvars

! imported from R source code
! --------------------------------------------------
SUBROUTINE pnorm(qval, pval)
! --------------------------------------------------
      IMPLICIT NONE

      INTEGER :: ii
      DOUBLE PRECISION :: qval, pval
      DOUBLE PRECISION :: y, xsq, xnum, xden, tmp, del
      DOUBLE PRECISION, PARAMETER :: eps = 1.0D-16
      DOUBLE PRECISION, PARAMETER :: a(5) = (/ &
        & 2.2352520354606839287, &
        & 161.02823106855587881, &
        & 1067.6894854603709582, &
        & 18154.981253343561249, &
        & 0.065682337918207449113 &
        & /)
      DOUBLE PRECISION, PARAMETER :: b(4) = (/ &
        & 47.20258190468824187, & 
        & 976.09855173777669322, & 
        & 10260.932208618978205, & 
        & 45507.789335026729956 & 
        & /)
      DOUBLE PRECISION, PARAMETER :: c(8) = (/ &
        & 0.39894151208813466764, &
        & 8.8831497943883759412, &
        & 93.506656132177855979, &
        & 597.27027639480026226, &
        & 2494.5375852903726711, &
        & 6848.1904505362823326, &
        & 11602.651437647350124, &
        & 9842.7148383839780218  &
        & /)
      DOUBLE PRECISION :: clast = 1.0765576773720192317D-8
      DOUBLE PRECISION, PARAMETER :: d(8) = (/ &
        & 22.266688044328115691, &
        & 235.38790178262499861, &
        & 1519.377599407554805, &
        & 6485.558298266760755, &
        & 18615.571640885098091, &
        & 34900.952721145977266, &
        & 38912.003286093271411, &
        & 19685.429676859990727 &
        & /)

      y = Abs(qval)
      del = 0.0
      IF (y .LE. 0.67448975) THEN
        IF (y > eps) THEN
          xsq = qval * qval
          xnum = a(5) * xsq
          xden = xsq;
          DO ii = 1, 3
            xnum = (xnum + a(ii)) * xsq
            xden = (xden + b(ii)) * xsq
          ENDDO
        ELSE
          xnum = 0.0
          xden = 0.0
        ENDIF
        tmp = qval * (xnum + a(4)) / (xden + b(4))
        pval = 0.5 + tmp
      ELSEIF (y .LE. 5.656854) THEN
        xnum = clast * y
        xden = y
        DO ii = 1, 7
          xnum = (xnum + c(ii)) * y
          xden = (xden + d(ii)) * y
        ENDDO
        tmp = (xnum + c(8)) / (xden + d(8))
        xsq = Aint(qval * 16) / 16
        del = (qval - xsq) * (qval + xsq)
        pval = Exp(-xsq * xsq * 0.5) * Exp(-del * 0.5) * tmp
        if (qval > 0.0) pval = 1.0 - pval
      ELSE
        IF (qval > 0.0) pval = 1.0
        IF (qval < 0.0) pval = 0.0
      ENDIF
END SUBROUTINE pnorm      




