% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Constraint.R
\name{Constraint}
\alias{Constraint}
\title{Define a custom \code{Constraint} class}
\usage{
Constraint(
  classname,
  call = NULL,
  get_config = NULL,
  ...,
  public = list(),
  private = list(),
  inherit = NULL,
  parent_env = parent.frame()
)
}
\arguments{
\item{classname}{String, the name of the custom class. (Conventionally, CamelCase).}

\item{call}{\if{html}{\out{<div class="sourceCode r">}}\preformatted{\\(w)
}\if{html}{\out{</div>}}

Applies the constraint to the input weight variable.

By default, the inputs weight variable is not modified.
Users should override this method to implement their own projection
function.

Args:
\itemize{
\item \code{w}: Input weight variable.
}

Returns:
Projected variable (by default, returns unmodified inputs).}

\item{get_config}{\if{html}{\out{<div class="sourceCode r">}}\preformatted{\\()
}\if{html}{\out{</div>}}

Function that returns a named list of the object config.

A constraint config is a named list (JSON-serializable) that can
be used to reinstantiate the same object
(via \verb{do.call(<constraint_class>, <config>)}).}

\item{..., public}{Additional methods or public members of the custom class.}

\item{private}{Named list of R objects (typically, functions) to include in
instance private environments. \code{private} methods will have all the same
symbols in scope as public methods (See section "Symbols in Scope"). Each
instance will have it's own \code{private} environment. Any objects
in \code{private} will be invisible from the Keras framework and the Python
runtime.}

\item{inherit}{What the custom class will subclass. By default, the base keras class.}

\item{parent_env}{The R environment that all class methods will have as a grandparent.}
}
\value{
A function that returns \code{Constraint} instances, similar to the
builtin constraint functions like \code{constraint_maxnorm()}.
}
\description{
Base class for weight constraints.

A \code{Constraint()} instance works like a stateless function.
Users who subclass the \code{Constraint} class should override
the \code{call()} method, which takes a single
weight parameter and return a projected version of that parameter
(e.g. normalized or clipped). Constraints can be used with various Keras
layers via the \code{kernel_constraint} or \code{bias_constraint} arguments.

Here's a simple example of a non-negative weight constraint:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{constraint_nonnegative <- Constraint("NonNegative",
  call = function(w) \{
    w * op_cast(w >= 0, dtype = w$dtype)
  \}
)
weight <- op_convert_to_tensor(c(-1, 1))
constraint_nonnegative()(weight)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([-0.  1.], shape=(2), dtype=float64)

}\if{html}{\out{</div>}}

Usage in a layer:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{layer_dense(units = 4, kernel_constraint = constraint_nonnegative())
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## <Dense name=dense, built=False>
##  signature: (*args, **kwargs)

}\if{html}{\out{</div>}}
}
\section{Symbols in scope}{
All R function custom methods (public and private) will have the
following symbols in scope:
\itemize{
\item \code{self}: The custom class instance.
\item \code{super}: The custom class superclass.
\item \code{private}: An R environment specific to the class instance.
Any objects assigned here are invisible to the Keras framework.
\item \verb{__class__} and \code{as.symbol(classname)}: the custom class type object.
}
}

\seealso{
Other constraints: \cr
\code{\link{constraint_maxnorm}()} \cr
\code{\link{constraint_minmaxnorm}()} \cr
\code{\link{constraint_nonneg}()} \cr
\code{\link{constraint_unitnorm}()} \cr
}
\concept{constraints}
