#' Gives default initial guesses for logistic-normal multinomial Factor analyzer algorithm.
#'
#'@param W_count The microbiome count matrix that you want to analyze.
#'@param G The number of component
#'@param Q_g The number of latent dimensions for each component, a vector.
#'@param cov_str The covaraince structure you choose, there are 8 different models belongs to
#'this family:UUU, UUG, UUD, UUC, GUU, GUG, GUD, GUC.
#'@param X The regression covariates matrix, which generated by model.matrix.
#'
#'@return \code{new_pi_g} Initial guess of proportion
#'@return \code{new_mu_g} Initial guess of mean vector
#'@return \code{new_sig_g} Initial guess of covariance matrix for each component
#'@return \code{new_B_g} Initial guess of loading matrix.
#'@return \code{new_T_g} The identity matrix of latent variable: u
#'@return \code{new_D_g} Initial guess of error matrix
#'@return \code{new_m} Initial guess of variational mean
#'@return \code{new_V} Initial guess of variational varaince
#'@return \code{new_beta_g} Initial guess of covariates coefficients.
#'



initial_variational_PGMM<-function(W_count,G,Q_g,cov_str,X){
  n<-dim(W_count)[1]
  K<-dim(W_count)[2]-1

  V<-vector("list",G)
  m<-vector("list",G)
  z_ig<-matrix(0,nrow=n,ncol=G)
  pi_g<-rep(0,G)
  T_g<-list()
  B_g<-list()
  D_g<-list()
  mu_g<-list()
  sig_g<-list()
  beta_g=list()

  code<-strsplit(cov_str,"")[[1]]
  code_B<-code[1]
  code_T<-code[2]
  code_D<-code[3]

  if(code_B=="G"&length(unique(Q_g))!=1)
  {return("Q has to be the same for all components when fitting G__ models")}
  else{
    W_count[W_count==0]<-0.001
    Z<-W_count/rowSums(W_count)
    Y<-log(Z[,1:K]/c(Z[,K+1]))

    #pgmm start
    invisible(capture.output(
      res_class<-pgmm::pgmmEM(Y,rG=G,rq=1:max(Q_g),zstart=2,modelSubset=c("UUU"),relax=TRUE)))
    lab<-res_class$map

    #eigen vector initial
    # res_class=Mclust(Y,G,modelNames = "VVV")
    # lab=res_class$classification


    if(is.null(res_class)){
      res_class<-kmeans(Y,G)
      lab<-res_class$cluster
    }

    #initial for pi_g
    pi_g<-table(lab)/n


    #initial for m
    for (g in 1:G) {
      m[[g]] <- t(Y)
    }

    #transform lab into z_ig
    for(g in 1:G){
      zzz<-lab
      zzz[zzz!=g]<-0
      zzz[zzz!=0]<-1
      z_ig[,g]<-zzz
    }

    #initial for mu_g, beta_g
    for (g in 1:G) {
      beta_g[[g]]<-ginv(t(X)%*%diag(z_ig[,g])%*%X)%*%t(X)%*%diag(z_ig[,g])%*%t(m[[g]])
      mu_g[[g]]<-t(X%*%beta_g[[g]])
    }

    #initial for sig_g
    for (g in 1:G) {
      sig_g[[g]]<-cov(Y[lab==g,])
    }


    #initial for B_g
    for(g in 1:G){
      fit=eigen(sig_g[[g]])
      B_g[[g]]<-fit$vectors[,1:Q_g[g]]%*%diag(sqrt(fit$values[1:Q_g[g]]),Q_g[g])
    }
    if(code_B=="G"&length(unique(Q_g))==1){
      for (i in 1:G) {
        B_g[[i]]<-B_g[[1]]
      }
    }



    #initial for T_u_g
    for (g in 1:G) {
      T_g[[g]]<-diag(1,Q_g[g])
    }



    #initial for D_g
    for (g in 1:G) {
      fit<-eigen(sig_g[[g]])
      s<-matrix(fit$vectors[,1:Q_g[g]],ncol=Q_g[g],nrow = K)
      l<-diag(fit$values[1:Q_g[g]],Q_g[g])
      D_g[[g]]<-diag(diag(sig_g[[g]]-s%*%l%*%t(s)))
    }

    int_D_g<-D_g
    if(code_D=="G"){
      beg_D<-0
      for (i in 1:G) {
        beg_D<-int_D_g[[i]]*pi_g[i]+beg_D
      }
      for (i in 1:G) {
        D_g[[i]]<-beg_D
      }
    }
    else if(code_D=="D"){
      for (i in 1:G) {
        D_g[[i]]<-diag(mean(diag(int_D_g[[i]])),K)
      }
    }
    else if(code_D=="C"){
      beg_D<-0
      for (i in 1:G) {
        beg_D<-int_D_g[[i]]*pi_g[i]+beg_D
      }
      for (i in 1:G) {
        D_g[[i]]<-diag(mean(diag(beg_D)),K)
      }
    }





    #inital for V
    for (g in 1:G) {
      vg<-sig_g[[g]]
      for (i in 1:n) {
        #V[[g]][[i]]=diag(diag(vg))
        V[[g]][[i]]<-diag(0.1,K)
      }
    }


    return(list(new_pi_g=pi_g,
                new_T_g=T_g,
                new_B_g=B_g,
                new_D_g=D_g,
                new_m=m,
                new_V=V,
                new_mu_g=mu_g,
                new_sig_g=sig_g,
                new_beta_g=beta_g
                ))

  }


}

