% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mixopt.R, R/mixopt_blockcd.R,
%   R/mixopt_coorddesc.R, R/mixopt_multistart.R
\name{mixopt}
\alias{mixopt}
\alias{mixopt_blockcd}
\alias{mixopt_coorddesc}
\alias{mixopt_multistart}
\title{Mixed variable optimization using coordinate descent}
\usage{
mixopt(
  par,
  fn,
  gr = NULL,
  global = "multistart",
  local = "coorddesc",
  ...,
  method,
  verbose = 0,
  track
)

mixopt_blockcd(
  par,
  fn,
  gr = NULL,
  ...,
  control = list(),
  maxblocksize = NULL,
  method,
  fngr = NULL,
  maxiter = 100,
  maxeval = NULL,
  maxtime = NULL,
  verbose = 0,
  track = FALSE
)

mixopt_coorddesc(
  par,
  fn,
  gr = NULL,
  ...,
  method,
  maxiter = 100,
  maxeval = NULL,
  maxtime = NULL,
  verbose = 0,
  track = FALSE
)

mixopt_multistart(
  par,
  fn,
  gr = NULL,
  ...,
  method,
  fngr = NULL,
  n0 = 20,
  n1 = 2,
  maxiter = 100,
  maxeval = NULL,
  verbose = 0,
  groupeval = FALSE,
  track = FALSE
)
}
\arguments{
\item{par}{List of parameters}

\item{fn}{Function to evaluate}

\item{gr}{Gradient of fn}

\item{global}{Global optimization algorithm to use.
`FALSE` if you only want local optimization.}

\item{local}{Local optimization algorithm to use.}

\item{...}{Additional args}

\item{method}{Optimization method}

\item{verbose}{How much to print. 0 is none, 1 is standard,
2 is some, 3 is a lot, etc.}

\item{track}{Should it track the parameters evaluated and value?}

\item{control}{Parameters for optimizing.}

\item{maxblocksize}{The maximum number of continuous dimensions that should
be placed into a single block.}

\item{fngr}{Function that returns the function and gradient value for the
given input as a list with names "fn" and "gr".}

\item{maxiter}{Maximum number of outer iterations.
For coordinate descent, one iteration is a loop over each parameter.}

\item{maxeval}{Maximum number of function evaluations.
It may go over this number while in an inner optimization loop,
but will exit after that.}

\item{maxtime}{Maximum time to run in seconds. Not an exact limit, only
checks occasionally.}

\item{n0}{For multistart, number of random initial points to evaluate.}

\item{n1}{For multistart, number of best starts to optimize with.
You should have `n0` less than `n1`, potentially by a large factor.
gradient descent.}

\item{groupeval}{Can multiple inputs be evaluated at once? This can speed
up greatly for certain circumstances. Use "matrix" to have it give a set
of points as rows of a matrix to all be evaluated at once.}
}
\value{
List
}
\description{
Mixed variable optimization using coordinate descent
}
\examples{
# Simple 1D example
mixopt_blockcd(par=list(mopar_cts(2,8)), fn=function(x) {(4.5-x[1])^2})
# With gradient (isn't faster)
mixopt_blockcd(par=list(mopar_cts(2,8)), fn=function(x) {(4.5-x[1])^2},
               gr=function(x) {-2*(4.5-x[1])})

# 1D discrete ordered
mixopt_blockcd(par=list(mopar_ordered(100:10000)),
                 fn=function(x) {(x[1] - 500.3)^2})

# 2D: one continuous, one factor
mixopt_blockcd(par=list(mopar_cts(2,8), mopar_unordered(letters[1:6])),
                 fn=function(x) {ifelse(x[2] == 'b', -1, 0) +
                                 (4.5-x[1])^2})
# Simple 1D example
mixopt_coorddesc(par=list(mopar_cts(2,8)), fn=function(x) {(4.5-x[1])^2})

# 1D discrete ordered
mixopt_coorddesc(par=list(mopar_ordered(100:10000)),
                 fn=function(x) {(x[1] - 500.3)^2})

# 2D: one continuous, one factor
mixopt_coorddesc(par=list(mopar_cts(2,8), mopar_unordered(letters[1:6])),
                 fn=function(x) {ifelse(x[2] == 'b', -1, 0) +
                                 (4.5-x[1])^2})
# 2D
library(ggplot2)
library(dplyr)
f6 <- function(x) {-(-x[1]*.5*sin(.5*x[1])*1 - 1e-2*x[2]^2 +
                       .2*x[1] - .3*x[2])}

if (requireNamespace("ContourFunctions", quietly = TRUE)) {
  ContourFunctions::cf_func(f6, xlim=c(0,100), ylim=c(-100,100))
}
m6 <- mixopt_coorddesc(par=list(mopar_cts(0,100), mopar_cts(-100,100)),
                       fn=f6, track = TRUE)
plot_track(m6)
ms6 <- mixopt_multistart(par=list(mopar_cts(0,100), mopar_cts(-100,100)),
                         fn=f6, track = TRUE)
plot_track(ms6)
if (requireNamespace("ContourFunctions", quietly = TRUE)) {
  ContourFunctions::cf_func(f6, xlim=c(0,100), ylim=c(-100,100),
                            gg = TRUE) +
    geom_point(data=as.data.frame(matrix(unlist(ms6$track$par),
                                         ncol=2, byrow=TRUE)) \%>\%
                 bind_cols(newbest=ms6$track$newbest),
               aes(V1, V2, color=newbest), alpha=.5)
}
}
\references{
https://en.wikipedia.org/wiki/Coordinate_descent

https://en.wikipedia.org/wiki/Coordinate_descent

https://www.uv.es/rmarti/paper/docs/multi2.pdf
}
