#' Retrieve a Color Palette from the okcolors package
#'
#' Returns a vector of `n` colors from a specified palette included in the okcolors package.
#'
#' @param palette Character. Name of the palette to use. One of `"it"`, `"stone"`, `"skyscrapers"`, `"obsession"`, or `"moment"`. Default is `"obsession"`.
#' @param n Integer. Number of colors to return. Default is `5`.
#'
#' @return A character vector of hex color codes of length `n`.
#'
#' @details
#' The available palettes are:
#' \itemize{
#'   \item \code{"it"} – dark reds to light pinks: \code{"#330000"} (dark maroon), \code{"#660000"} (deep red), \code{"#cc0033"} (vivid red), \code{"#ff6699"} (pink coral), \code{"#ffcc99"} (peach).
#'   \item \code{"stone"} – earthy tones with pastels: \code{"#bc8553"} (bronze), \code{"#dea466"} (sand), \code{"#ff99cc"} (bubblegum pink), \code{"#ffccff"} (lavender blush), \code{"#c3dda2"} (sage).
#'   \item \code{"skyscrapers"} – vibrant contrasting urban tones: \code{"#ff3366"} (raspberry), \code{"#ff6633"} (orange coral), \code{"#ffcc66"} (light gold), \code{"#33cccc"} (turquoise), \code{"#663366"} (plum).
#'   \item \code{"obsession"} – bold and vivid colors for attention-grabbing visuals: \code{"#45924c"} (forest green), \code{"#e96cd2"} (orchid), \code{"#4561cf"} (royal blue), \code{"#f44f49"} (tomato red), \code{"#f0cc36"} (sunflower).
#'   \item \code{"moment"} – saturated and modern hues: \code{"#ffccff"} (lavender blush), \code{"#e96cd2"} (orchid), \code{"#ff3366"} (raspberry), \code{"#33cccc"} (turquoise), \code{"#f99841"} (orange sherbet).
#' }
#'
#' @examples
#' okcolors("stone", 3)
#' okcolors("obsession", 7)
#'
#' @export
okcolors <- function(palette = "obsession", n = 5) {
  
  palettes <- list(
    it  = grDevices::colorRampPalette(c('#330000', '#660000', '#cc0033', '#ff6699', '#ffcc99')),
    stone  = grDevices::colorRampPalette(c('#ff99cc', '#ffccff', '#dea466', '#bc8553', '#c3dda2')),
    skyscrapers  = grDevices::colorRampPalette(c('#ff3366', '#ff6633', '#ffcc66', '#33cccc', '#663366')),
    obsession  = grDevices::colorRampPalette(c('#45924c', '#f0cc36', '#e96cd2', '#f44f49', '#4561cf')),
    moment  = grDevices::colorRampPalette(c('#ff3366', '#e96cd2', '#ffccff', '#f99841', '#33cccc'))
  )
  
  if (!palette %in% names(palettes)) {
    stop("Palette not found. Choose from: ", paste(names(palettes), collapse = ", "))
  } 
  
  palettes[[palette]](n)
  
}
