% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_species.r
\name{species}
\alias{species}
\title{Generates species response parameters for n dimensions}
\usage{
species(
  nspp = 30,
  Amax,
  fun,
  xpar,
  srange,
  alpha = 4,
  gamma = 4,
  ndim,
  sdistr,
  ocor,
  odistr
)
}
\arguments{
\item{nspp}{Number of species to be generated.}

\item{Amax}{Maximum abundance of a species.
\code{Amax} currently allows three options:
i) a function how to generate maximum abundances (e.g. \code{runif}, \code{rgamma})
ii) a vector of length \code{nspp}
iii) a single number that is used as maximum abundance for all the species.}

\item{fun}{Function to generate species optima (e.g. \code{rnorm}, \code{runif}).
The two parameters in \code{xpar} are passed to function \code{fun}.
If omitted species optima are generated at regular intervals between the two
values in \code{xpar}.}

\item{xpar}{Two numbers describing a distribution
e.g mu and sigma for a normal distribution,
lower and upper bound for a random uniform distribution.}

\item{srange}{Length of the ecological gradient to which individual species
respond. Either one number or a matrix with \code{nspp} rows and \code{ndim} columns.
If \code{srange} should be different for different environmental variables a
simpler solution is to change argument \code{elen} in \code{\link{make.env}}
accordingly. E.g. \code{elen = c(100,50,50)} when using three environmental
gradients.}

\item{alpha}{Shape parameter of the beta distribution.
One number or a matrix with \code{nspp} rows and \code{ndim} columns.}

\item{gamma}{Shape parameter of the beta distribution.
One number or a matrix with \code{nspp} rows and \code{ndim} columns.}

\item{ndim}{Number of environmental variables to which generated species
should respond.}

\item{sdistr}{Users may supply own distributions of species optima.
Matrix with \code{nspp} rows and \code{ndim} columns
(as well in the special case of \code{ndim = 1}).}

\item{ocor}{Correlation matrix of the species optima.
May be generated by code {\link{cor.mat.fun}}.}

\item{odistr}{Distribution of the correlated optima either 'uniform' or
'Gaussian'}
}
\value{
List with \code{ndim} elements. Each list contains the species response
parameters to one environmental gradient.
}
\description{
Generates species response parameters to n environmental variables following
Minchin (1987).
}
\details{
Details on the exact generation of species response functions from parameters
\code{Amax}, \code{m}, \code{r}, \code{gamma} and \code{alpha} are given in Minchin (1987).
Species response curves are determined by five parameters:
a parameter determining the maximum abundance (\code{Amax})
and one describing the location (\code{m}) of this mode.
A parameter determining to which environmental range the species respond
(\code{srange} in the input \code{r} in the output) and two parameters
(\code{alpha}, \code{gamma}) describing the shape of the species response function.
If \code{alpha} = \code{gamma} the response curve is symmetric
(\code{alpha} = \code{gamma} = 4, yields approximately Gaussian distributions).
Additionally, species optima for several environmental variables may be
correlated.
Currently this is only possible for Gaussian or uniform distributions of
species optima. Users may as well supply previously generated optima
(e.g. optima similar to a real dataset).
}
\examples{
spec.par <- species(
  nspp = 30, Amax = runif, srange = 200, fun = runif,
  xpar = c(-50, 150),
  ndim = 5, alpha = 4, gamma = 4
)
spec.par <- species(
  nspp = 30, ndim = 3, Amax = runif, xpar = c(-50, 150),
  srange = 200, alpha = 4, gamma = 4
)

# example where srange, alpha and gamma are different for each
# species and environmental gradient.
spec.par <- species(
  nspp = 30, ndim = 3, Amax = runif, xpar = c(-50, 150),
  srange = matrix(ncol = 3, runif(90, 100, 200)),
  alpha = matrix(ncol = 3, runif(90, 1, 5)),
  gamma = matrix(ncol = 3, runif(90, 1, 5))
)

# example where species optima are correlated
correlations <- list(c(1, 2, 0.5), c(1, 3, 0.3), c(2, 3, 0.1))
spec.cor.mat <- cor.mat.fun(3, correlations)
spec.par <- species(
  nspp = 30, ndim = 3, Amax = runif, xpar = c(50, 50),
  srange = 200, alpha = 4, gamma = 4,
  ocor = spec.cor.mat, odistr = "Gaussian"
)

# example for species response curves (users should alter alpha and gamma)
spec.par <- species(
  nspp = 1, Amax = 200, srange = 200, fun = runif,
  xpar = c(50, 50),
  ndim = 1, alpha = 3, gamma = 1
)
env <- -50:150
response <- palaeoSig:::make.abundances(
  env = -50:150,
  param = spec.par[[1]]$spp
)
plot(env, response, type = "l")
}
\references{
Minchin, P.R. (1987) Multidimensional Community Patterns:
Towards a Comprehensive Model. \emph{Vegetatio}, \strong{71}, 145-156.
\doi{10.1007/BF00039167}
}
\seealso{
\code{\link{make.env}}, \code{\link{abundances}}
}
\author{
Mathias Trachsel and Richard J. Telford
}
\keyword{datagen}
