% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/helperFunctions.R
\encoding{UTF-8}
\name{conditions}
\alias{conditions}
\title{scapesClassification conditions}
\usage{
conditions(names_attTbl, cond, silent = FALSE)
}
\arguments{
\item{names_attTbl}{character vector, the column (i.e. variable) names of the
attribute table returned by the function \code{\link{attTbl}}.}

\item{cond}{character string, the condition string used by the \code{cond.*}
functions to classify raster cells (see \code{\link{conditions}}).}

\item{silent}{logic, when true, the function returns only error messages.}
}
\value{
An error message if the function finds spelling or syntax errors or a
string with the types of rules detected in the condition string.
}
\description{
Check for spelling and syntax errors in conditions (\code{cond} argument) and
detect the type of conditions being used.
}
\details{
\strong{Conditions (alias classification rules)}
\itemize{
\item Classification rules evaluate either to true or false and determine what
raster cells have to be classified.
\item Conditions are passed to \code{scapesClassification} functions as a
single character string. They can consist of combination of variables names
(as named in the attribute table, see \code{\link{attTbl}}), arithmetic
\code{(+|-|*|/|^|\%\%|\%/\%)}, relational \code{(>|<|>=|<=|==|!=|\%/\%)} and
logic operators \code{(&||)} and base R functions (e.g.,
\code{abs(variable_name)}).
\item All variables included in an attribute table (see \code{\link{attTbl}})
can be included in a condition string by name (e.g., var name =
\code{"dummy_var"}; condition = \code{"dummy_var > 1"}).\cr
}

\strong{Class vectors}
\itemize{
\item Class vectors map raster cells to numeric classes.
\item The n^th^ element of a class vector stores the class of the raster cell
stored in the n^th^ row of the corresponding attribute table (see
\code{\link{attTbl}}).
\item Class vectors can serve also as a function input. As inputs, they provide
information about the groups of cells that have already been classified.
\item Every time a class vector is provided as a function input, it is
\emph{updated} by assigning a numeric class to \emph{unclassified cells} that meet
function conditions.
\item Unclassified cells are represented as NA values.\cr
}

\strong{Rule evaluation: Global evaluation}
\itemize{
\item Classification rules are applied to all unclassified raster
cells.
\item Function using \emph{global evaluation}: \code{\link{cond.4.all}}.
}

\strong{Rule evaluation: Focal evaluation}
\itemize{
\item Classification rules are applied only to raster cells at specific
locations and are based on class (dis)contiguity and class continuity.
\item \strong{Class contiguity:} \cr classification rules are applied only to raster
cells contiguous to focal cells (identified in the \code{cond.*} functions
by the argument \code{nbs_of}).
\item \strong{Class continuity:} \cr join into the same class cells that respect the
same rules and that are connected to the same focal cells. This means that,
at each iteration, newly classified cells become focal cells and conditions
are tested in their neighborhood.
\item Function using \emph{focal evaluation}: \code{\link{anchor.seed}},
\code{\link{cond.4.nofn}}, \code{\link{cond.reclass}},
\code{\link{reclass.nbs}} and \code{\link{classify.all}}.\cr
}

\strong{Focal evaluation: Definitions}
\itemize{
\item \strong{Cell neighborhood:} a cell with coordinates \code{(x, y)} has 8
neighbors with coordinates: \code{(x±1, y)},  \code{(x, y±1)} and
\code{(x±1, y±1)}. Cells on the edge of a raster have less than 8
neighbors. See \code{\link{ngbList}}.
\item \strong{Focal cell:} cells whose neighbors are evaluated against the
classification rule(s). In the classification functions \emph{focal cells} are
identified by the argument \code{nbs_of}.
\item \strong{Test cell:} the cell in the neighborhood of the focal cell that is
being tested. At turns all cells in the neighborhood of a focal cell are
tested against the classification rule(s).
\item \strong{Directional neighborhood:} it consists of the intersection between the
focal and the test cell neighborhoods.\cr
}

\strong{Condition type: Absolute conditions}

\strong{1) Absolute test cell condition:} compares cell values against a
threshold value.
\itemize{
\item This type of condition applies to all functions with a \code{conditions}
argument.
\item In global evaluations all cells meeting absolute conditions receive a
classification number. In focal evaluations all \code{test cells} meeting
absolute conditions receive a classification number.
\item \emph{Examples of valid conditions:} \code{"variable_A > 1 & variable_B !=
  0"}; \code{"(variable_A^2 < 50 & variable_B == 0) | abs(variable_C) > 50"}.\cr
\emph{Functions:} \code{\link{anchor.seed}}, \code{\link{cond.4.all}},
\code{\link{cond.4.nofn}} and \code{\link{cond.reclass}}.\cr
}

\strong{2) Absolute neighborhood condition:} compares the values of the
\code{test cell} and of its \code{neighborhood} against a threshold value.
\itemize{
\item An absolute neighborhood condition is identified by a variable name
followed by curly brackets (e.g., \code{"variable_name{}"}).
\item A maximum of 9 evaluations are performed for each test cell (the test
cell itself and the cells of its neighborhood are compared against a
threshold value).
\item Test cells receive a classification number if the rule is true for at
least as many evaluations as the ones specified by the argument
\code{peval}. The argument \code{peval} ranges from 0 to 1. When 9
evaluations are performed, \code{peval = 1} means that all \code{9}
evaluations have to be true; \code{peval = 0.5} means that at least
\code{4.5} (rounded to \code{5}) evaluations have to be true.
\item Only one neighborhood rule is allowed for each condition string (e.g., it
is not possible to have a condition string like \code{"variable_A{} > 0 &
  variable_B{} > 1"}).
\item The function \code{\link{cond.4.nofn}} can consider a \code{directional
  neighborhood} instead of the test cell neighborhood by setting the argument
\code{directional = TRUE}.
\item \emph{Example of valid conditions:} \code{"variable_A{} > 1 & abs(variable_B)
  != 0"}. \cr \emph{Functions:} \code{\link{cond.4.nofn}} and
\code{\link{cond.reclass}}.\cr
}

\strong{Condition type: Relative conditions}

\strong{1) Relative focal cell condition:} compares the \code{test cell} value
against the \code{focal cell} value.
\itemize{
\item A relative focal cell condition is identified by a variable name followed
by square brackets (e.g., \code{"variable_name[]"}).
\item Rules are defined repeating twice the same variable name, once with
square brackets and once without. Square brackets indicate the focal cell
value. As an example, the rule \code{"dummy_var < dummy_var[]"} compares
the value of the the test cell (\code{"dummy_var"}) against the value of
the focal cell (\code{"dummy_var[]"}).
\item Test cells are classified if the rule is true.
\item \emph{Examples of valid conditions:} \code{"variable_A > variable_A[]"};
\code{"(variable_A > variable_A[] & variable_B{} < 10) | variable_C > 1"}.
Note that the last example is a combination of absolute and focal cell
conditions. \cr \emph{Functions:} \code{\link{anchor.seed}} and
\code{\link{cond.4.nofn}}.
}

\strong{2) Relative neighborhood condition:} compares the values of the \code{test
  cell} against the values of the \code{test cell neighborhood}.
\itemize{
\item A relative neighborhood condition is identified by a variable name
followed by curly brackets (e.g., \code{"variable_name{}"}).
\item Rules are defined repeating twice the same variable name, once with curly
brackets and once without. Curly brackets indicate the test cell
neighborhood. As an example, the rule \code{'dummy_var < dummy_var{}'}
compares the value of the the test cell (\code{dummy_var}) against the
values of cells included in the test cell neighborhood
(\code{dummy_var{}}).
\item A maximum of 8 evaluations are performed for each test cell (the test
cell is compared against each cell included in its neighborhood).
\item Test cells receive a classification number if the rule is true for at
least as many evaluations as the ones specified by the argument
\code{peval}. The argument \code{peval} ranges from 0 to 1. When 8
evaluations are performed, \code{peval = 1} means that all \code{8}
evaluations have to be true; \code{peval = 0.5} means that at least
\code{4} evaluations have to be true.
\item Only one neighborhood rule is allowed for each condition string (e.g., it
is not possible to have a condition string like \code{"variable_A{} > 0 &
  variable_B{} > variable_B"}).
\item The function \code{\link{cond.4.nofn}} can consider a \code{directional
  neighborhood} instead of the test cell neighborhood by setting the argument
\code{directional = TRUE}.
\item \emph{Example of valid conditions:} \code{"variable_A > variable_A{}"};
\code{"(variable_A > variable_A{} & variable_B != variable_B[]) |
  variable_C > 1"}. Note that the last example is a combination of absolute
and relative conditions. \cr \emph{Functions:} \code{\link{cond.4.nofn}} and
\code{\link{cond.reclass}}.
}
}
\examples{
# LOAD LIBRARIES
library(scapesClassification)
library(terra)

################################################################################
# TYPES OF CONDITIONS
################################################################################

# As an example consider an attribute with the following columns
names_attTbl <- c("bathymetry", "slope")

# And the following conditions
cond <- "bathymetry>10"
conditions(names_attTbl, cond)

cond <- "bathymetry[]>bathymetry | abs(slope{}) < 5"
conditions(names_attTbl, cond)

cond <- "bathymetry[]>bathymetry | abs(slope{}) < slope"
conditions(names_attTbl, cond)

################################################################################
# FOCAL EVALUATION DEFINITIONS
################################################################################

# CELL NUMBERS OF A DUMMY RASTER (7X7)
r   <- terra::rast(matrix(1:49, nrow = 7, byrow = TRUE), extent=c(0,7,0,7))
nbs <- ngbList(r)

# CLASS VECTOR WITH ONE TEST AND ONE FOCAL CELL
cv <- as.numeric(rep(NA, 49))
cv[c(32, 25)] <- c(1, 2) # tc (class 1), fc (class 2)
r_cv <- cv.2.rast(r, classVector = cv)

# POLYGONS REPRESENTING NEIGHBORHOODS
fcn <- rbind(c(2,5), c(5,5), c(5,2), c(2,2), c(2,5))
fcn <- terra::vect(fcn, type="polygons")

tcn <- rbind(c(2,4), c(5,4), c(5,1), c(2,1), c(2,4))
tcn <- terra::vect(tcn, type="polygons")

# PLOT - FOCAL EVALUATION DEFINITIONS
m <- c(3.5, 8, 1.2, 8)
plot(r_cv, type = "class", asp = NA, legend = FALSE, axes = FALSE, mar = m,
     col = c("goldenrod3","#1088a0"), colNA= "grey90")
text(r)
mtext(side=3, line=0, adj=0, cex=1, font=2, "FOCAL EVALUATION")
mtext(side=1, line=0, adj=0, cex=0.9,
      "Focal cell neighborhood: 17, 18, 19, 24, 26, 31, 32, 33")
mtext(side=1, line=1, cex=0.9, adj=0,
      "Test cell neighborhood: 24, 25, 26, 31, 33, 38, 39, 40")
mtext(side=1, line=2, cex=0.9, adj=0,
     "Directional neighborhood: 24, 25, 26, 31, 32, 33")
lines(fcn, col="#1088a0", lwd=2)
lines(tcn, col="#cfad8999", lwd=2)
legend("bottomleft", ncol = 1, bg = "white", y.intersp= 1.3,
       legend = c("Focal cell", "Test cell"), fill = c("#1088a0", "goldenrod3"))
}
\seealso{
\code{\link[=cond.4.all]{cond.4.all()}}, \code{\link[=cond.4.nofn]{cond.4.nofn()}}, \code{\link[=anchor.seed]{anchor.seed()}}, \code{\link[=cond.reclass]{cond.reclass()}},
\code{\link[=conditions]{conditions()}}

\code{\link[=anchor.seed]{anchor.seed()}}, \code{\link[=attTbl]{attTbl()}}, \code{\link[=cond.4.all]{cond.4.all()}}, \code{\link[=cond.4.nofn]{cond.4.nofn()}},
\code{\link[=cond.reclass]{cond.reclass()}}, \code{\link[=classify.all]{classify.all()}}
}
