% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tvSURE.R
\name{tvSURE}
\alias{tvSURE}
\alias{tvsure-class}
\alias{tvsure}
\title{Time-Varying Seemingly Unrelated Regression Equations Model}
\usage{
tvSURE(
  formula,
  z = NULL,
  ez = NULL,
  bw = NULL,
  cv.block = 0,
  data,
  method = c("tvOLS", "tvFGLS", "tvGLS"),
  Sigma = NULL,
  est = c("lc", "ll"),
  tkernel = c("Triweight", "Epa", "Gaussian"),
  bw.cov = NULL,
  singular.ok = TRUE,
  R = NULL,
  r = NULL,
  control = tvreg.control(...),
  ...
)
}
\arguments{
\item{formula}{A list of formulas, one for each equation.}

\item{z}{A vector containing the smoothing variable.}

\item{ez}{(optional) A scalar or vector with the smoothing values. If 
values are not included then the vector \code{z} is used instead.}

\item{bw}{An optional scalar or vector of length the number of equations. It represents the bandwidth in
the estimation of trend coefficients. If NULL, it is selected by cross validation.}

\item{cv.block}{A positive scalar with the size of the block in leave one block out cross-validation.
By default 'cv.block = 0' meaning leave one out cross-validation.}

\item{data}{A matrix or data frame containing variables in the formula.}

\item{method}{A character, a matrix of dimensions neq x neq or an array of dimensions obs x neq x neq, where
\code{obs} is the number of observations and \code{neq} is the number of equations.
If method = \code{identity} or \code{tvOLS} (default) then the method used is a time-varying OLS.
If method is a matrix (constant over time) or an array, then the \code{tvGLS} is called.
If method = \code{tvFGLS}, then the covariance matrix is estimated nonparametrically and the
estimation of the system is done as a whole.}

\item{Sigma}{A matrix of dimensions neq x neq or an array of dimensions neq x neq x obs
(neq = number of equations, obs = number of observations). It represents
the covariance matrix of the error term. Only necessary for method \code{tvGLS}.}

\item{est}{The nonparametric estimation method, one of "lc" (default) for linear constant or "ll" for local linear.}

\item{tkernel}{A character, either "Triweight" (default), "Epa" or "Gaussian" kernel function.}

\item{bw.cov}{An optional scalar. It represents the bandwidth in the nonparametric estimation of the
varying covariance matrix. If NULL, it is selected by cross validation.}

\item{singular.ok}{Logical. If FALSE, a singular model is an error.}

\item{R}{An optional nrest x nvar x neq (nrest =  number of restrictions, nvar = number of variables in each equation,
neq = number of equations).}

\item{r}{An optional vector of length the number of restrictions. By default it contains zeros.}

\item{control}{list of control parameters.  The default is constructed by
the function \code{\link{tvreg.control}}.  See the documentation of
\code{\link{tvreg.control}} for details.}

\item{...}{Other parameters passed to specific methods.}
}
\value{
\code{tvSURE} returns a list of the class \code{tvsure} containing the results of the whole system, results of the estimation
and confidence instervals if chosen.
The object of class \code{tvsure} have the following components:
\item{coefficients}{An array of dimension obs x nvar x neq (obs = number of observations, nvar = number of variables
in each equation, neq = number of equations in the system) with the time-varying coefficients estimates.}
\item{Lower}{If \code{level} non equal zero, an array of dimension obs x nvar x neq containing the confidence 
interval lower band.}
\item{Upper}{If \code{level} non equal zero, an array of dimension obs x nvar x neq containing the confidence 
interval upper band.}
\item{Sigma}{An array of dimension obs x neq x neq with the estimates of the errors covariance matrix.}
\item{fitted}{The fitted values.}
\item{residuals}{Estimation residuals.}
\item{x}{A list with the regressors data.}
\item{y}{A matrix with the dependent variable data.}
\item{z}{A vector with the smoothing variable.}
\item{ez}{A vector with the smoothing estimation values.}
\item{bw}{Bandwidth of mean estimation.}
\item{obs}{Integer specifying the number of observations in each equation (balanced sample).}
\item{neq}{Integer specifying the number of equations.}
\item{nvar}{Vector of integers specifying the number of variables in each equation.}
\item{method}{Estimation method.}
\item{est}{Nonparemtric estimation methodology.}
\item{tkernel}{Kernel type.}
\item{bw.cov}{Bandwidht of Sigma estimation.}
\item{level}{Confidence interval range.}
\item{runs}{Number of bootstrap replications.}
\item{tboot}{Type of bootstrap.}
\item{BOOT}{List with all bootstrap replications of \code{coefficients}, if done.}
\item{R}{Restrictions matrix.}
\item{r}{Restrictions vector.}
\item{formula}{Initial formula.}
}
\description{
Fits a set of balanced linear structural equations using Time-varying Ordinary Least 
Squares (tvOLS), Time-varying Seemingly Unrelated Regression (tvGLS), when the error 
variance-covariance matrix is known, or Time-varying Feasible Seemingly Unrelated 
Regression (tvFGLS), when the error variance-covariance matrix is unknown.
}
\details{
This function wraps up the kernel smoothing "tvOLS" and "tvGLS" estimators. The former is used when
equations are considered independent while the later assumes that the error term is correlated
amongst equations. This relation is given in matrix "Sigma" which is used in the estimation. When
"Sigma" is known, the estimates are calculated via the "tvGLS", and via the "tvFGLS" when "Sigma"
is unknown and must be estimated.

Bandwidth selection is of great importance in kernel smoothing methodologies and it is done
automatically by cross-validation. One important aspect in the current packages is that the
bandwidth is selected independently for each equation and then the average is taken to use the
same bandwidth for each equation. It has been shown in Casas et al. (2017) that using
different bandwidths for each equation is in general a bad practice, even for uncorrelated equations.
Even though, the user may be able to use different bandwidths calling functions \code{\link{bw}} and
\code{\link{tvGLS}} separatedly.

A system consists of "neq" number of equations with "obs" number of observations each and a number of
variables not necessarily equal for all equations. The matrix notation is:
\deqn{Y_{t} = X_t \beta_{t}+u_{t}}
where \eqn{Y_t  = (y_{1t}, y_{2t}, \ldots, y_{neq t})'}, \eqn{X_t = diag (x_{1t}, x_{2t}, \ldots, x_{neq t})}
and \eqn{\beta_{t} = \left(\beta _{1t}', \ldots, \beta _{neq t}'\right)'} is a vector of order the
total number of variables in the system. The error vector \eqn{u_{t} = (u_{1t}, u_{2t}, \ldots, u_{neq t})'}
has zero mean and  covariance matrix \eqn{E(u_t u_t') = \Sigma_t}.
}
\examples{
\dontrun{
data("Kmenta", package = "systemfit")
eqDemand <- consump ~ price + income
eqSupply <- consump ~ price + farmPrice + trend
system <- list(demand = eqDemand, supply = eqSupply)
eqSupply2 <- consump ~  price + farmPrice 
system2 <- list(demand = eqDemand, supply = eqSupply2)

##OLS estimation of a system
OLS <- systemfit::systemfit(system, method = "OLS", data = Kmenta)
##tvOLS estimation of a system with the local linear estimator
##removing trend because it is included in the intercept changing over time
TVOLS <- tvSURE(system2, data = Kmenta,  est = "ll")

##SUR/FGLS estimation
FGLS <- systemfit::systemfit(system, data = Kmenta, method = "SUR")
##tvSURE estimation
TVFGLS <- tvSURE(system, data = Kmenta, method = "tvFGLS")
}

}
\references{
Casas, I., Ferreira, E., and Orbe, S. (2017) Time-Varying Coefficient Estimation 
in SURE Models: Application to Portfolio Management. Available at SSRN: 
https://ssrn.com/abstract=3043137

Chen, X. B., Gao, J., Li, D., and Silvapulle, P (2017) Nonparametric Estimation and 
Forecasting for Time-Varying Coefficient Realized Volatility Models.
\emph{Journal of Business and Economic Statistics}, pp.1-13

Granger, C. W (2008) Non-Linear Models: Where Do We Go Next - Time Varying
Parameter Models? \emph{Studies in Nonlinear Dynamics and Econometrics}, 12, pp. 1-11.

Kristensen, D (2012) Non-parametric detection and estimation of structural change.
\emph{Econometrics Journal}, 15, pp. 420-461.

Orbe, S., Ferreira, E., and Rodriguez-Poo, J (2004) On the estimation and testing of
time varying constraints in econometric models, \emph{Statistica Sinica}.
}
\seealso{
\code{\link{bw}}, \code{\link{tvCov}},  \code{\link{tvVAR}}, 
\code{\link{confint}}, \code{\link{plot}}, \code{\link{print}} and \code{\link{summary}}
}
