/* l2xisymt.h  Symbol table common for LTX2X interpreter */
/*          the principal typedefs are in file licomsym.h  */

#ifndef symtab_h
#define symtab_h

#include "l2xicmon.h" 

#ifndef licomsym_h
#include "licomsym.h"   /* contains the typedefs */
#endif

typedef char *STRING;    /* a pointer to a char */
typedef STRING *PTRADR;  /* pointer to a pointer to a char */

  /* use names rather than integers for TRUE (1) /FALSE (0) values */
#define FALSE_REP -999
#define UNKNOWN_REP -99
#define TRUE_REP -9

/* the simple pre-defined types */
extern TYPE_STRUCT dummy_type;
extern TYPE_STRUCT_PTR integer_typep, real_typep;
extern TYPE_STRUCT_PTR string_typep, binary_typep;
extern TYPE_STRUCT_PTR boolean_typep, logical_typep;
extern TYPE_STRUCT_PTR generic_typep, any_typep;


/* FUNCTIONS */

SYMTAB_NODE_PTR search_symtab();
SYMTAB_NODE_PTR search_symtab_display();
SYMTAB_NODE_PTR enter_symtab();
SYMTAB_NODE_PTR exit_scope();
TYPE_STRUCT_PTR make_string_typep();


/* MACROS */

#define is_array(tp) (tp->form == ARRAY_FORM)

#define is_dynagg(tp) (tp->form == BAG_FORM  ||  \
                       tp->form == LIST_FORM ||  \
                       tp->form == SET_FORM)



/* table searching macros */

/**************************************************************************/
/* search_local_symtab() search the local symtab for the current id name. */
/*                       set a pointer to the entry, else NULL            */
#define search_local_symtab(idp)                 \
  idp = search_symtab(word_string, symtab_display[level])
/**************************************************************************/


/**************************************************************************/
/* search_this_symtab() search the given symtab for the current id name.  */
/*                       set a pointer to the entry, else NULL            */
#define search_this_symtab(idp, this_symtab)       \
  idp = search_symtab(word_string, this_symtab)
/**************************************************************************/


/**************************************************************************/
/* search_all_symtab() search the symtab display for the current id name. */
/*                       set a pointer to the entry, else NULL            */
#define search_all_symtab(idp)                      \
  idp = search_symtab_display(word_string) 
/**************************************************************************/


/**************************************************************************/
/* search_and_find_all_symtab() search the symtab display for the current */
/*                       id name. If not found, ID UNDEFINED error, and   */
/*                       enter into the local symtab.                     */
/*                       set a pointer to the entry.                      */
#define search_and_find_all_symtab(idp)                      \
  if ((idp = search_symtab_display(word_string)) == NULL) {  \
    error(UNDEFINED_IDENTIFIER);                             \
    idp = enter_symtab(word_string, &symtab_display[level]); \
    idp->defn.key = UNDEFINED;                               \
    idp->typep = &dummy_type;                                \
    }
/**************************************************************************/


/**************************************************************************/
/* enter_local_symtab() enter the current id name into the local symtab   */
/*                      and set pointer to the entry                      */
#define enter_local_symtab(idp)                     \
  idp = enter_symtab(word_string, &symtab_display[level])
/**************************************************************************/


/**************************************************************************/
/* enter_name_local_symtab() enter the given name into the local symtab   */
/*                      and set pointer to the entry                      */
#define enter_name_local_symtab(idp, name)                     \
  idp = enter_symtab(name, &symtab_display[level])
/**************************************************************************/


/**************************************************************************/
/* search_and_enter_local_symtab() search the symtab display for the      */
/*                       current id name. If not found enter it, else     */
/*                       ID REDEFINED error.                              */
/*                       set a pointer to the entry.                      */
#define search_and_enter_local_symtab(idp)                      \
  if ((idp = search_symtab(word_string, symtab_display[level])) == NULL) {  \
    idp = enter_symtab(word_string, &symtab_display[level]);    \
  }                                                             \
  else error(REDEFINED_IDENTIFIER)
/**************************************************************************/



/**************************************************************************/
/* search_and_enter_this_symtab() search the given symtab for the        */
/*                       current id name. If not found enter it, else     */
/*                       ID REDEFINED error.                              */
/*                       set a pointer to the entry.                      */
#define search_and_enter_this_symtab(idp, this_symtab)            \
  if ((idp = search_symtab(word_string, this_symtab)) == NULL) {  \
    idp = enter_symtab(word_string, &this_symtab);                \
  }                                                               \
  else error(REDEFINED_IDENTIFIER)
/**************************************************************************/

#endif








