/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.collect.MapMaker;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.lang.ref.WeakReference;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Function;
import org.apache.cassandra.concurrent.ExecutorFactory;
import org.apache.cassandra.concurrent.ExecutorPlus;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.CompactionStrategyManager;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.ExecutorUtils;
import org.apache.cassandra.utils.NoSpamLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionLogger {
    private static final JsonNodeFactory json = JsonNodeFactory.instance;
    private static final Logger logger = LoggerFactory.getLogger(CompactionLogger.class);
    private static final CompactionLogSerializer serializer = new CompactionLogSerializer();
    private final WeakReference<ColumnFamilyStore> cfsRef;
    private final WeakReference<CompactionStrategyManager> csmRef;
    private final AtomicInteger identifier = new AtomicInteger(0);
    private final Map<AbstractCompactionStrategy, String> compactionStrategyMapping = new MapMaker().weakKeys().makeMap();
    private final AtomicBoolean enabled = new AtomicBoolean(false);

    public CompactionLogger(ColumnFamilyStore cfs, CompactionStrategyManager csm) {
        this.csmRef = new WeakReference<CompactionStrategyManager>(csm);
        this.cfsRef = new WeakReference<ColumnFamilyStore>(cfs);
    }

    private void forEach(Consumer<AbstractCompactionStrategy> consumer) {
        CompactionStrategyManager csm = (CompactionStrategyManager)this.csmRef.get();
        if (csm == null) {
            return;
        }
        csm.getStrategies().forEach((? super T l) -> l.forEach(consumer));
    }

    private ArrayNode compactionStrategyMap(Function<AbstractCompactionStrategy, JsonNode> select) {
        ArrayNode node = json.arrayNode();
        this.forEach(acs -> node.add((JsonNode)select.apply((AbstractCompactionStrategy)acs)));
        return node;
    }

    private ArrayNode sstableMap(Collection<SSTableReader> sstables, CompactionStrategyAndTableFunction csatf) {
        CompactionStrategyManager csm = (CompactionStrategyManager)this.csmRef.get();
        ArrayNode node = json.arrayNode();
        if (csm == null) {
            return node;
        }
        sstables.forEach((? super T t) -> node.add(csatf.apply(csm.getCompactionStrategyFor((SSTableReader)t), (SSTableReader)t)));
        return node;
    }

    private String getId(AbstractCompactionStrategy strategy) {
        return this.compactionStrategyMapping.computeIfAbsent(strategy, s -> String.valueOf(this.identifier.getAndIncrement()));
    }

    private JsonNode formatSSTables(AbstractCompactionStrategy strategy) {
        ArrayNode node = json.arrayNode();
        CompactionStrategyManager csm = (CompactionStrategyManager)this.csmRef.get();
        ColumnFamilyStore cfs = (ColumnFamilyStore)this.cfsRef.get();
        if (csm == null || cfs == null) {
            return node;
        }
        for (SSTableReader sstable : cfs.getLiveSSTables()) {
            if (csm.getCompactionStrategyFor(sstable) != strategy) continue;
            node.add(this.formatSSTable(strategy, sstable));
        }
        return node;
    }

    private JsonNode formatSSTable(AbstractCompactionStrategy strategy, SSTableReader sstable) {
        ObjectNode node = json.objectNode();
        node.put("generation", sstable.descriptor.id.toString());
        node.put("version", sstable.descriptor.version.version);
        node.put("size", sstable.onDiskLength());
        JsonNode logResult = strategy.strategyLogger().sstable(sstable);
        if (logResult != null) {
            node.set("details", logResult);
        }
        return node;
    }

    private JsonNode startStrategy(AbstractCompactionStrategy strategy) {
        ObjectNode node = json.objectNode();
        CompactionStrategyManager csm = (CompactionStrategyManager)this.csmRef.get();
        if (csm == null) {
            return node;
        }
        node.put("strategyId", this.getId(strategy));
        node.put("type", strategy.getName());
        node.set("tables", this.formatSSTables(strategy));
        node.put("repaired", csm.isRepaired(strategy));
        List<String> folders = csm.getStrategyFolders(strategy);
        ArrayNode folderNode = json.arrayNode();
        for (String folder : folders) {
            folderNode.add(folder);
        }
        node.set("folders", (JsonNode)folderNode);
        JsonNode logResult = strategy.strategyLogger().options();
        if (logResult != null) {
            node.set("options", logResult);
        }
        return node;
    }

    private JsonNode shutdownStrategy(AbstractCompactionStrategy strategy) {
        ObjectNode node = json.objectNode();
        node.put("strategyId", this.getId(strategy));
        return node;
    }

    private JsonNode describeSSTable(AbstractCompactionStrategy strategy, SSTableReader sstable) {
        ObjectNode node = json.objectNode();
        node.put("strategyId", this.getId(strategy));
        node.set("table", this.formatSSTable(strategy, sstable));
        return node;
    }

    private void describeStrategy(ObjectNode node) {
        ColumnFamilyStore cfs = (ColumnFamilyStore)this.cfsRef.get();
        if (cfs == null) {
            return;
        }
        node.put("keyspace", cfs.getKeyspaceName());
        node.put("table", cfs.getTableName());
        node.put("time", Clock.Global.currentTimeMillis());
    }

    private JsonNode startStrategies() {
        ObjectNode node = json.objectNode();
        node.put("type", "enable");
        this.describeStrategy(node);
        node.set("strategies", (JsonNode)this.compactionStrategyMap(this::startStrategy));
        return node;
    }

    public void enable() {
        if (this.enabled.compareAndSet(false, true)) {
            serializer.writeStart(this.startStrategies(), this);
        }
    }

    public void disable() {
        if (this.enabled.compareAndSet(true, false)) {
            ObjectNode node = json.objectNode();
            node.put("type", "disable");
            this.describeStrategy(node);
            node.set("strategies", (JsonNode)this.compactionStrategyMap(this::shutdownStrategy));
            serializer.write((JsonNode)node, this::startStrategies, this);
        }
    }

    public void flush(Collection<SSTableReader> sstables) {
        if (this.enabled.get()) {
            ObjectNode node = json.objectNode();
            node.put("type", "flush");
            this.describeStrategy(node);
            node.set("tables", (JsonNode)this.sstableMap(sstables, this::describeSSTable));
            serializer.write((JsonNode)node, this::startStrategies, this);
        }
    }

    public void compaction(long startTime, Collection<SSTableReader> input, long endTime, Collection<SSTableReader> output) {
        if (this.enabled.get()) {
            ObjectNode node = json.objectNode();
            node.put("type", "compaction");
            this.describeStrategy(node);
            node.put("start", String.valueOf(startTime));
            node.put("end", String.valueOf(endTime));
            node.set("input", (JsonNode)this.sstableMap(input, this::describeSSTable));
            node.set("output", (JsonNode)this.sstableMap(output, this::describeSSTable));
            serializer.write((JsonNode)node, this::startStrategies, this);
        }
    }

    public void pending(AbstractCompactionStrategy strategy, int remaining) {
        if (remaining != 0 && this.enabled.get()) {
            ObjectNode node = json.objectNode();
            node.put("type", "pending");
            this.describeStrategy(node);
            node.put("strategyId", this.getId(strategy));
            node.put("pending", remaining);
            serializer.write((JsonNode)node, this::startStrategies, this);
        }
    }

    public static void shutdownNowAndWait(long timeout, TimeUnit unit) throws InterruptedException, TimeoutException {
        ExecutorUtils.shutdownNowAndWait(timeout, unit, CompactionLogger.serializer.loggerService);
    }

    private static class CompactionLogSerializer
    implements Writer {
        private static final String logDirectory = CassandraRelevantProperties.LOG_DIR.getString();
        private final ExecutorPlus loggerService = ExecutorFactory.Global.executorFactory().sequential("CompactionLogger");
        private final Set<Object> rolled = new HashSet<Object>();
        private OutputStreamWriter stream;

        private CompactionLogSerializer() {
        }

        private static OutputStreamWriter createStream() throws IOException {
            int count = 0;
            Path compactionLog = new File(logDirectory, "compaction.log").toPath();
            if (Files.exists(compactionLog, new LinkOption[0])) {
                Path tryPath = compactionLog;
                while (Files.exists(tryPath, new LinkOption[0])) {
                    tryPath = new File(logDirectory, String.format("compaction-%d.log", count++)).toPath();
                }
                Files.move(compactionLog, tryPath, new CopyOption[0]);
            }
            return new OutputStreamWriter(Files.newOutputStream(compactionLog, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE));
        }

        private void writeLocal(String toWrite) {
            try {
                if (this.stream == null) {
                    this.stream = CompactionLogSerializer.createStream();
                }
                this.stream.write(toWrite);
                this.stream.flush();
            }
            catch (IOException ioe) {
                NoSpamLogger.log(logger, NoSpamLogger.Level.ERROR, 1L, TimeUnit.MINUTES, "Could not write to the log file: {}", ioe);
            }
        }

        @Override
        public void writeStart(JsonNode statement, Object tag) {
            String toWrite = statement.toString() + System.lineSeparator();
            this.loggerService.execute(() -> {
                this.rolled.add(tag);
                this.writeLocal(toWrite);
            });
        }

        @Override
        public void write(JsonNode statement, StrategySummary summary, Object tag) {
            String toWrite = statement.toString() + System.lineSeparator();
            this.loggerService.execute(() -> {
                if (!this.rolled.contains(tag)) {
                    this.writeLocal(summary.getSummary().toString() + System.lineSeparator());
                    this.rolled.add(tag);
                }
                this.writeLocal(toWrite);
            });
        }
    }

    private static interface CompactionStrategyAndTableFunction {
        public JsonNode apply(AbstractCompactionStrategy var1, SSTableReader var2);
    }

    public static interface Writer {
        public void writeStart(JsonNode var1, Object var2);

        public void write(JsonNode var1, StrategySummary var2, Object var3);
    }

    public static interface StrategySummary {
        public JsonNode getSummary();
    }

    public static interface Strategy {
        public static final Strategy none = new Strategy(){

            @Override
            public JsonNode sstable(SSTableReader sstable) {
                return null;
            }

            @Override
            public JsonNode options() {
                return null;
            }
        };

        public JsonNode sstable(SSTableReader var1);

        public JsonNode options();
    }
}

