/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.index.sai.utils;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableSet;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.apache.cassandra.cql3.CQL3Type;
import org.apache.cassandra.cql3.Operator;
import org.apache.cassandra.cql3.statements.schema.IndexTarget;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.filter.RowFilter;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.AsciiType;
import org.apache.cassandra.db.marshal.BooleanType;
import org.apache.cassandra.db.marshal.ByteBufferAccessor;
import org.apache.cassandra.db.marshal.CollectionType;
import org.apache.cassandra.db.marshal.CompositeType;
import org.apache.cassandra.db.marshal.DecimalType;
import org.apache.cassandra.db.marshal.InetAddressType;
import org.apache.cassandra.db.marshal.IntegerType;
import org.apache.cassandra.db.marshal.LongType;
import org.apache.cassandra.db.marshal.StringType;
import org.apache.cassandra.db.marshal.UTF8Type;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.db.marshal.VectorType;
import org.apache.cassandra.db.rows.Cell;
import org.apache.cassandra.db.rows.ComplexColumnData;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.index.sai.plan.Expression;
import org.apache.cassandra.schema.ColumnMetadata;
import org.apache.cassandra.serializers.MarshalException;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FastByteOperations;
import org.apache.cassandra.utils.bytecomparable.ByteComparable;
import org.apache.cassandra.utils.bytecomparable.ByteSource;
import org.apache.cassandra.utils.bytecomparable.ByteSourceInverse;

public class IndexTermType {
    private static final Set<AbstractType<?>> EQ_ONLY_TYPES = ImmutableSet.of((Object)UTF8Type.instance, (Object)AsciiType.instance, (Object)BooleanType.instance, (Object)UUIDType.instance);
    private static final byte[] IPV4_PREFIX = new byte[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, -1};
    private static final int DECIMAL_APPROXIMATION_BYTES = 24;
    private static final int BIG_INTEGER_APPROXIMATION_BYTES = 20;
    private static final int INET_ADDRESS_SIZE = 16;
    private static final int DEFAULT_FIXED_LENGTH = 16;
    private final ColumnMetadata columnMetadata;
    private final IndexTarget.Type indexTargetType;
    private final AbstractType<?> indexType;
    private final List<IndexTermType> subTypes;
    private final AbstractType<?> vectorElementType;
    private final int vectorDimension;
    private final EnumSet<Capability> capabilities;

    public static IndexTermType create(ColumnMetadata columnMetadata, List<ColumnMetadata> partitionColumns, IndexTarget.Type indexTargetType) {
        return new IndexTermType(columnMetadata, partitionColumns, indexTargetType);
    }

    private IndexTermType(ColumnMetadata columnMetadata, List<ColumnMetadata> partitionColumns, IndexTarget.Type indexTargetType) {
        this.columnMetadata = columnMetadata;
        this.indexTargetType = indexTargetType;
        this.capabilities = this.calculateCapabilities(columnMetadata, partitionColumns, indexTargetType);
        this.indexType = this.calculateIndexType(columnMetadata.type, this.capabilities, indexTargetType);
        AbstractType<?> baseType = this.indexType.unwrap();
        if (baseType.subTypes().isEmpty() || indexTargetType == IndexTarget.Type.SIMPLE || indexTargetType == IndexTarget.Type.FULL) {
            this.subTypes = Collections.emptyList();
        } else {
            ArrayList<IndexTermType> subTypes = new ArrayList<IndexTermType>(baseType.subTypes().size());
            for (AbstractType<?> subType : baseType.subTypes()) {
                subTypes.add(new IndexTermType(columnMetadata.withNewType(subType), partitionColumns, indexTargetType));
            }
            this.subTypes = Collections.unmodifiableList(subTypes);
        }
        if (this.isVector()) {
            VectorType vectorType = (VectorType)baseType;
            this.vectorElementType = vectorType.elementType;
            this.vectorDimension = vectorType.dimension;
        } else {
            this.vectorElementType = null;
            this.vectorDimension = -1;
        }
    }

    public boolean isLiteral() {
        return this.capabilities.contains((Object)Capability.LITERAL);
    }

    public boolean isString() {
        return this.capabilities.contains((Object)Capability.STRING);
    }

    public boolean isVector() {
        return this.capabilities.contains((Object)Capability.VECTOR);
    }

    public boolean isReversed() {
        return this.capabilities.contains((Object)Capability.REVERSED);
    }

    public boolean isFrozen() {
        return this.capabilities.contains((Object)Capability.FROZEN);
    }

    public boolean isNonFrozenCollection() {
        return this.capabilities.contains((Object)Capability.NON_FROZEN_COLLECTION);
    }

    public boolean isFrozenCollection() {
        return this.capabilities.contains((Object)Capability.COLLECTION) && this.capabilities.contains((Object)Capability.FROZEN);
    }

    public boolean isComposite() {
        return this.capabilities.contains((Object)Capability.COMPOSITE);
    }

    public boolean isMultiExpression(RowFilter.Expression expression) {
        boolean multiExpression = false;
        switch (expression.operator()) {
            case EQ: {
                multiExpression = this.isNonFrozenCollection();
                break;
            }
            case CONTAINS: 
            case CONTAINS_KEY: {
                multiExpression = true;
            }
        }
        return multiExpression;
    }

    public boolean isValid(ByteBuffer term) {
        try {
            this.indexType.validate(term);
            return true;
        }
        catch (MarshalException e) {
            return false;
        }
    }

    public boolean skipsEmptyValue() {
        return !this.indexType.allowsEmpty() || !this.isLiteral();
    }

    public AbstractType<?> indexType() {
        return this.indexType;
    }

    public Collection<IndexTermType> subTypes() {
        return this.subTypes;
    }

    public CQL3Type asCQL3Type() {
        return this.indexType.asCQL3Type();
    }

    public ColumnMetadata columnMetadata() {
        return this.columnMetadata;
    }

    public String columnName() {
        return this.columnMetadata.name.toString();
    }

    public AbstractType<?> vectorElementType() {
        assert (this.isVector());
        return this.vectorElementType;
    }

    public int vectorDimension() {
        assert (this.isVector());
        return this.vectorDimension;
    }

    public boolean dependsOn(ColumnMetadata columnMetadata) {
        return this.columnMetadata.compareTo(columnMetadata) == 0;
    }

    public static boolean isEqOnlyType(AbstractType<?> type) {
        return EQ_ONLY_TYPES.contains(type);
    }

    public boolean supportsRounding() {
        return this.isBigInteger() || this.isBigDecimal();
    }

    public int fixedSizeOf() {
        if (this.indexType.isValueLengthFixed()) {
            return this.indexType.valueLengthIfFixed();
        }
        if (this.isInetAddress()) {
            return 16;
        }
        if (this.isBigInteger()) {
            return 20;
        }
        if (this.isBigDecimal()) {
            return 24;
        }
        return 16;
    }

    public String asString(ByteBuffer value) {
        if (this.isComposite()) {
            return ByteBufferUtil.bytesToHex(value);
        }
        return this.indexType.getString(value);
    }

    public ByteBuffer fromString(String value) {
        if (this.isComposite()) {
            return ByteBufferUtil.hexToBytes(value);
        }
        return this.indexType.fromString(value);
    }

    public ByteBuffer valueOf(DecoratedKey key, Row row, long nowInSecs) {
        if (row == null) {
            return null;
        }
        switch (this.columnMetadata.kind) {
            case PARTITION_KEY: {
                return this.isCompositePartition() ? CompositeType.extractComponent(key.getKey(), this.columnMetadata.position()) : key.getKey();
            }
            case CLUSTERING: {
                return row.isStatic() ? null : row.clustering().bufferAt(this.columnMetadata.position());
            }
            case STATIC: {
                if (!row.isStatic()) {
                    return null;
                }
            }
            case REGULAR: {
                Cell<?> cell = row.getCell(this.columnMetadata);
                return cell == null || !cell.isLive(nowInSecs) ? null : cell.buffer();
            }
        }
        return null;
    }

    public Iterator<ByteBuffer> valuesOf(Row row, long nowInSecs) {
        if (row == null) {
            return null;
        }
        switch (this.columnMetadata.kind) {
            case STATIC: {
                if (!row.isStatic()) {
                    return null;
                }
            }
            case REGULAR: {
                return this.collectionIterator(row.getComplexColumnData(this.columnMetadata), nowInSecs);
            }
        }
        return null;
    }

    public Comparator<ByteBuffer> comparator() {
        if (this.isBigInteger() || this.isBigDecimal() || this.isComposite() || this.isFrozen()) {
            return FastByteOperations::compareUnsigned;
        }
        return this.indexType;
    }

    public int compare(ByteBuffer b1, ByteBuffer b2) {
        if (this.isInetAddress()) {
            return IndexTermType.compareInet(b1, b2);
        }
        if (this.isLong()) {
            return this.indexType.unwrap().compare(b1, b2);
        }
        if (this.isBigInteger() || this.isBigDecimal() || this.isComposite() || this.isFrozen()) {
            return FastByteOperations.compareUnsigned(b1, b2);
        }
        return this.indexType.compare(b1, b2);
    }

    public ByteBuffer min(ByteBuffer a, ByteBuffer b) {
        return a == null ? b : (b == null || this.compare(b, a) > 0 ? a : b);
    }

    public ByteBuffer max(ByteBuffer a, ByteBuffer b) {
        return a == null ? b : (b == null || this.compare(b, a) < 0 ? a : b);
    }

    public int comparePostFilter(Expression.Value requestedValue, Expression.Value columnValue) {
        if (this.isInetAddress()) {
            return IndexTermType.compareInet(requestedValue.encoded, columnValue.encoded);
        }
        if (this.isLong() || this.isBigDecimal() || this.isBigInteger()) {
            return this.indexType.unwrap().compare(requestedValue.raw, columnValue.raw);
        }
        if (this.isComposite() || this.isFrozen()) {
            return FastByteOperations.compareUnsigned(requestedValue.raw, columnValue.raw);
        }
        return this.indexType.compare(requestedValue.raw, columnValue.raw);
    }

    public void toComparableBytes(ByteBuffer value, byte[] bytes) {
        if (this.isInetAddress()) {
            ByteBufferUtil.copyBytes(value, value.hasArray() ? value.arrayOffset() + value.position() : value.position(), bytes, 0, 16);
        } else if (this.isBigInteger()) {
            ByteBufferUtil.copyBytes(value, value.hasArray() ? value.arrayOffset() + value.position() : value.position(), bytes, 0, 20);
        } else if (this.isBigDecimal()) {
            ByteBufferUtil.copyBytes(value, value.hasArray() ? value.arrayOffset() + value.position() : value.position(), bytes, 0, 24);
        } else {
            ByteSourceInverse.copyBytes(this.asComparableBytes(value, ByteComparable.Version.OSS50), bytes);
        }
    }

    public ByteSource asComparableBytes(ByteBuffer value, ByteComparable.Version version) {
        if (value.remaining() == 0) {
            return ByteSource.EMPTY;
        }
        if (this.isInetAddress() || this.isBigInteger() || this.isBigDecimal()) {
            return ByteSource.optionalFixedLength(ByteBufferAccessor.instance, value);
        }
        if (this.isLong()) {
            return ByteSource.optionalSignedFixedLengthNumber(ByteBufferAccessor.instance, value);
        }
        if (this.isFrozen()) {
            return ByteSource.of(value, version);
        }
        return this.indexType.asComparableBytes(value, version);
    }

    public ByteBuffer asIndexBytes(ByteBuffer value) {
        if (value == null || value.remaining() == 0) {
            return value;
        }
        if (this.isInetAddress()) {
            return IndexTermType.encodeInetAddress(value);
        }
        if (this.isBigInteger()) {
            return IndexTermType.encodeBigInteger(value);
        }
        if (this.isBigDecimal()) {
            return IndexTermType.encodeDecimal(value);
        }
        return value;
    }

    public float[] decomposeVector(ByteBuffer byteBuffer) {
        assert (this.isVector());
        return ((VectorType)this.indexType).composeAsFloat(byteBuffer);
    }

    public boolean supports(Operator operator) {
        if (operator == Operator.LIKE || operator == Operator.LIKE_CONTAINS || operator == Operator.LIKE_PREFIX || operator == Operator.LIKE_MATCHES || operator == Operator.LIKE_SUFFIX) {
            return false;
        }
        if (operator == Operator.ANN) {
            return this.isVector();
        }
        Expression.IndexOperator indexOperator = Expression.IndexOperator.valueOf(operator);
        if (this.isNonFrozenCollection()) {
            if (this.indexTargetType == IndexTarget.Type.KEYS) {
                return indexOperator == Expression.IndexOperator.CONTAINS_KEY;
            }
            if (this.indexTargetType == IndexTarget.Type.VALUES) {
                return indexOperator == Expression.IndexOperator.CONTAINS_VALUE;
            }
            return this.indexTargetType == IndexTarget.Type.KEYS_AND_VALUES && indexOperator == Expression.IndexOperator.EQ;
        }
        if (this.indexTargetType == IndexTarget.Type.FULL) {
            return indexOperator == Expression.IndexOperator.EQ;
        }
        if (indexOperator != Expression.IndexOperator.EQ && EQ_ONLY_TYPES.contains(this.indexType)) {
            return false;
        }
        return indexOperator != null && (!this.isLiteral() || indexOperator != Expression.IndexOperator.RANGE);
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("column", (Object)this.columnMetadata).add("type", this.indexType).add("indexType", (Object)this.indexTargetType).toString();
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof IndexTermType)) {
            return false;
        }
        IndexTermType other = (IndexTermType)obj;
        return Objects.equals(this.columnMetadata, other.columnMetadata) && this.indexTargetType == other.indexTargetType;
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.columnMetadata, this.indexTargetType});
    }

    private EnumSet<Capability> calculateCapabilities(ColumnMetadata columnMetadata, List<ColumnMetadata> partitionKeyColumns, IndexTarget.Type indexTargetType) {
        AbstractType<?> indexType;
        EnumSet<Capability> capabilities = EnumSet.noneOf(Capability.class);
        if (partitionKeyColumns.contains(columnMetadata) && partitionKeyColumns.size() > 1) {
            capabilities.add(Capability.COMPOSITE_PARTITION);
        }
        AbstractType type = columnMetadata.type;
        boolean reversed = type.isReversed();
        AbstractType baseType = type.unwrap();
        if (baseType.isCollection()) {
            capabilities.add(Capability.COLLECTION);
        }
        if (baseType.isCollection() && baseType.isMultiCell()) {
            capabilities.add(Capability.NON_FROZEN_COLLECTION);
        }
        if (!baseType.subTypes().isEmpty() && !baseType.isMultiCell()) {
            capabilities.add(Capability.FROZEN);
        }
        if ((indexType = this.calculateIndexType(baseType, capabilities, indexTargetType)) instanceof CompositeType) {
            capabilities.add(Capability.COMPOSITE);
        } else if (!indexType.subTypes().isEmpty() && !indexType.isMultiCell()) {
            capabilities.add(Capability.FROZEN);
        }
        if (indexType instanceof StringType) {
            capabilities.add(Capability.STRING);
        }
        if (indexType instanceof BooleanType) {
            capabilities.add(Capability.BOOLEAN);
        }
        if (capabilities.contains((Object)Capability.STRING) || capabilities.contains((Object)Capability.BOOLEAN) || capabilities.contains((Object)Capability.FROZEN) || capabilities.contains((Object)Capability.COMPOSITE)) {
            capabilities.add(Capability.LITERAL);
        }
        if (indexType instanceof VectorType) {
            capabilities.add(Capability.VECTOR);
        }
        if (indexType instanceof InetAddressType) {
            capabilities.add(Capability.INET_ADDRESS);
            reversed = false;
        }
        if (indexType instanceof IntegerType) {
            capabilities.add(Capability.BIG_INTEGER);
            reversed = false;
        }
        if (indexType instanceof DecimalType) {
            capabilities.add(Capability.BIG_DECIMAL);
            reversed = false;
        }
        if (indexType instanceof LongType) {
            capabilities.add(Capability.LONG);
            reversed = false;
        }
        if (reversed) {
            capabilities.add(Capability.REVERSED);
        }
        return capabilities;
    }

    private AbstractType<?> calculateIndexType(AbstractType<?> baseType, EnumSet<Capability> capabilities, IndexTarget.Type indexTargetType) {
        return capabilities.contains((Object)Capability.NON_FROZEN_COLLECTION) ? this.collectionCellValueType(baseType, indexTargetType) : baseType;
    }

    private Iterator<ByteBuffer> collectionIterator(ComplexColumnData cellData, long nowInSecs) {
        if (cellData == null) {
            return null;
        }
        Stream<ByteBuffer> stream = StreamSupport.stream(cellData.spliterator(), false).filter(cell -> cell != null && cell.isLive(nowInSecs)).map(this::cellValue);
        if (this.isInetAddress()) {
            stream = stream.sorted((c1, c2) -> IndexTermType.compareInet(IndexTermType.encodeInetAddress(c1), IndexTermType.encodeInetAddress(c2)));
        }
        return stream.iterator();
    }

    private ByteBuffer cellValue(Cell<?> cell) {
        if (this.isNonFrozenCollection()) {
            switch (((CollectionType)this.columnMetadata.type).kind) {
                case LIST: {
                    return cell.buffer();
                }
                case SET: {
                    return cell.path().get(0);
                }
                case MAP: {
                    switch (this.indexTargetType) {
                        case KEYS: {
                            return cell.path().get(0);
                        }
                        case VALUES: {
                            return cell.buffer();
                        }
                        case KEYS_AND_VALUES: {
                            return CompositeType.build(ByteBufferAccessor.instance, cell.path().get(0), cell.buffer());
                        }
                    }
                }
            }
        }
        return cell.buffer();
    }

    private AbstractType<?> collectionCellValueType(AbstractType<?> type, IndexTarget.Type indexType) {
        CollectionType collection = (CollectionType)type;
        switch (collection.kind) {
            case LIST: {
                return collection.valueComparator();
            }
            case SET: {
                return collection.nameComparator();
            }
            case MAP: {
                switch (indexType) {
                    case KEYS: {
                        return collection.nameComparator();
                    }
                    case VALUES: {
                        return collection.valueComparator();
                    }
                    case KEYS_AND_VALUES: {
                        return CompositeType.getInstance(collection.nameComparator(), collection.valueComparator());
                    }
                }
            }
        }
        throw new IllegalArgumentException("Unsupported collection type: " + collection.kind);
    }

    private boolean isCompositePartition() {
        return this.capabilities.contains((Object)Capability.COMPOSITE_PARTITION);
    }

    private boolean isInetAddress() {
        return this.capabilities.contains((Object)Capability.INET_ADDRESS);
    }

    private boolean isBigInteger() {
        return this.capabilities.contains((Object)Capability.BIG_INTEGER);
    }

    private boolean isBigDecimal() {
        return this.capabilities.contains((Object)Capability.BIG_DECIMAL);
    }

    private boolean isLong() {
        return this.capabilities.contains((Object)Capability.LONG);
    }

    private static int compareInet(ByteBuffer b1, ByteBuffer b2) {
        assert (IndexTermType.isIPv6(b1) && IndexTermType.isIPv6(b2));
        return FastByteOperations.compareUnsigned(b1, b2);
    }

    private static boolean isIPv6(ByteBuffer address) {
        return address.remaining() == 16;
    }

    private static ByteBuffer encodeInetAddress(ByteBuffer value) {
        if (value.remaining() == 4) {
            int position = value.hasArray() ? value.arrayOffset() + value.position() : value.position();
            ByteBuffer mapped = ByteBuffer.allocate(16);
            System.arraycopy(IPV4_PREFIX, 0, mapped.array(), 0, IPV4_PREFIX.length);
            ByteBufferUtil.copyBytes(value, position, mapped, IPV4_PREFIX.length, value.remaining());
            return mapped;
        }
        return value;
    }

    public static ByteBuffer encodeBigInteger(ByteBuffer value) {
        int size = value.remaining();
        int position = value.hasArray() ? value.arrayOffset() + value.position() : value.position();
        byte[] bytes = new byte[20];
        if (size < 16) {
            ByteBufferUtil.copyBytes(value, position, bytes, bytes.length - size, size);
            if ((bytes[bytes.length - size] & 0x80) != 0) {
                Arrays.fill(bytes, 4, bytes.length - size, (byte)-1);
            } else {
                Arrays.fill(bytes, 4, bytes.length - size, (byte)0);
            }
        } else {
            ByteBufferUtil.copyBytes(value, position, bytes, 4, 16);
        }
        if ((bytes[4] & 0x80) != 0) {
            size = -size;
        }
        bytes[0] = (byte)(size >> 24 & 0xFF);
        bytes[1] = (byte)(size >> 16 & 0xFF);
        bytes[2] = (byte)(size >> 8 & 0xFF);
        bytes[3] = (byte)(size & 0xFF);
        bytes[0] = (byte)(bytes[0] ^ 0x80);
        return ByteBuffer.wrap(bytes);
    }

    public static ByteBuffer encodeDecimal(ByteBuffer value) {
        ByteSource bs = DecimalType.instance.asComparableBytes(value, ByteComparable.Version.OSS50);
        bs = ByteSource.cutOrRightPad(bs, 24, 0);
        return ByteBuffer.wrap(ByteSourceInverse.readBytes(bs, 24));
    }

    private static enum Capability {
        STRING,
        VECTOR,
        INET_ADDRESS,
        BIG_INTEGER,
        BIG_DECIMAL,
        LONG,
        BOOLEAN,
        LITERAL,
        REVERSED,
        FROZEN,
        COLLECTION,
        NON_FROZEN_COLLECTION,
        COMPOSITE,
        COMPOSITE_PARTITION;

    }
}

