/*
 *
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.apache.flex.compiler.internal.definitions.mxml;

import org.apache.flex.compiler.common.ISourceLocation;
import org.apache.flex.compiler.definitions.references.IReference;
import org.apache.flex.compiler.internal.definitions.ParameterDefinition;
import org.apache.flex.compiler.internal.scopes.ASScope;
import org.apache.flex.compiler.internal.scopes.FunctionScope;
import org.apache.flex.compiler.tree.mxml.IMXMLEventSpecifierNode;

/**
 * Sub-class of {@link FunctionScope} for MXML event handlers.
 */
public final class MXMLEventHandlerScope extends FunctionScope
{
    /**
     * The name of the argument of the autogenerated event handler method.
     */
    private static final String EVENT_HANDLER_PARAMETER_NAME = "event";
    
    public MXMLEventHandlerScope(IMXMLEventSpecifierNode eventSpecifierNode)
    {
        super((ASScope)eventSpecifierNode.getClassDefinitionNode().getClassDefinition().getContainedScope());
        this.eventSpecifierNode = eventSpecifierNode;
    }
    
    /**
     * The definition for the handler's 'event' parameter.
     */
    private ParameterDefinition eventParameter;
    private final IMXMLEventSpecifierNode eventSpecifierNode;
    
    public void buildEventParameter(IReference typeRef)
    {
        assert eventParameter == null : "Event parameter should only be built once!";
        eventParameter = new ParameterDefinition(EVENT_HANDLER_PARAMETER_NAME);
        eventParameter.setTypeReference(typeRef);
        eventParameter.setPublic();
        eventParameter.setImplicit();
        addDefinition(eventParameter);
    }
    
    /**
     * Gets the definition of the <code>event</code> argument of the implied
     * event handler method.
     * 
     * @return An {@link ParameterDefinition} object.
     */
    public ParameterDefinition getEventParameterDefinition()
    {
        return eventParameter;
    }
    
    /**
     * Gets the {@link ISourceLocation} of this scope.
     * @return An {@link ISourceLocation}.
     */
    public ISourceLocation getLocation()
    {
        return eventSpecifierNode;
    }

}
