/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.metastorage.server;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.ignite.internal.close.ManuallyCloseable;
import org.apache.ignite.internal.failure.FailureContext;
import org.apache.ignite.internal.failure.FailureManager;
import org.apache.ignite.internal.failure.FailureType;
import org.apache.ignite.internal.hlc.HybridTimestamp;
import org.apache.ignite.internal.lang.NodeStoppingException;
import org.apache.ignite.internal.logger.IgniteLogger;
import org.apache.ignite.internal.logger.Loggers;
import org.apache.ignite.internal.metastorage.CompactionRevisionUpdateListener;
import org.apache.ignite.internal.metastorage.Entry;
import org.apache.ignite.internal.metastorage.EntryEvent;
import org.apache.ignite.internal.metastorage.RevisionUpdateListener;
import org.apache.ignite.internal.metastorage.WatchEvent;
import org.apache.ignite.internal.metastorage.WatchListener;
import org.apache.ignite.internal.metastorage.server.Watch;
import org.apache.ignite.internal.metastorage.server.WatchAndEvents;
import org.apache.ignite.internal.metastorage.server.WatchEventHandlingCallback;
import org.apache.ignite.internal.metastorage.server.raft.MetaStorageWriteHandler;
import org.apache.ignite.internal.thread.IgniteThreadFactory;
import org.apache.ignite.internal.thread.ThreadOperation;
import org.apache.ignite.internal.util.CompletableFutures;
import org.apache.ignite.internal.util.ExceptionUtils;
import org.apache.ignite.internal.util.IgniteUtils;

public class WatchProcessor
implements ManuallyCloseable {
    private static final IgniteLogger LOG = Loggers.forClass(WatchProcessor.class);
    private static final int WATCH_EVENT_PROCESSING_LOG_THRESHOLD_MILLIS = 100;
    private static final int WATCH_EVENT_PROCESSING_LOG_KEYS = 10;
    private final List<Watch> watches = new CopyOnWriteArrayList<Watch>();
    private volatile CompletableFuture<Void> notificationFuture = CompletableFutures.nullCompletedFuture();
    private final EntryReader entryReader;
    private volatile WatchEventHandlingCallback watchEventHandlingCallback;
    private final ExecutorService watchExecutor;
    private final List<RevisionUpdateListener> revisionUpdateListeners = new CopyOnWriteArrayList<RevisionUpdateListener>();
    private final List<CompactionRevisionUpdateListener> compactionRevisionUpdateListeners = new CopyOnWriteArrayList<CompactionRevisionUpdateListener>();
    private final FailureManager failureManager;
    private final AtomicBoolean firedFailureOnChain = new AtomicBoolean(false);

    public WatchProcessor(String nodeName, EntryReader entryReader, FailureManager failureManager) {
        this.entryReader = entryReader;
        this.watchExecutor = Executors.newFixedThreadPool(4, IgniteThreadFactory.create((String)nodeName, (String)"metastorage-watch-executor", (IgniteLogger)LOG, (ThreadOperation[])ThreadOperation.NOTHING_ALLOWED));
        this.failureManager = failureManager;
    }

    public void addWatch(Watch watch) {
        this.watches.add(watch);
    }

    void removeWatch(WatchListener listener) {
        this.watches.removeIf(watch -> watch.listener() == listener);
    }

    public OptionalLong minWatchRevision() {
        return this.watches.stream().mapToLong(Watch::startRevision).min();
    }

    public void setWatchEventHandlingCallback(WatchEventHandlingCallback callback) {
        assert (this.watchEventHandlingCallback == null);
        this.watchEventHandlingCallback = callback;
    }

    public CompletableFuture<Void> notifyWatches(List<Entry> updatedEntries, HybridTimestamp time) {
        assert (time != null);
        CompletionStage newFuture = ((CompletableFuture)this.notificationFuture.thenComposeAsync(v -> {
            long newRevision = ((Entry)updatedEntries.get(0)).revision();
            List<Entry> filteredUpdatedEntries = updatedEntries.stream().filter(WatchProcessor::isNotIdempotentCacheCommand).collect(Collectors.toList());
            List<WatchAndEvents> watchAndEvents = this.collectWatchesAndEvents(filteredUpdatedEntries, newRevision);
            long startTimeNanos = System.nanoTime();
            CompletableFuture<Void> notifyWatchesFuture = WatchProcessor.notifyWatches(watchAndEvents, newRevision, time);
            CompletableFuture<Void> notifyUpdateRevisionFuture = this.notifyUpdateRevisionListeners(newRevision);
            CompletionStage notificationFuture = CompletableFuture.allOf(notifyWatchesFuture, notifyUpdateRevisionFuture).thenRunAsync(() -> this.invokeOnRevisionCallback(newRevision, time), this.watchExecutor);
            ((CompletableFuture)notificationFuture).whenComplete((unused, e) -> WatchProcessor.maybeLogLongProcessing(filteredUpdatedEntries, startTimeNanos));
            return notificationFuture;
        }, (Executor)this.watchExecutor)).whenComplete((unused, e) -> {
            if (e != null) {
                this.notifyFailureHandlerOnFirstFailureInNotificationChain((Throwable)e);
            }
        });
        this.notificationFuture = newFuture;
        return newFuture;
    }

    private static CompletableFuture<Void> notifyWatches(List<WatchAndEvents> watchAndEventsList, long revision, HybridTimestamp time) {
        if (watchAndEventsList.isEmpty()) {
            return CompletableFutures.nullCompletedFuture();
        }
        CompletableFuture[] notifyWatchFutures = new CompletableFuture[watchAndEventsList.size()];
        for (int i = 0; i < watchAndEventsList.size(); ++i) {
            CompletableFuture<Object> notifyWatchFuture;
            WatchAndEvents watchAndEvents = watchAndEventsList.get(i);
            try {
                WatchEvent event = new WatchEvent(watchAndEvents.events, revision, time);
                notifyWatchFuture = watchAndEvents.watch.onUpdate(event);
            }
            catch (Throwable throwable) {
                notifyWatchFuture = CompletableFuture.failedFuture(throwable);
            }
            notifyWatchFutures[i] = notifyWatchFuture;
        }
        return CompletableFuture.allOf(notifyWatchFutures);
    }

    private static void maybeLogLongProcessing(List<Entry> updatedEntries, long startTimeNanos) {
        long durationMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos);
        if (durationMillis > 100L) {
            String keysHead = updatedEntries.stream().limit(10L).map(entry -> new String(entry.key(), StandardCharsets.UTF_8)).collect(Collectors.joining(", "));
            String keysTail = updatedEntries.size() > 10 ? ", ..." : "";
            LOG.warn("Watch event processing has been too long [duration={}, keys=[{}{}]]", new Object[]{durationMillis, keysHead, keysTail});
        }
    }

    private List<WatchAndEvents> collectWatchesAndEvents(List<Entry> updatedEntries, long revision) {
        if (this.watches.isEmpty()) {
            return List.of();
        }
        ArrayList<WatchAndEvents> watchAndEvents = new ArrayList<WatchAndEvents>();
        for (Watch watch : this.watches) {
            List<EntryEvent> events = List.of();
            for (Entry newEntry : updatedEntries) {
                byte[] newKey = newEntry.key();
                assert (newEntry.revision() == revision);
                if (!watch.matches(newKey, revision)) continue;
                Entry oldEntry = this.entryReader.get(newKey, revision - 1L);
                if (events.isEmpty()) {
                    events = new ArrayList<EntryEvent>();
                }
                events.add(new EntryEvent(oldEntry, newEntry));
            }
            if (events.isEmpty()) continue;
            watchAndEvents.add(new WatchAndEvents(watch, events));
        }
        return watchAndEvents;
    }

    private void invokeOnRevisionCallback(long revision, HybridTimestamp time) {
        this.watchEventHandlingCallback.onSafeTimeAdvanced(time);
        this.watchEventHandlingCallback.onRevisionApplied(revision);
    }

    public void advanceSafeTime(HybridTimestamp time) {
        assert (time != null);
        this.notificationFuture = ((CompletableFuture)this.notificationFuture.thenRunAsync(() -> this.watchEventHandlingCallback.onSafeTimeAdvanced(time), this.watchExecutor)).whenComplete((ignored, e) -> {
            if (e != null) {
                this.notifyFailureHandlerOnFirstFailureInNotificationChain((Throwable)e);
            }
        });
    }

    private void notifyFailureHandlerOnFirstFailureInNotificationChain(Throwable e) {
        if (this.firedFailureOnChain.compareAndSet(false, true)) {
            boolean nodeStopping = ExceptionUtils.hasCauseOrSuppressed((Throwable)e, (Class[])new Class[]{NodeStoppingException.class});
            if (!nodeStopping) {
                LOG.error("Notification chain encountered an error, so no notifications will be ever fired for subsequent revisions until a restart. Notifying the FailureManager", new Object[0]);
                this.failureManager.process(new FailureContext(FailureType.CRITICAL_ERROR, e));
            } else {
                LOG.info("Notification chain encountered a NodeStoppingException, so no notifications will be ever fired for subsequent revisions until a restart.", new Object[0]);
            }
        }
    }

    public void close() {
        this.notificationFuture.cancel(true);
        IgniteUtils.shutdownAndAwaitTermination((ExecutorService)this.watchExecutor, (long)10L, (TimeUnit)TimeUnit.SECONDS);
    }

    void registerRevisionUpdateListener(RevisionUpdateListener listener) {
        this.revisionUpdateListeners.add(listener);
    }

    void unregisterRevisionUpdateListener(RevisionUpdateListener listener) {
        this.revisionUpdateListeners.remove(listener);
    }

    void registerCompactionRevisionUpdateListener(CompactionRevisionUpdateListener listener) {
        this.compactionRevisionUpdateListeners.add(listener);
    }

    void unregisterCompactionRevisionUpdateListener(CompactionRevisionUpdateListener listener) {
        this.compactionRevisionUpdateListeners.remove(listener);
    }

    CompletableFuture<Void> notifyUpdateRevisionListeners(long newRevision) {
        List<CompletableFuture> futures = List.of();
        for (RevisionUpdateListener listener : this.revisionUpdateListeners) {
            if (futures.isEmpty()) {
                futures = new ArrayList();
            }
            futures.add(listener.onUpdated(newRevision));
        }
        return futures.isEmpty() ? CompletableFutures.nullCompletedFuture() : CompletableFuture.allOf((CompletableFuture[])futures.toArray(CompletableFuture[]::new));
    }

    void updateCompactionRevision(long compactionRevision, HybridTimestamp time) {
        this.notificationFuture = ((CompletableFuture)this.notificationFuture.thenRunAsync(() -> {
            this.compactionRevisionUpdateListeners.forEach(listener -> listener.onUpdate(compactionRevision));
            this.watchEventHandlingCallback.onSafeTimeAdvanced(time);
        }, this.watchExecutor)).whenComplete((ignored, e) -> {
            if (e != null) {
                this.notifyFailureHandlerOnFirstFailureInNotificationChain((Throwable)e);
            }
        });
    }

    void updateOnlyRevision(long newRevision, HybridTimestamp time) {
        this.notificationFuture = ((CompletableFuture)((CompletableFuture)this.notificationFuture.thenComposeAsync(unused -> this.notifyUpdateRevisionListeners(newRevision), (Executor)this.watchExecutor)).thenRunAsync(() -> this.invokeOnRevisionCallback(newRevision, time), this.watchExecutor)).whenComplete((ignored, e) -> {
            if (e != null) {
                this.notifyFailureHandlerOnFirstFailureInNotificationChain((Throwable)e);
            }
        });
    }

    private static boolean isNotIdempotentCacheCommand(Entry entry) {
        int prefixLength = MetaStorageWriteHandler.IDEMPOTENT_COMMAND_PREFIX_BYTES.length;
        if (entry.key().length <= prefixLength) {
            return true;
        }
        return !Arrays.equals(entry.key(), 0, prefixLength, MetaStorageWriteHandler.IDEMPOTENT_COMMAND_PREFIX_BYTES, 0, prefixLength);
    }

    @FunctionalInterface
    public static interface EntryReader {
        public Entry get(byte[] var1, long var2);
    }
}

