/*
 * Decompiled with CFR 0.152.
 */
package com.google.crypto.tink.jwt;

import com.google.crypto.tink.KeyTemplate;
import com.google.crypto.tink.KeysetHandle;
import com.google.crypto.tink.KeysetManager;
import com.google.crypto.tink.KeysetWriter;
import com.google.crypto.tink.jwt.JwtFormat;
import com.google.crypto.tink.proto.EncryptedKeyset;
import com.google.crypto.tink.proto.JwtEcdsaAlgorithm;
import com.google.crypto.tink.proto.JwtEcdsaPublicKey;
import com.google.crypto.tink.proto.JwtRsaSsaPkcs1Algorithm;
import com.google.crypto.tink.proto.JwtRsaSsaPkcs1PublicKey;
import com.google.crypto.tink.proto.JwtRsaSsaPssAlgorithm;
import com.google.crypto.tink.proto.JwtRsaSsaPssPublicKey;
import com.google.crypto.tink.proto.KeyData;
import com.google.crypto.tink.proto.KeyStatusType;
import com.google.crypto.tink.proto.Keyset;
import com.google.crypto.tink.proto.KeysetInfo;
import com.google.crypto.tink.proto.OutputPrefixType;
import com.google.crypto.tink.subtle.Base64;
import com.google.crypto.tink.tinkkey.KeyAccess;
import com.google.crypto.tink.tinkkey.KeyHandle;
import com.google.crypto.tink.tinkkey.internal.ProtoKey;
import com.google.errorprone.annotations.InlineMe;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.internal.Streams;
import com.google.gson.stream.JsonReader;
import com.google.protobuf.ByteString;
import com.google.protobuf.ExtensionRegistryLite;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.Optional;

public final class JwkSetConverter {
    private static final String JWT_ECDSA_PUBLIC_KEY_URL = "type.googleapis.com/google.crypto.tink.JwtEcdsaPublicKey";
    private static final String JWT_RSA_SSA_PKCS1_PUBLIC_KEY_URL = "type.googleapis.com/google.crypto.tink.JwtRsaSsaPkcs1PublicKey";
    private static final String JWT_RSA_SSA_PSS_PUBLIC_KEY_URL = "type.googleapis.com/google.crypto.tink.JwtRsaSsaPssPublicKey";

    public static String fromPublicKeysetHandle(KeysetHandle handle) throws IOException, GeneralSecurityException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        handle.writeNoSecret(new JwkSetWriter(outputStream));
        return outputStream.toString();
    }

    public static KeysetHandle toPublicKeysetHandle(String jwkSet) throws IOException, GeneralSecurityException {
        JsonObject jsonKeyset;
        try {
            JsonReader jsonReader = new JsonReader((Reader)new StringReader(jwkSet));
            jsonReader.setLenient(false);
            jsonKeyset = Streams.parse((JsonReader)jsonReader).getAsJsonObject();
        }
        catch (JsonParseException | IllegalStateException | StackOverflowError ex) {
            throw new IOException("JWK set is invalid JSON", ex);
        }
        KeysetManager manager = KeysetManager.withEmptyKeyset();
        JsonArray jsonKeys = jsonKeyset.get("keys").getAsJsonArray();
        for (JsonElement element : jsonKeys) {
            KeyData keyData;
            String algPrefix;
            JsonObject jsonKey = element.getAsJsonObject();
            switch (algPrefix = JwkSetConverter.getStringItem(jsonKey, "alg").substring(0, 2)) {
                case "RS": {
                    keyData = JwkSetConverter.convertToRsaSsaPkcs1Key(jsonKey);
                    break;
                }
                case "PS": {
                    keyData = JwkSetConverter.convertToRsaSsaPssKey(jsonKey);
                    break;
                }
                case "ES": {
                    keyData = JwkSetConverter.convertToEcdsaKey(jsonKey);
                    break;
                }
                default: {
                    throw new IOException("unexpected alg value: " + JwkSetConverter.getStringItem(jsonKey, "alg"));
                }
            }
            manager.add(KeyHandle.createFromKey(new ProtoKey(keyData, KeyTemplate.OutputPrefixType.RAW), KeyAccess.publicAccess()));
        }
        KeysetInfo info = manager.getKeysetHandle().getKeysetInfo();
        if (info.getKeyInfoCount() <= 0) {
            throw new IOException("empty keyset");
        }
        manager.setPrimary(info.getKeyInfo(0).getKeyId());
        return manager.getKeysetHandle();
    }

    private static String getStringItem(JsonObject obj, String name) throws IOException {
        if (!obj.has(name)) {
            throw new IOException(name + " not found");
        }
        if (!obj.get(name).isJsonPrimitive() || !obj.get(name).getAsJsonPrimitive().isString()) {
            throw new IOException(name + " is not a string");
        }
        return obj.get(name).getAsString();
    }

    private static void expectStringItem(JsonObject obj, String name, String expectedValue) throws IOException {
        String value = JwkSetConverter.getStringItem(obj, name);
        if (!value.equals(expectedValue)) {
            throw new IOException("unexpected " + name + " value: " + value);
        }
    }

    private static void validateUseIsSig(JsonObject jsonKey) throws IOException {
        if (!jsonKey.has("use")) {
            return;
        }
        JwkSetConverter.expectStringItem(jsonKey, "use", "sig");
    }

    private static void validateKeyOpsIsVerify(JsonObject jsonKey) throws IOException {
        if (!jsonKey.has("key_ops")) {
            return;
        }
        if (!jsonKey.get("key_ops").isJsonArray()) {
            throw new IOException("key_ops is not an array");
        }
        JsonArray keyOps = jsonKey.get("key_ops").getAsJsonArray();
        if (keyOps.size() != 1) {
            throw new IOException("key_ops must contain exactly one element");
        }
        if (!keyOps.get(0).isJsonPrimitive() || !keyOps.get(0).getAsJsonPrimitive().isString()) {
            throw new IOException("key_ops is not a string");
        }
        if (!keyOps.get(0).getAsString().equals("verify")) {
            throw new IOException("unexpected keyOps value: " + keyOps.get(0).getAsString());
        }
    }

    private static KeyData convertToRsaSsaPkcs1Key(JsonObject jsonKey) throws IOException {
        JwtRsaSsaPkcs1Algorithm algorithm;
        switch (JwkSetConverter.getStringItem(jsonKey, "alg")) {
            case "RS256": {
                algorithm = JwtRsaSsaPkcs1Algorithm.RS256;
                break;
            }
            case "RS384": {
                algorithm = JwtRsaSsaPkcs1Algorithm.RS384;
                break;
            }
            case "RS512": {
                algorithm = JwtRsaSsaPkcs1Algorithm.RS512;
                break;
            }
            default: {
                throw new IOException("Unknown Rsa Algorithm: " + JwkSetConverter.getStringItem(jsonKey, "alg"));
            }
        }
        if (jsonKey.has("p") || jsonKey.has("q") || jsonKey.has("dp") || jsonKey.has("dq") || jsonKey.has("d") || jsonKey.has("qi")) {
            throw new UnsupportedOperationException("importing RSA private keys is not implemented");
        }
        JwkSetConverter.expectStringItem(jsonKey, "kty", "RSA");
        JwkSetConverter.validateUseIsSig(jsonKey);
        JwkSetConverter.validateKeyOpsIsVerify(jsonKey);
        JwtRsaSsaPkcs1PublicKey.Builder pkcs1PubKeyBuilder = JwtRsaSsaPkcs1PublicKey.newBuilder().setVersion(0).setAlgorithm(algorithm).setE(ByteString.copyFrom((byte[])Base64.urlSafeDecode(JwkSetConverter.getStringItem(jsonKey, "e")))).setN(ByteString.copyFrom((byte[])Base64.urlSafeDecode(JwkSetConverter.getStringItem(jsonKey, "n"))));
        if (jsonKey.has("kid")) {
            pkcs1PubKeyBuilder.setCustomKid(JwtRsaSsaPkcs1PublicKey.CustomKid.newBuilder().setValue(JwkSetConverter.getStringItem(jsonKey, "kid")).build());
        }
        return KeyData.newBuilder().setTypeUrl(JWT_RSA_SSA_PKCS1_PUBLIC_KEY_URL).setValue(pkcs1PubKeyBuilder.build().toByteString()).setKeyMaterialType(KeyData.KeyMaterialType.ASYMMETRIC_PUBLIC).build();
    }

    private static KeyData convertToRsaSsaPssKey(JsonObject jsonKey) throws IOException {
        JwtRsaSsaPssAlgorithm algorithm;
        switch (JwkSetConverter.getStringItem(jsonKey, "alg")) {
            case "PS256": {
                algorithm = JwtRsaSsaPssAlgorithm.PS256;
                break;
            }
            case "PS384": {
                algorithm = JwtRsaSsaPssAlgorithm.PS384;
                break;
            }
            case "PS512": {
                algorithm = JwtRsaSsaPssAlgorithm.PS512;
                break;
            }
            default: {
                throw new IOException("Unknown Rsa Algorithm: " + JwkSetConverter.getStringItem(jsonKey, "alg"));
            }
        }
        if (jsonKey.has("p") || jsonKey.has("q") || jsonKey.has("dq") || jsonKey.has("dq") || jsonKey.has("d") || jsonKey.has("qi")) {
            throw new UnsupportedOperationException("importing RSA private keys is not implemented");
        }
        JwkSetConverter.expectStringItem(jsonKey, "kty", "RSA");
        JwkSetConverter.validateUseIsSig(jsonKey);
        JwkSetConverter.validateKeyOpsIsVerify(jsonKey);
        JwtRsaSsaPssPublicKey.Builder pssPubKeyBuilder = JwtRsaSsaPssPublicKey.newBuilder().setVersion(0).setAlgorithm(algorithm).setE(ByteString.copyFrom((byte[])Base64.urlSafeDecode(JwkSetConverter.getStringItem(jsonKey, "e")))).setN(ByteString.copyFrom((byte[])Base64.urlSafeDecode(JwkSetConverter.getStringItem(jsonKey, "n"))));
        if (jsonKey.has("kid")) {
            pssPubKeyBuilder.setCustomKid(JwtRsaSsaPssPublicKey.CustomKid.newBuilder().setValue(JwkSetConverter.getStringItem(jsonKey, "kid")).build());
        }
        return KeyData.newBuilder().setTypeUrl(JWT_RSA_SSA_PSS_PUBLIC_KEY_URL).setValue(pssPubKeyBuilder.build().toByteString()).setKeyMaterialType(KeyData.KeyMaterialType.ASYMMETRIC_PUBLIC).build();
    }

    private static KeyData convertToEcdsaKey(JsonObject jsonKey) throws IOException {
        JwtEcdsaAlgorithm algorithm;
        switch (JwkSetConverter.getStringItem(jsonKey, "alg")) {
            case "ES256": {
                JwkSetConverter.expectStringItem(jsonKey, "crv", "P-256");
                algorithm = JwtEcdsaAlgorithm.ES256;
                break;
            }
            case "ES384": {
                JwkSetConverter.expectStringItem(jsonKey, "crv", "P-384");
                algorithm = JwtEcdsaAlgorithm.ES384;
                break;
            }
            case "ES512": {
                JwkSetConverter.expectStringItem(jsonKey, "crv", "P-521");
                algorithm = JwtEcdsaAlgorithm.ES512;
                break;
            }
            default: {
                throw new IOException("Unknown Ecdsa Algorithm: " + JwkSetConverter.getStringItem(jsonKey, "alg"));
            }
        }
        if (jsonKey.has("d")) {
            throw new UnsupportedOperationException("importing ECDSA private keys is not implemented");
        }
        JwkSetConverter.expectStringItem(jsonKey, "kty", "EC");
        JwkSetConverter.validateUseIsSig(jsonKey);
        JwkSetConverter.validateKeyOpsIsVerify(jsonKey);
        JwtEcdsaPublicKey.Builder ecdsaPubKeyBuilder = JwtEcdsaPublicKey.newBuilder().setVersion(0).setAlgorithm(algorithm).setX(ByteString.copyFrom((byte[])Base64.urlSafeDecode(JwkSetConverter.getStringItem(jsonKey, "x")))).setY(ByteString.copyFrom((byte[])Base64.urlSafeDecode(JwkSetConverter.getStringItem(jsonKey, "y"))));
        if (jsonKey.has("kid")) {
            ecdsaPubKeyBuilder.setCustomKid(JwtEcdsaPublicKey.CustomKid.newBuilder().setValue(JwkSetConverter.getStringItem(jsonKey, "kid")).build());
        }
        return KeyData.newBuilder().setTypeUrl(JWT_ECDSA_PUBLIC_KEY_URL).setValue(ecdsaPubKeyBuilder.build().toByteString()).setKeyMaterialType(KeyData.KeyMaterialType.ASYMMETRIC_PUBLIC).build();
    }

    @Deprecated
    @InlineMe(replacement="JwkSetConverter.fromPublicKeysetHandle(handle)", imports={"com.google.crypto.tink.jwt.JwkSetConverter"})
    public static String fromKeysetHandle(KeysetHandle handle, KeyAccess keyAccess) throws IOException, GeneralSecurityException {
        return JwkSetConverter.fromPublicKeysetHandle(handle);
    }

    @Deprecated
    @InlineMe(replacement="JwkSetConverter.toPublicKeysetHandle(jwkSet)", imports={"com.google.crypto.tink.jwt.JwkSetConverter"})
    public static KeysetHandle toKeysetHandle(String jwkSet, KeyAccess keyAccess) throws IOException, GeneralSecurityException {
        return JwkSetConverter.toPublicKeysetHandle(jwkSet);
    }

    private JwkSetConverter() {
    }

    private static final class JwkSetWriter
    implements KeysetWriter {
        private final OutputStream outputStream;

        private JwkSetWriter(OutputStream outputStream) {
            this.outputStream = outputStream;
        }

        @Override
        public void write(Keyset keyset) throws IOException {
            JsonObject jwkSet;
            try {
                jwkSet = JwkSetWriter.convertKeyset(keyset);
            }
            catch (GeneralSecurityException exception) {
                throw new IOException(exception);
            }
            this.outputStream.write(jwkSet.toString().getBytes(StandardCharsets.UTF_8));
        }

        @Override
        public void write(EncryptedKeyset keyset) {
            throw new UnsupportedOperationException("EncryptedKeyset are not implemented");
        }

        private static JsonObject convertKeyset(Keyset keyset) throws IOException, GeneralSecurityException {
            JsonArray keys = new JsonArray();
            block10: for (Keyset.Key key : keyset.getKeyList()) {
                if (key.getStatus() != KeyStatusType.ENABLED) continue;
                if (key.getKeyData().getKeyMaterialType() != KeyData.KeyMaterialType.ASYMMETRIC_PUBLIC) {
                    throw new GeneralSecurityException("only public keys can be converted");
                }
                if (key.getOutputPrefixType() != OutputPrefixType.RAW && key.getOutputPrefixType() != OutputPrefixType.TINK) {
                    throw new GeneralSecurityException("only OutputPrefixType RAW and TINK are supported");
                }
                switch (key.getKeyData().getTypeUrl()) {
                    case "type.googleapis.com/google.crypto.tink.JwtEcdsaPublicKey": {
                        keys.add((JsonElement)JwkSetWriter.convertJwtEcdsaKey(key));
                        continue block10;
                    }
                    case "type.googleapis.com/google.crypto.tink.JwtRsaSsaPkcs1PublicKey": {
                        keys.add((JsonElement)JwkSetWriter.convertJwtRsaSsaPkcs1(key));
                        continue block10;
                    }
                    case "type.googleapis.com/google.crypto.tink.JwtRsaSsaPssPublicKey": {
                        keys.add((JsonElement)JwkSetWriter.convertJwtRsaSsaPss(key));
                        continue block10;
                    }
                }
                throw new GeneralSecurityException(String.format("key type %s is not supported", key.getKeyData().getTypeUrl()));
            }
            JsonObject jwkSet = new JsonObject();
            jwkSet.add("keys", (JsonElement)keys);
            return jwkSet;
        }

        private static JsonObject convertJwtEcdsaKey(Keyset.Key key) throws IOException, GeneralSecurityException {
            String crv;
            String alg;
            JwtEcdsaPublicKey jwtEcdsaPublicKey = JwtEcdsaPublicKey.parseFrom(key.getKeyData().getValue(), ExtensionRegistryLite.getEmptyRegistry());
            switch (jwtEcdsaPublicKey.getAlgorithm()) {
                case ES256: {
                    alg = "ES256";
                    crv = "P-256";
                    break;
                }
                case ES384: {
                    alg = "ES384";
                    crv = "P-384";
                    break;
                }
                case ES512: {
                    alg = "ES512";
                    crv = "P-521";
                    break;
                }
                default: {
                    throw new GeneralSecurityException("unknown algorithm");
                }
            }
            JsonObject jsonKey = new JsonObject();
            jsonKey.addProperty("kty", "EC");
            jsonKey.addProperty("crv", crv);
            jsonKey.addProperty("x", Base64.urlSafeEncode(jwtEcdsaPublicKey.getX().toByteArray()));
            jsonKey.addProperty("y", Base64.urlSafeEncode(jwtEcdsaPublicKey.getY().toByteArray()));
            jsonKey.addProperty("use", "sig");
            jsonKey.addProperty("alg", alg);
            JsonArray keyOps = new JsonArray();
            keyOps.add("verify");
            jsonKey.add("key_ops", (JsonElement)keyOps);
            Optional<String> kid = JwtFormat.getKid(key.getKeyId(), key.getOutputPrefixType());
            if (kid.isPresent()) {
                jsonKey.addProperty("kid", kid.get());
            } else if (jwtEcdsaPublicKey.hasCustomKid()) {
                jsonKey.addProperty("kid", jwtEcdsaPublicKey.getCustomKid().getValue());
            }
            return jsonKey;
        }

        private static JsonObject convertJwtRsaSsaPkcs1(Keyset.Key key) throws IOException, GeneralSecurityException {
            String alg;
            JwtRsaSsaPkcs1PublicKey jwtRsaSsaPkcs1PublicKey = JwtRsaSsaPkcs1PublicKey.parseFrom(key.getKeyData().getValue(), ExtensionRegistryLite.getEmptyRegistry());
            switch (jwtRsaSsaPkcs1PublicKey.getAlgorithm()) {
                case RS256: {
                    alg = "RS256";
                    break;
                }
                case RS384: {
                    alg = "RS384";
                    break;
                }
                case RS512: {
                    alg = "RS512";
                    break;
                }
                default: {
                    throw new GeneralSecurityException("unknown algorithm");
                }
            }
            JsonObject jsonKey = new JsonObject();
            jsonKey.addProperty("kty", "RSA");
            jsonKey.addProperty("n", Base64.urlSafeEncode(jwtRsaSsaPkcs1PublicKey.getN().toByteArray()));
            jsonKey.addProperty("e", Base64.urlSafeEncode(jwtRsaSsaPkcs1PublicKey.getE().toByteArray()));
            jsonKey.addProperty("use", "sig");
            jsonKey.addProperty("alg", alg);
            JsonArray keyOps = new JsonArray();
            keyOps.add("verify");
            jsonKey.add("key_ops", (JsonElement)keyOps);
            Optional<String> kid = JwtFormat.getKid(key.getKeyId(), key.getOutputPrefixType());
            if (kid.isPresent()) {
                jsonKey.addProperty("kid", kid.get());
            } else if (jwtRsaSsaPkcs1PublicKey.hasCustomKid()) {
                jsonKey.addProperty("kid", jwtRsaSsaPkcs1PublicKey.getCustomKid().getValue());
            }
            return jsonKey;
        }

        private static JsonObject convertJwtRsaSsaPss(Keyset.Key key) throws IOException, GeneralSecurityException {
            String alg;
            JwtRsaSsaPssPublicKey jwtRsaSsaPssPublicKey = JwtRsaSsaPssPublicKey.parseFrom(key.getKeyData().getValue(), ExtensionRegistryLite.getEmptyRegistry());
            switch (jwtRsaSsaPssPublicKey.getAlgorithm()) {
                case PS256: {
                    alg = "PS256";
                    break;
                }
                case PS384: {
                    alg = "PS384";
                    break;
                }
                case PS512: {
                    alg = "PS512";
                    break;
                }
                default: {
                    throw new GeneralSecurityException("unknown algorithm");
                }
            }
            JsonObject jsonKey = new JsonObject();
            jsonKey.addProperty("kty", "RSA");
            jsonKey.addProperty("n", Base64.urlSafeEncode(jwtRsaSsaPssPublicKey.getN().toByteArray()));
            jsonKey.addProperty("e", Base64.urlSafeEncode(jwtRsaSsaPssPublicKey.getE().toByteArray()));
            jsonKey.addProperty("use", "sig");
            jsonKey.addProperty("alg", alg);
            JsonArray keyOps = new JsonArray();
            keyOps.add("verify");
            jsonKey.add("key_ops", (JsonElement)keyOps);
            Optional<String> kid = JwtFormat.getKid(key.getKeyId(), key.getOutputPrefixType());
            if (kid.isPresent()) {
                jsonKey.addProperty("kid", kid.get());
            } else if (jwtRsaSsaPssPublicKey.hasCustomKid()) {
                jsonKey.addProperty("kid", jwtRsaSsaPssPublicKey.getCustomKid().getValue());
            }
            return jsonKey;
        }
    }
}

