"use strict";
/*!
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.shallowMerge = exports.simpleShallowMerge = exports.deepCopy = exports.buildPath = exports.resolve = exports.appendIf = exports.assignIf = exports.append = exports.assign = void 0;
const Es2019Array_1 = require("./Es2019Array");
/**
 * A nop as assign functionality (aka ignore assign)
 */
class IgnoreAssign {
    constructor(parent) {
        this.parent = parent;
    }
    set value(value) {
    }
    get value() {
        return this.parent;
    }
}
;
/**
 * uses the known pattern from config
 * assign(target, key1, key2, key3).value = value;
 * @param target
 * @param keys
 */
function assign(target, ...accessPath) {
    if (accessPath.length < 1) {
        return new IgnoreAssign(target);
    }
    const lastPathItem = buildPath(target, ...accessPath);
    let assigner = new (class {
        set value(value) {
            lastPathItem.target[lastPathItem.key] = value;
        }
        get value() {
            return lastPathItem.target[lastPathItem.key];
        }
    })();
    return assigner;
}
exports.assign = assign;
function append(target, ...accessPath) {
    if (accessPath.length < 1) {
        return new IgnoreAssign(target);
    }
    const lastPathItem = buildPath(target, ...accessPath);
    let appender = new (class {
        set value(value) {
            if (!Array.isArray(value)) {
                value = [value];
            }
            if (!lastPathItem.target[lastPathItem.key]) {
                lastPathItem.target[lastPathItem.key] = value;
            }
            else {
                if (!Array.isArray(lastPathItem.target[lastPathItem.key])) {
                    lastPathItem.target[lastPathItem.key] = [lastPathItem.target[lastPathItem.key]];
                }
                lastPathItem.target[lastPathItem.key].push(...value);
            }
        }
    })();
    return appender;
}
exports.append = append;
/**
 * uses the known pattern from config
 * assign(target, key1, key2, key3).value = value;
 * @param target
 * @param keys
 */
function assignIf(condition, target, ...accessPath) {
    if ((!condition) || accessPath.length < 1) {
        return new IgnoreAssign(target);
    }
    return assign(target, ...accessPath);
}
exports.assignIf = assignIf;
/**
 * uses the known pattern from config
 * assign(target, key1, key2, key3).value = value;
 * @param target
 * @param keys
 */
function appendIf(condition, target, ...accessPath) {
    if ((!condition) || accessPath.length < 1) {
        return new IgnoreAssign(target);
    }
    return append(target, ...accessPath);
}
exports.appendIf = appendIf;
function resolve(target, ...accessPath) {
    let ret = null;
    accessPath = flattenAccessPath(accessPath);
    let currPtr = target;
    for (let cnt = 0; cnt < accessPath.length; cnt++) {
        let accessKeyIndex = accessPath[cnt];
        accessKeyIndex = arrayIndex(accessKeyIndex) != -1 ? arrayIndex(accessKeyIndex) : accessKeyIndex;
        currPtr = currPtr === null || currPtr === void 0 ? void 0 : currPtr[accessKeyIndex];
        if ('undefined' == typeof currPtr) {
            return null;
        }
        ret = currPtr;
    }
    return currPtr;
}
exports.resolve = resolve;
function keyVal(key) {
    let start = key.indexOf("[");
    if (start >= 0) {
        return key.substring(0, start);
    }
    else {
        return key;
    }
}
function arrayIndex(key) {
    let start = key.indexOf("[");
    let end = key.indexOf("]");
    if (start >= 0 && end > 0 && start < end) {
        return parseInt(key.substring(start + 1, end));
    }
    else {
        return -1;
    }
}
function isArrayPos(currKey, arrPos) {
    return currKey === "" && arrPos >= 0;
}
function isNoArray(arrPos) {
    return arrPos == -1;
}
function alloc(arr, length, defaultVal = {}) {
    let toAdd = [];
    toAdd.length = length;
    toAdd[length - 1] = defaultVal;
    arr.push(...toAdd);
}
function flattenAccessPath(accessPath) {
    return new Es2019Array_1.Es2019Array(...accessPath).flatMap(path => path.split("["))
        .map(path => path.indexOf("]") != -1 ? "[" + path : path)
        .filter(path => path != "");
}
/**
 * builds up a path, only done if no data is present!
 * @param target
 * @param accessPath
 * @returns the last assignable entry
 */
function buildPath(target, ...accessPath) {
    accessPath = flattenAccessPath(accessPath);
    //we now have a pattern of having the array accessors always in separate items
    let parentPtr = target;
    let parKeyArrPos = null;
    let currKey = null;
    let arrPos = -1;
    for (let cnt = 0; cnt < accessPath.length; cnt++) {
        currKey = keyVal(accessPath[cnt]);
        arrPos = arrayIndex(accessPath[cnt]);
        //it now is either key or arrPos
        if (arrPos != -1) {
            //case root(array)[5] -> root must be array and allocate 5 elements
            //case root.item[5] root.item must be array and of 5 elements
            if (!Array.isArray(parentPtr)) {
                throw Error("Associative array referenced as index array in path reference");
            }
            //we need to look ahead for proper allocation
            //not end reached
            let nextArrPos = -1;
            if (cnt < accessPath.length - 1) {
                nextArrPos = arrayIndex(accessPath[cnt + 1]);
            }
            let dataPresent = 'undefined' != typeof (parentPtr === null || parentPtr === void 0 ? void 0 : parentPtr[arrPos]);
            //no data present check here is needed, because alloc only reserves if not present
            alloc(parentPtr, arrPos + 1, nextArrPos != -1 ? [] : {});
            parKeyArrPos = arrPos;
            //we now go to the reserved element
            if (cnt == accessPath.length - 1) {
                parentPtr[arrPos] = (dataPresent) ? parentPtr[arrPos] : null;
            }
            else {
                parentPtr = parentPtr[arrPos];
            }
        }
        else {
            if (Array.isArray(parentPtr)) {
                throw Error("Index array referenced as associative array in path reference");
            }
            //again look ahead whether the next value is an array or assoc array
            let nextArrPos = -1;
            if (cnt < accessPath.length - 1) {
                nextArrPos = arrayIndex(accessPath[cnt + 1]);
            }
            parKeyArrPos = currKey;
            let dataPresent = 'undefined' != typeof (parentPtr === null || parentPtr === void 0 ? void 0 : parentPtr[currKey]);
            if (cnt == accessPath.length - 1) {
                if (!dataPresent) {
                    parentPtr[currKey] = null;
                }
            }
            else {
                if (!dataPresent) {
                    parentPtr[currKey] = nextArrPos == -1 ? {} : [];
                }
                parentPtr = parentPtr[currKey];
            }
        }
    }
    return { target: parentPtr, key: parKeyArrPos };
}
exports.buildPath = buildPath;
function deepCopy(fromAssoc) {
    return JSON.parse(JSON.stringify(fromAssoc));
}
exports.deepCopy = deepCopy;
/**
 * simple left to right merge
 *
 * @param assocArrays
 */
function simpleShallowMerge(...assocArrays) {
    return shallowMerge(true, false, ...assocArrays);
}
exports.simpleShallowMerge = simpleShallowMerge;
/**
 * Shallow merge as in config, but on raw associative arrays
 *
 * @param overwrite
 * @param withAppend
 * @param assocArrays
 */
function shallowMerge(overwrite = true, withAppend = false, ...assocArrays) {
    let target = {};
    new Es2019Array_1.Es2019Array(...assocArrays).map(arr => {
        return { arr, keys: Object.keys(arr) };
    }).forEach(({ arr, keys }) => {
        keys.forEach(key => {
            let toAssign = arr[key];
            if (!Array.isArray(toAssign) && withAppend) {
                toAssign = new Es2019Array_1.Es2019Array(...[toAssign]);
            }
            if (overwrite || !(target === null || target === void 0 ? void 0 : target[key])) {
                if (!withAppend) {
                    target[key] = arr[key];
                }
                else {
                    if ('undefined' == typeof (target === null || target === void 0 ? void 0 : target[key])) {
                        target[key] = toAssign;
                    }
                    else if (!Array.isArray(target[key])) {
                        let oldVal = target[key];
                        target[key] = new Es2019Array_1.Es2019Array(...[]);
                        target[key].push(oldVal);
                        target[key].push(...toAssign);
                    }
                    else {
                        target[key].push(...toAssign);
                    }
                }
            }
        });
    });
    return target;
}
exports.shallowMerge = shallowMerge;
//# sourceMappingURL=AssocArray.js.map