/* Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */

package org.apache.myfaces.portlet.faces.util;

import java.io.IOException;

import java.net.URL;

import java.util.Enumeration;
import java.util.jar.Manifest;

/**
 * <a href="FacesUtil.java.html"><b><i>View Source</i></b></a>
 * 
 * @author Atul Patel
 * @author Neil Griffin
 * 
 */
public class FacesVersionUtil
{

  public static final String FACES_IMPL_NAME_MOJARRA = "mojarra";
  public static final String FACES_IMPL_NAME_SUN = "sun";
  public static final String FACES_IMPL_NAME_MYFACES = "myfaces";

  private static final String JSF_IMPL_TITLE = "Implementation-Title";
  private static final String JSF_IMPL_VERSION = "Implementation-Version";

  private static String mFacesImplTitle;
  private static int mFacesImplMajorVersion = -1;
  private static int mFacesImplMinorVersion = -1;
  private static int mFacesImplPatchVersion = -1;
  private static FACES_TYPE mFacesType = FACES_TYPE.UNKNOWN;

  public static enum FACES_TYPE
  {
    MOJARRA, // Sun RI
    MYFACES, // Apache Faces
    UNKNOWN  // couldn't determine
  }

  static {
    try
    {
      Class c = Class.forName("javax.faces.webapp.FacesServlet");
      mFacesImplTitle = c.getPackage().getImplementationTitle();

      // Try and determine the FACES_TYPE based on the Title
      if (mFacesImplTitle != null)
      {
        String title = mFacesImplTitle.toLowerCase();
        if (title.indexOf(FACES_IMPL_NAME_MYFACES) >= 0)
        {
          mFacesType = FACES_TYPE.MYFACES;
        } else if (title.indexOf(FACES_IMPL_NAME_MOJARRA) >= 0 || 
                   title.indexOf(FACES_IMPL_NAME_SUN) >= 0)
        {
          mFacesType = FACES_TYPE.MOJARRA;
        }
        // Otherwise -- unknown
      }

      // Try and determine the Impl version number.
      String implVersion = c.getPackage().getImplementationVersion();

      if (implVersion != null)
      {
        String[] versions = implVersion.split("[\\-_\\.a-z]");
        try {
          mFacesImplMajorVersion = (versions.length < 1? -1: Integer.parseInt(versions[0]));
          mFacesImplMinorVersion = (versions.length < 1? -2: Integer.parseInt(versions[1]));
          // return -1 if a patch version can not be determined
          mFacesImplPatchVersion = (versions.length < 3? -1: Integer.parseInt(versions[2]));
        } 
        catch (NumberFormatException e)
        {
          // do nothing -- can't parse the version
        }
      }
    }
    catch (ClassNotFoundException e)
    {
      // Do nothing -- can't determine the impl/version
    }
  }

  public static int getFacesImplMajorVersion()
  {
    return mFacesImplMajorVersion;
  }

  public static int getFacesImplMinorVersion()
  {
    return mFacesImplMinorVersion;
  }

  public static String getFacesImplName()
  {
    return mFacesImplTitle;
  }

  public static int getFacesImplPatchVersion()
  {
    return mFacesImplPatchVersion;
  }
  
  public static FACES_TYPE getFacesType()
  {
    return mFacesType;
  }

}
