/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.safemode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerManager;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.safemode.SCMSafeModeManager;
import org.apache.hadoop.hdds.scm.safemode.SafeModeExitRule;
import org.apache.hadoop.hdds.scm.server.SCMDatanodeProtocolServer;
import org.apache.hadoop.hdds.server.events.EventQueue;
import org.apache.hadoop.hdds.server.events.TypedEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerSafeModeRule
extends SafeModeExitRule<SCMDatanodeProtocolServer.NodeRegistrationContainerReport> {
    public static final Logger LOG = LoggerFactory.getLogger(ContainerSafeModeRule.class);
    private static final String NAME = "ContainerSafeModeRule";
    private final ContainerManager containerManager;
    private final double safeModeCutoff;
    private final Set<Long> ratisContainers;
    private final Set<Long> ecContainers;
    private final Map<Long, Set<UUID>> ecContainerDNsMap;
    private final AtomicLong ratisContainerWithMinReplicas = new AtomicLong(0L);
    private final AtomicLong ecContainerWithMinReplicas = new AtomicLong(0L);
    private double ratisMaxContainer;
    private double ecMaxContainer;

    public ContainerSafeModeRule(EventQueue eventQueue, ConfigurationSource conf, ContainerManager containerManager, SCMSafeModeManager manager) {
        super(manager, NAME, eventQueue);
        this.safeModeCutoff = ContainerSafeModeRule.getSafeModeCutoff(conf);
        this.containerManager = containerManager;
        this.ratisContainers = new HashSet<Long>();
        this.ecContainers = new HashSet<Long>();
        this.ecContainerDNsMap = new ConcurrentHashMap<Long, Set<UUID>>();
        this.initializeRule();
    }

    private static double getSafeModeCutoff(ConfigurationSource conf) {
        double cutoff = conf.getDouble("hdds.scm.safemode.threshold.pct", 0.99);
        Preconditions.checkArgument((cutoff >= 0.0 && cutoff <= 1.0 ? 1 : 0) != 0, (Object)"hdds.scm.safemode.threshold.pct value should be >= 0.0 and <= 1.0");
        return cutoff;
    }

    @Override
    protected TypedEvent<SCMDatanodeProtocolServer.NodeRegistrationContainerReport> getEventType() {
        return SCMEvents.CONTAINER_REGISTRATION_REPORT;
    }

    @Override
    protected synchronized boolean validate() {
        if (this.validateBasedOnReportProcessing()) {
            return this.getCurrentContainerThreshold() >= this.safeModeCutoff && this.getCurrentECContainerThreshold() >= this.safeModeCutoff;
        }
        List<ContainerInfo> containers = this.containerManager.getContainers(HddsProtos.ReplicationType.RATIS);
        return containers.stream().filter(this::isClosed).map(ContainerInfo::containerID).noneMatch(this::isMissing);
    }

    private boolean isMissing(ContainerID id) {
        try {
            return this.containerManager.getContainerReplicas(id).isEmpty();
        }
        catch (ContainerNotFoundException ex) {
            return false;
        }
    }

    @VisibleForTesting
    public double getCurrentContainerThreshold() {
        return this.ratisMaxContainer == 0.0 ? 1.0 : this.ratisContainerWithMinReplicas.doubleValue() / this.ratisMaxContainer;
    }

    @VisibleForTesting
    public double getCurrentECContainerThreshold() {
        return this.ecMaxContainer == 0.0 ? 1.0 : this.ecContainerWithMinReplicas.doubleValue() / this.ecMaxContainer;
    }

    @Override
    protected synchronized void process(SCMDatanodeProtocolServer.NodeRegistrationContainerReport reportsProto) {
        DatanodeDetails datanodeDetails = reportsProto.getDatanodeDetails();
        UUID datanodeUUID = datanodeDetails.getUuid();
        StorageContainerDatanodeProtocolProtos.ContainerReportsProto report = (StorageContainerDatanodeProtocolProtos.ContainerReportsProto)reportsProto.getReport();
        report.getReportsList().forEach(c -> {
            long containerID = c.getContainerID();
            if (this.ratisContainers.contains(containerID)) {
                this.recordReportedContainer(containerID, Boolean.FALSE);
                this.ratisContainers.remove(containerID);
            }
            if (this.ecContainers.contains(containerID)) {
                this.putInContainerDNsMap(containerID, this.ecContainerDNsMap, datanodeUUID);
                this.recordReportedContainer(containerID, Boolean.TRUE);
            }
        });
        if (this.scmInSafeMode()) {
            SCMSafeModeManager.getLogger().info("SCM in safe mode. {} % containers [Ratis] have at least one reported replica, {} % containers [EC] have at N reported replica.", (Object)(this.getCurrentContainerThreshold() * 100.0), (Object)(this.getCurrentECContainerThreshold() * 100.0));
        }
    }

    private void recordReportedContainer(long containerID, boolean isEcContainer) {
        int minReplica;
        int uuids = 1;
        if (isEcContainer && this.ecContainerDNsMap.containsKey(containerID)) {
            uuids = this.ecContainerDNsMap.get(containerID).size();
        }
        if (uuids >= (minReplica = this.getMinReplica(containerID))) {
            if (isEcContainer) {
                this.getSafeModeMetrics().incCurrentContainersWithECDataReplicaReportedCount();
                this.ecContainerWithMinReplicas.getAndAdd(1L);
            } else {
                this.ratisContainerWithMinReplicas.getAndAdd(1L);
                this.getSafeModeMetrics().incCurrentContainersWithOneReplicaReportedCount();
            }
        }
    }

    private int getMinReplica(long pContainerID) {
        try {
            ContainerID containerID = ContainerID.valueOf((long)pContainerID);
            ContainerInfo container = this.containerManager.getContainer(containerID);
            ReplicationConfig replicationConfig = container.getReplicationConfig();
            return replicationConfig.getMinimumNodes();
        }
        catch (ContainerNotFoundException e) {
            LOG.error("containerId = {} not found.", (Object)pContainerID, (Object)e);
        }
        catch (Exception e) {
            LOG.error("containerId = {} not found.", (Object)pContainerID, (Object)e);
        }
        return 1;
    }

    private void putInContainerDNsMap(long containerID, Map<Long, Set<UUID>> containerDNsMap, UUID datanodeUUID) {
        containerDNsMap.computeIfAbsent(containerID, key -> Sets.newHashSet());
        containerDNsMap.get(containerID).add(datanodeUUID);
    }

    @Override
    protected synchronized void cleanup() {
        this.ratisContainers.clear();
        this.ecContainers.clear();
        this.ecContainerDNsMap.clear();
    }

    @Override
    public String getStatusText() {
        String status = String.format("%1.2f%% of [Ratis] Containers(%s / %s) with at least one reported replica (=%1.2f) >= safeModeCutoff (=%1.2f);", this.getCurrentContainerThreshold() * 100.0, this.ratisContainerWithMinReplicas, (long)this.ratisMaxContainer, this.getCurrentContainerThreshold(), this.safeModeCutoff);
        Set sampleRatisContainers = this.ratisContainers.stream().limit(5L).collect(Collectors.toSet());
        if (!sampleRatisContainers.isEmpty()) {
            String sampleContainerText = "Sample Ratis Containers not satisfying the criteria : " + sampleRatisContainers + ";";
            status = status.concat("\n").concat(sampleContainerText);
        }
        String ecStatus = String.format("%1.2f%% of [EC] Containers(%s / %s) with at least N reported replica (=%1.2f) >= safeModeCutoff (=%1.2f);", this.getCurrentECContainerThreshold() * 100.0, this.ecContainerWithMinReplicas, (long)this.ecMaxContainer, this.getCurrentECContainerThreshold(), this.safeModeCutoff);
        status = status.concat("\n").concat(ecStatus);
        Set sampleEcContainers = this.ecContainerDNsMap.entrySet().stream().filter(entry -> {
            Long containerId = (Long)entry.getKey();
            int minReplica = this.getMinReplica(containerId);
            Set allReplicas = (Set)entry.getValue();
            return allReplicas.size() < minReplica;
        }).map(Map.Entry::getKey).limit(5L).collect(Collectors.toSet());
        if (!sampleEcContainers.isEmpty()) {
            String sampleECContainerText = "Sample EC Containers not satisfying the criteria : " + sampleEcContainers + ";";
            status = status.concat("\n").concat(sampleECContainerText);
        }
        return status;
    }

    @Override
    public synchronized void refresh(boolean forceRefresh) {
        if (forceRefresh || !this.validate()) {
            this.initializeRule();
        }
    }

    private boolean isClosed(ContainerInfo container) {
        HddsProtos.LifeCycleState state = container.getState();
        return state == HddsProtos.LifeCycleState.QUASI_CLOSED || state == HddsProtos.LifeCycleState.CLOSED;
    }

    private void initializeRule() {
        List<ContainerInfo> containers = this.containerManager.getContainers();
        this.ratisContainers.clear();
        this.ecContainers.clear();
        containers.forEach(container -> {
            HddsProtos.ReplicationType replicationType = container.getReplicationType();
            if (this.isClosed((ContainerInfo)container) && container.getNumberOfKeys() > 0L) {
                if (replicationType.equals((Object)HddsProtos.ReplicationType.RATIS)) {
                    this.ratisContainers.add(container.getContainerID());
                }
                if (replicationType.equals((Object)HddsProtos.ReplicationType.EC)) {
                    this.ecContainers.add(container.getContainerID());
                }
            }
        });
        this.ratisMaxContainer = this.ratisContainers.size();
        this.ecMaxContainer = this.ecContainers.size();
        long ratisCutOff = (long)Math.ceil(this.ratisMaxContainer * this.safeModeCutoff);
        long ecCutOff = (long)Math.ceil(this.ecMaxContainer * this.safeModeCutoff);
        this.getSafeModeMetrics().setNumContainerWithOneReplicaReportedThreshold(ratisCutOff);
        this.getSafeModeMetrics().setNumContainerWithECDataReplicaReportedThreshold(ecCutOff);
        LOG.info("Refreshed Containers with one replica threshold count {}, with ec n replica threshold count {}.", (Object)ratisCutOff, (Object)ecCutOff);
    }
}

