/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue.helpers;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.utils.MetadataKeyFilters;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.helpers.ContainerUtils;
import org.apache.hadoop.ozone.container.common.interfaces.BlockIterator;
import org.apache.hadoop.ozone.container.common.interfaces.DBHandle;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.utils.ContainerInspectorUtil;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.BlockUtils;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerLocationUtil;
import org.apache.hadoop.ozone.container.metadata.AbstractDatanodeStore;
import org.apache.hadoop.ozone.container.metadata.DatanodeStore;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreSchemaOneImpl;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreSchemaTwoImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KeyValueContainerUtil {
    private static final Logger LOG = LoggerFactory.getLogger(KeyValueContainerUtil.class);

    private KeyValueContainerUtil() {
    }

    public static void createContainerMetaData(File containerMetaDataPath, File chunksPath, File dbFile, String schemaVersion, ConfigurationSource conf) throws IOException {
        AbstractDatanodeStore store;
        Preconditions.checkNotNull((Object)containerMetaDataPath);
        Preconditions.checkNotNull((Object)conf);
        if (!containerMetaDataPath.mkdirs()) {
            LOG.error("Unable to create directory for metadata storage. Path: {}", (Object)containerMetaDataPath);
            throw new IOException("Unable to create directory for metadata storage. Path: " + containerMetaDataPath);
        }
        if (!chunksPath.mkdirs()) {
            LOG.error("Unable to create chunks directory Container {}", (Object)chunksPath);
            FileUtils.deleteDirectory((File)containerMetaDataPath);
            FileUtils.deleteDirectory((File)containerMetaDataPath.getParentFile());
            throw new IOException("Unable to create directory for data storage. Path: " + chunksPath);
        }
        if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "1")) {
            store = new DatanodeStoreSchemaOneImpl(conf, dbFile.getAbsolutePath(), false);
        } else if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "2")) {
            store = new DatanodeStoreSchemaTwoImpl(conf, dbFile.getAbsolutePath(), false);
        } else {
            if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "3")) {
                return;
            }
            throw new IllegalArgumentException("Unrecognized schema version for container: " + schemaVersion);
        }
        BlockUtils.addDB(store, dbFile.getAbsolutePath(), conf, schemaVersion);
    }

    public static void removeContainer(KeyValueContainerData containerData, ConfigurationSource conf) throws IOException {
        Preconditions.checkNotNull((Object)containerData);
        KeyValueContainerUtil.removeContainerDB(containerData, conf);
        KeyValueContainerUtil.moveToDeletedContainerDir(containerData, containerData.getVolume());
    }

    public static void removeContainerDB(KeyValueContainerData containerData, ConfigurationSource conf) throws IOException {
        if (containerData.hasSchema("3")) {
            try {
                BlockUtils.removeContainerFromDB(containerData, conf);
            }
            catch (IOException ex) {
                LOG.error("DB failure, unable to remove container. Disk need to be replaced.", (Throwable)ex);
                throw ex;
            }
        } else {
            BlockUtils.removeDB(containerData, conf);
        }
    }

    public static boolean noBlocksInContainer(DatanodeStore store, KeyValueContainerData containerData, boolean bCheckChunksFilePath) throws IOException {
        Preconditions.checkNotNull((Object)store);
        Preconditions.checkNotNull((Object)containerData);
        if (containerData.isOpen()) {
            return false;
        }
        try (BlockIterator<BlockData> blockIterator = store.getBlockIterator(containerData.getContainerID());){
            if (blockIterator.hasNext()) {
                boolean bl = false;
                return bl;
            }
        }
        if (bCheckChunksFilePath) {
            File chunksPath = new File(containerData.getChunksPath());
            Preconditions.checkArgument((boolean)chunksPath.isDirectory());
            try (DirectoryStream<Path> dir = Files.newDirectoryStream(chunksPath.toPath());){
                boolean bl = !dir.iterator().hasNext();
                return bl;
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void parseKVContainerData(KeyValueContainerData kvContainerData, ConfigurationSource config) throws IOException {
        File dbFile;
        long containerID = kvContainerData.getContainerID();
        ContainerUtils.verifyChecksum(kvContainerData, config);
        if (kvContainerData.getSchemaVersion() == null) {
            kvContainerData.setSchemaVersion("1");
        }
        if (!(dbFile = KeyValueContainerLocationUtil.getContainerDBFile(kvContainerData)).exists()) {
            LOG.error("Container DB file is missing for ContainerID {}. Skipping loading of this container.", (Object)containerID);
            throw new IOException("Container DB file is missing for containerID " + containerID);
        }
        kvContainerData.setDbFile(dbFile);
        DatanodeConfiguration dnConf = (DatanodeConfiguration)((Object)config.getObject(DatanodeConfiguration.class));
        boolean bCheckChunksFilePath = dnConf.getCheckEmptyContainerDir();
        if (kvContainerData.hasSchema("3")) {
            try (DBHandle db = BlockUtils.getDB(kvContainerData, config);){
                KeyValueContainerUtil.populateContainerMetadata(kvContainerData, db.getStore(), bCheckChunksFilePath);
            }
            return;
        }
        Closeable cachedDB = null;
        DatanodeStore store = null;
        try {
            try {
                boolean readOnly = ContainerInspectorUtil.isReadOnly(ContainerProtos.ContainerType.KeyValueContainer);
                store = BlockUtils.getUncachedDatanodeStore(kvContainerData, config, readOnly);
            }
            catch (IOException e) {
                cachedDB = BlockUtils.getDB(kvContainerData, config);
                store = ((DBHandle)cachedDB).getStore();
                LOG.warn("Attempt to get an uncached RocksDB handle failed and an instance was retrieved from the cache. This should only happen in tests");
            }
            KeyValueContainerUtil.populateContainerMetadata(kvContainerData, store, bCheckChunksFilePath);
        }
        finally {
            if (cachedDB != null) {
                cachedDB.close();
            } else if (store != null) {
                try {
                    store.stop();
                }
                catch (IOException e) {
                    throw e;
                }
                catch (Exception e) {
                    throw new RuntimeException("Unexpected exception closing the RocksDB when loading containers", e);
                }
            }
        }
    }

    private static void populateContainerMetadata(KeyValueContainerData kvContainerData, DatanodeStore store, boolean bCheckChunksFilePath) throws IOException {
        File chunksDir;
        Long blockCount;
        Long bytesUsed;
        Long bcsId;
        boolean isBlockMetadataSet = false;
        Table<String, Long> metadataTable = store.getMetadataTable();
        Long pendingDeleteBlockCount = (Long)metadataTable.get((Object)kvContainerData.getPendingDeleteBlockCountKey());
        if (pendingDeleteBlockCount != null) {
            kvContainerData.incrPendingDeletionBlocks(pendingDeleteBlockCount);
        } else {
            MetadataKeyFilters.KeyPrefixFilter filter = kvContainerData.getDeletingBlockKeyFilter();
            int numPendingDeletionBlocks = store.getBlockDataTable().getSequentialRangeKVs((Object)kvContainerData.startKeyEmpty(), Integer.MAX_VALUE, (Object)kvContainerData.containerPrefix(), new MetadataKeyFilters.MetadataKeyFilter[]{filter}).size();
            kvContainerData.incrPendingDeletionBlocks(numPendingDeletionBlocks);
        }
        Long delTxnId = (Long)metadataTable.get((Object)kvContainerData.getLatestDeleteTxnKey());
        if (delTxnId != null) {
            kvContainerData.updateDeleteTransactionId(delTxnId);
        }
        if ((bcsId = (Long)metadataTable.get((Object)kvContainerData.getBcsIdKey())) != null) {
            kvContainerData.updateBlockCommitSequenceId(bcsId);
        }
        if ((bytesUsed = (Long)metadataTable.get((Object)kvContainerData.getBytesUsedKey())) != null) {
            isBlockMetadataSet = true;
            kvContainerData.setBytesUsed(bytesUsed);
        }
        if ((blockCount = (Long)metadataTable.get((Object)kvContainerData.getBlockCountKey())) != null) {
            isBlockMetadataSet = true;
            kvContainerData.setBlockCount(blockCount);
        }
        if (!isBlockMetadataSet) {
            KeyValueContainerUtil.initializeUsedBytesAndBlockCount(store, kvContainerData);
        }
        if (!(chunksDir = new File(kvContainerData.getChunksPath())).exists()) {
            Files.createDirectories(chunksDir.toPath(), new FileAttribute[0]);
        }
        if (KeyValueContainerUtil.noBlocksInContainer(store, kvContainerData, bCheckChunksFilePath)) {
            kvContainerData.markAsEmpty();
        }
        ContainerInspectorUtil.process(kvContainerData, store);
        KeyValueContainerUtil.populateContainerFinalizeBlock(kvContainerData, store);
    }

    private static void populateContainerFinalizeBlock(KeyValueContainerData kvContainerData, DatanodeStore store) throws IOException {
        if (store.getFinalizeBlocksTable() != null) {
            try (BlockIterator<Long> iter = store.getFinalizeBlockIterator(kvContainerData.getContainerID(), kvContainerData.getUnprefixedKeyFilter());){
                while (iter.hasNext()) {
                    kvContainerData.addToFinalizedBlockSet(iter.nextBlock());
                }
            }
        }
    }

    private static void initializeUsedBytesAndBlockCount(DatanodeStore store, KeyValueContainerData kvData) throws IOException {
        String errorMessage = "Failed to parse block data for Container " + kvData.getContainerID();
        long blockCount = 0L;
        long usedBytes = 0L;
        try (BlockIterator<BlockData> blockIter = store.getBlockIterator(kvData.getContainerID(), kvData.getUnprefixedKeyFilter());){
            while (blockIter.hasNext()) {
                ++blockCount;
                try {
                    usedBytes += KeyValueContainerUtil.getBlockLength(blockIter.nextBlock());
                }
                catch (Exception ex) {
                    LOG.error(errorMessage, (Throwable)ex);
                }
            }
        }
        blockIter = store.getBlockIterator(kvData.getContainerID(), kvData.getDeletingBlockKeyFilter());
        var8_6 = null;
        try {
            while (blockIter.hasNext()) {
                ++blockCount;
                try {
                    usedBytes += KeyValueContainerUtil.getBlockLength(blockIter.nextBlock());
                }
                catch (IOException ex) {
                    LOG.error(errorMessage);
                }
            }
        }
        catch (Throwable throwable) {
            var8_6 = throwable;
            throw throwable;
        }
        finally {
            if (blockIter != null) {
                if (var8_6 != null) {
                    try {
                        blockIter.close();
                    }
                    catch (Throwable throwable) {
                        var8_6.addSuppressed(throwable);
                    }
                } else {
                    blockIter.close();
                }
            }
        }
        kvData.setBytesUsed(usedBytes);
        kvData.setBlockCount(blockCount);
    }

    public static long getBlockLength(BlockData block) throws IOException {
        return block.getChunks().stream().mapToLong(ContainerProtos.ChunkInfo::getLen).sum();
    }

    public static boolean isSameSchemaVersion(String schema, String other) {
        String effective1 = schema != null ? schema : "1";
        String effective2 = other != null ? other : "1";
        return effective1.equals(effective2);
    }

    public static void moveToDeletedContainerDir(KeyValueContainerData keyValueContainerData, HddsVolume hddsVolume) throws IOException {
        String containerPath = keyValueContainerData.getContainerPath();
        File container = new File(containerPath);
        Path destinationDirPath = KeyValueContainerUtil.getTmpDirectoryPath(keyValueContainerData, hddsVolume);
        File destinationDirFile = destinationDirPath.toFile();
        if (destinationDirFile.exists()) {
            FileUtils.deleteDirectory((File)destinationDirFile);
        }
        Files.move(container.toPath(), destinationDirPath, new CopyOption[0]);
        LOG.debug("Container {} has been successfully moved under {}", (Object)container.getName(), (Object)hddsVolume.getDeletedContainerDir());
    }

    public static Path getTmpDirectoryPath(KeyValueContainerData keyValueContainerData, HddsVolume hddsVolume) {
        String containerPath = keyValueContainerData.getContainerPath();
        File container = new File(containerPath);
        String containerDirName = container.getName();
        Path destinationDirPath = hddsVolume.getDeletedContainerDir().toPath().resolve(Paths.get(containerDirName, new String[0]));
        return destinationDirPath;
    }
}

