package org.apache.stratum.scheduler;

/*
 * Copyright 2001-2005 The Apache Software Foundation or its licensors,
 * as applicable.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.util.ArrayList;
import java.util.List;

/**
 * This bean represents all of the configuration information needed to load the Quartz scheduler.
 *
 * @author <a href="mailto:john@thorhauer.com">John Thorhauer</a>
 * @version $Id: SchedulerConfig.java 264723 2005-08-30 07:44:51Z henning $
 */
public class SchedulerConfig
{
    /** Scheduler reference name */
    private String instanceName;

    /** List of job configurations */
    private List jobConfigs;

    /** List of triggers */
    private List triggerConfigs;

    /** The jobstore for the Scheduler */
    private JobStoreConfig jobStoreConfig;

    /** The threadpool for the Scheduler */
    private ThreadPoolConfig threadPoolConfig;

    /**
     * Default contructor
     */
    public SchedulerConfig()
    {
        jobConfigs = new ArrayList();
        triggerConfigs = new ArrayList();
    }

    /**
     * Set the reference name given to the scheduler that is loaded into Quartz
     *
     * @param s instanceName
     */
    public void setInstanceName(String s)
    {
        this.instanceName = s;
    }

    /**
     * Return the reference name of the scheduler
     *
     * @return instance name
     */
    public String getInstanceName()
    {
        return this.instanceName;
    }

    /**
     * Add a Job Configuraton to Scheduler Config
     *
     * @param jconf Job configuration
     */
    public void addJobConfig(JobConfig jconf)
    {
        jobConfigs.add(jconf);
    }

    /**
     * Return a list of Job Configurations
     *
     * @return List of job configs
     */
    public List getJobConfigs()
    {
        return this.jobConfigs;
    }

    /**
     * Add a trigger to Scheduler Config
     *
     * @param trigger
     */
    public void addTriggerConfig(TriggerConfig trigger)
    {
        triggerConfigs.add(trigger);
    }

    /**
     * Return a list of triggers
     *
     * @return List of triggers
     */
    public List getTriggerConfigs()
    {
        return this.triggerConfigs;
    }

    /**
     * Set the threadPool for the scheduler to use
     *
     * @param thdPool threadPool
     */
    public void setThreadPoolConfig(ThreadPoolConfig thdPool)
    {
        this.threadPoolConfig = thdPool;
    }

    /**
     * Return the the ThreadPool object
     *
     * @return threadPool
     */
    public ThreadPoolConfig getThreadPoolConfig()
    {
        return this.threadPoolConfig;
    }

    /**
     * Set the jobstore for the scheduler to use
     *
     * @param jobStr jobstore
     */
    public void setJobStoreConfig(JobStoreConfig jobStr)
    {
        this.jobStoreConfig = jobStr;
    }

    /**
     * Return the the JobStore object
     *
     * @return jobStore
     */
    public JobStoreConfig getJobStoreConfig()
    {
        return this.jobStoreConfig;
    }
}
