/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.linear;

import org.apache.commons.math3.exception.DimensionMismatchException;
import org.apache.commons.math3.linear.Array2DRowRealMatrix;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.DecompositionSolver;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.NonSquareMatrixException;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.apache.commons.math3.linear.SingularMatrixException;
import org.apache.commons.math3.util.FastMath;

public class LUDecomposition {
    private static final double DEFAULT_TOO_SMALL = 1.0E-11;
    private final double[][] lu;
    private final int[] pivot;
    private boolean even;
    private boolean singular;
    private RealMatrix cachedL;
    private RealMatrix cachedU;
    private RealMatrix cachedP;

    public LUDecomposition(RealMatrix matrix) {
        this(matrix, 1.0E-11);
    }

    public LUDecomposition(RealMatrix matrix, double singularityThreshold) {
        if (!matrix.isSquare()) {
            throw new NonSquareMatrixException(matrix.getRowDimension(), matrix.getColumnDimension());
        }
        int m = matrix.getColumnDimension();
        this.lu = matrix.getData();
        this.pivot = new int[m];
        this.cachedL = null;
        this.cachedU = null;
        this.cachedP = null;
        int row = 0;
        while (row < m) {
            this.pivot[row] = row;
            ++row;
        }
        this.even = true;
        this.singular = false;
        int col = 0;
        while (col < m) {
            int i;
            int row2 = 0;
            while (row2 < col) {
                double[] luRow = this.lu[row2];
                double sum = luRow[col];
                int i2 = 0;
                while (i2 < row2) {
                    sum -= luRow[i2] * this.lu[i2][col];
                    ++i2;
                }
                luRow[col] = sum;
                ++row2;
            }
            int max = col;
            double largest = Double.NEGATIVE_INFINITY;
            int row3 = col;
            while (row3 < m) {
                double[] luRow = this.lu[row3];
                double sum = luRow[col];
                i = 0;
                while (i < col) {
                    sum -= luRow[i] * this.lu[i][col];
                    ++i;
                }
                luRow[col] = sum;
                if (FastMath.abs(sum) > largest) {
                    largest = FastMath.abs(sum);
                    max = row3;
                }
                ++row3;
            }
            if (FastMath.abs(this.lu[max][col]) < singularityThreshold) {
                this.singular = true;
                return;
            }
            if (max != col) {
                double tmp = 0.0;
                double[] luMax = this.lu[max];
                double[] luCol = this.lu[col];
                i = 0;
                while (i < m) {
                    tmp = luMax[i];
                    luMax[i] = luCol[i];
                    luCol[i] = tmp;
                    ++i;
                }
                int temp = this.pivot[max];
                this.pivot[max] = this.pivot[col];
                this.pivot[col] = temp;
                this.even = !this.even;
            }
            double luDiag = this.lu[col][col];
            int row4 = col + 1;
            while (row4 < m) {
                double[] dArray = this.lu[row4];
                int n = col;
                dArray[n] = dArray[n] / luDiag;
                ++row4;
            }
            ++col;
        }
    }

    public RealMatrix getL() {
        if (this.cachedL == null && !this.singular) {
            int m = this.pivot.length;
            this.cachedL = MatrixUtils.createRealMatrix(m, m);
            int i = 0;
            while (i < m) {
                double[] luI = this.lu[i];
                int j = 0;
                while (j < i) {
                    this.cachedL.setEntry(i, j, luI[j]);
                    ++j;
                }
                this.cachedL.setEntry(i, i, 1.0);
                ++i;
            }
        }
        return this.cachedL;
    }

    public RealMatrix getU() {
        if (this.cachedU == null && !this.singular) {
            int m = this.pivot.length;
            this.cachedU = MatrixUtils.createRealMatrix(m, m);
            int i = 0;
            while (i < m) {
                double[] luI = this.lu[i];
                int j = i;
                while (j < m) {
                    this.cachedU.setEntry(i, j, luI[j]);
                    ++j;
                }
                ++i;
            }
        }
        return this.cachedU;
    }

    public RealMatrix getP() {
        if (this.cachedP == null && !this.singular) {
            int m = this.pivot.length;
            this.cachedP = MatrixUtils.createRealMatrix(m, m);
            int i = 0;
            while (i < m) {
                this.cachedP.setEntry(i, this.pivot[i], 1.0);
                ++i;
            }
        }
        return this.cachedP;
    }

    public int[] getPivot() {
        return (int[])this.pivot.clone();
    }

    public double getDeterminant() {
        if (this.singular) {
            return 0.0;
        }
        int m = this.pivot.length;
        double determinant = this.even ? 1 : -1;
        int i = 0;
        while (i < m) {
            determinant *= this.lu[i][i];
            ++i;
        }
        return determinant;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.lu, this.pivot, this.singular);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[][] lu;
        private final int[] pivot;
        private final boolean singular;

        private Solver(double[][] lu, int[] pivot, boolean singular) {
            this.lu = lu;
            this.pivot = pivot;
            this.singular = singular;
        }

        @Override
        public boolean isNonSingular() {
            return !this.singular;
        }

        @Override
        public RealVector solve(RealVector b) {
            int i;
            double bpCol;
            int m = this.pivot.length;
            if (b.getDimension() != m) {
                throw new DimensionMismatchException(b.getDimension(), m);
            }
            if (this.singular) {
                throw new SingularMatrixException();
            }
            double[] bp = new double[m];
            int row = 0;
            while (row < m) {
                bp[row] = b.getEntry(this.pivot[row]);
                ++row;
            }
            int col = 0;
            while (col < m) {
                bpCol = bp[col];
                i = col + 1;
                while (i < m) {
                    int n = i;
                    bp[n] = bp[n] - bpCol * this.lu[i][col];
                    ++i;
                }
                ++col;
            }
            col = m - 1;
            while (col >= 0) {
                int n = col;
                bp[n] = bp[n] / this.lu[col][col];
                bpCol = bp[col];
                i = 0;
                while (i < col) {
                    int n2 = i;
                    bp[n2] = bp[n2] - bpCol * this.lu[i][col];
                    ++i;
                }
                --col;
            }
            return new ArrayRealVector(bp, false);
        }

        @Override
        public RealMatrix solve(RealMatrix b) {
            double[] bpCol;
            int m = this.pivot.length;
            if (b.getRowDimension() != m) {
                throw new DimensionMismatchException(b.getRowDimension(), m);
            }
            if (this.singular) {
                throw new SingularMatrixException();
            }
            int nColB = b.getColumnDimension();
            double[][] bp = new double[m][nColB];
            int row = 0;
            while (row < m) {
                double[] bpRow = bp[row];
                int pRow = this.pivot[row];
                int col = 0;
                while (col < nColB) {
                    bpRow[col] = b.getEntry(pRow, col);
                    ++col;
                }
                ++row;
            }
            int col = 0;
            while (col < m) {
                bpCol = bp[col];
                int i = col + 1;
                while (i < m) {
                    double[] bpI = bp[i];
                    double luICol = this.lu[i][col];
                    int j = 0;
                    while (j < nColB) {
                        int n = j;
                        bpI[n] = bpI[n] - bpCol[j] * luICol;
                        ++j;
                    }
                    ++i;
                }
                ++col;
            }
            col = m - 1;
            while (col >= 0) {
                bpCol = bp[col];
                double luDiag = this.lu[col][col];
                int j = 0;
                while (j < nColB) {
                    int n = j++;
                    bpCol[n] = bpCol[n] / luDiag;
                }
                int i = 0;
                while (i < col) {
                    double[] bpI = bp[i];
                    double luICol = this.lu[i][col];
                    int j2 = 0;
                    while (j2 < nColB) {
                        int n = j2;
                        bpI[n] = bpI[n] - bpCol[j2] * luICol;
                        ++j2;
                    }
                    ++i;
                }
                --col;
            }
            return new Array2DRowRealMatrix(bp, false);
        }

        @Override
        public RealMatrix getInverse() {
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.pivot.length));
        }
    }
}

