/*******************************************************************************
 * Copyright (c) 2011, 2012 CEA LIST.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Guyomar (Mia-Software) - Bug 340738 - Utility method to create a coherent tableInstance
 *    Nicolas Guyomar (Mia-Software) - Bug 340940 - To be able to view facet attributes and facet references in a table
 *    Nicolas Guyomar (Mia-Software) - Bug 340681 - Facet column implementation
 *    Vincent Lorenzo (CEA-LIST) - Bug 341328 - We need to be able to specify which column have to be hidden/visible using the customization mechanism
 *    Nicolas Guyomar (Mia-Software) - Bug 344921 - Undo/Redo just after the creation of the table
 *    Gregoire Dupe (Mia-Software) - Bug 366804 - [Restructuring] Table widget upgrade
 *    Gregoire Dupe (Mia-Software) - Bug 369987 - [Restructuring][Table] Switch to the new customization and facet framework
 *    Gregoire Dupe (Mia-Software) - Bug 364325 - [Restructuring] The user must be able to navigate into a model using the Facet.
 *    Gregoire Dupe (Mia-Software) - Bug 373078 - API Cleaning
 *    Nicolas Bros (Mia-Software) - Bug 378475 - unit test failures after table refactoring
 *    Nicolas Bros (Mia-Software) - Bug 378649 - [Table] Errors with non-applicable features
 *    Olivier Remaud (Soft-Maint) - Bug 378499 - optimizing table opening
 *    Gregoire Dupe (Mia-Software) - Bug 388422 - [Table] Queries for InstanciationMethod needs to have 2 parameters
 *    Vincent Lorenzo (CEA-LIST) - Bug 391412 - [Table] The allowed contents is done set during the table creation
 *******************************************************************************/
package org.eclipse.emf.facet.widgets.table.ui.internal;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.ETypedElement;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.CustomFactory;
import org.eclipse.emf.facet.custom.metamodel.v0_2_0.custom.Customization;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.Facet;
import org.eclipse.emf.facet.efacet.metamodel.v0_2_0.efacet.FacetSet;
import org.eclipse.emf.facet.util.core.Logger;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.FeatureColumn;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.NavigationTable;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.PrimitiveTypeQueryRow;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Row;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.Table;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.table.TableFactory;
import org.eclipse.emf.facet.widgets.table.metamodel.v0_2_0.tableconfiguration.TableConfiguration;

/**
 * @since 0.3
 */
public final class CreateTableUtils {

	private CreateTableUtils() {
		// Prevent instantiation
	}

	private static void applyDefaultSettings(
			final TableConfiguration tableConfig, final Table table) {
		if (tableConfig != null) {
			applyDefaultFacets(tableConfig, table);
			applyDefaultCustom(tableConfig, table);
			applyDefaultLocalCustom(tableConfig, table);
			table.setCanBePresentedInTheTable(tableConfig.getCanBePresentedInTheTable());
			// Default columns (provided by the table configuration) are created
			// by
			// org.eclipse.emf.facet.widgets.table.ui.internal.CreateTableUtils.createColumns(Table)
		}
	}

	private static void applyDefaultLocalCustom(
			final TableConfiguration tableConfig, final Table table) {
		for (Customization custom : tableConfig.getDefaultLocalCustomizations()) {
			if (!table.getLocalCustomizations().contains(custom)) {
				table.getLocalCustomizations().add(custom);
			}
		}
	}

	private static void applyDefaultCustom(
			final TableConfiguration tableConfig, final Table table) {
		for (Customization custom : tableConfig.getDefaultCustomizations()) {
			if (!table.getCustomizations().contains(custom)) {
				table.getCustomizations().add(custom);
			}
		}
	}

	private static void applyDefaultFacets(
			final TableConfiguration tableConfig, final Table table) {
		for (Facet facet : tableConfig.getDefaultFacets()) {
			try {
				// TODO Update the meta-model to use "unique" EList.
				if (!table.getFacetSets().contains(facet.eContainer())) {
					table.getFacetSets().add((FacetSet) facet.eContainer());
				}
			} catch (Exception e) {
				// TODO should be externalized
				Logger.logError(
						"The referenced facet:" + facet + " could not be loaded", //$NON-NLS-1$ //$NON-NLS-2$
						Activator.getDefault());
			}
		}
	}

	public static void createColumns(final Table table) {
		if (table instanceof NavigationTable) {
			table.getColumns().add(TableFactory.eINSTANCE.createSourceColumn());
			for (Row row : table.getRows()) {
				if (row instanceof PrimitiveTypeQueryRow) {
					table.getColumns().add(
							TableFactory.eINSTANCE.createValueColumn());
					break;
				}
			}
		}
		final List<ETypedElement> features = new ArrayList<ETypedElement>();
		if (table.getTableConfiguration() != null) {
			features.addAll(table.getTableConfiguration().getDefaultColumns());
		}
		for (EStructuralFeature feature : InternalTableUtils
				.getAllStructuralFeatures(table)) {
			if (!features.contains(feature)) {
				features.add(feature);
			}
		}
		for (final ETypedElement typedElt : features) {
			final FeatureColumn fColumn = TableFactory.eINSTANCE
					.createFeatureColumn();
			fColumn.setFeature(typedElt);
			table.getColumns().add(fColumn);
		}
	}

	public static Table createTableInstance(
			final List<? extends EObject> elements, final String description,
			final TableConfiguration tableConfig, final EObject context,
			final Object parameter) {
		final Table table = TableFactory.eINSTANCE.createTable();
		table.setTableConfiguration(tableConfig);
		table.setContext(context);
		table.setParameter(parameter);
		table.setDescription(description);
		final Set<EObject> added = new HashSet<EObject>();
		for (EObject eObject : elements) {
			if (!added.contains(eObject)) {
				final Row row = TableFactory.eINSTANCE.createRow();
				row.setElement(eObject);
				table.getRows().add(row);
				added.add(eObject);
			}
		}
		applyDefaultSettings(tableConfig, table);
		createColumns(table);
		createLocalCustoms(table);
		return table;
	}

	private static void createLocalCustoms(final Table table) {
		// we create the localCustomization if they don't exists
		// we store the nsURI which have been already added in the local
		// customization
		final Set<EPackage> alreadyDone = new HashSet<EPackage>();
		final Set<EStructuralFeature> allFeatures = InternalTableUtils
				.getAllStructuralFeatures(table);
		for (EStructuralFeature structuralFeature : allFeatures) {
			final EObject sfContainerL1 = structuralFeature.eContainer();
			EObject sfContainerL2 = null;
			if (sfContainerL1 != null) {
				sfContainerL2 = sfContainerL1.eContainer();
			}
			if (sfContainerL2 instanceof EPackage) {
				final EPackage ePackage = (EPackage) sfContainerL2;
				if (!alreadyDone.contains(ePackage)
						&& CustomizationUtils
								.findCustomizationExtendingEPackage(
										table.getLocalCustomizations(),
										ePackage) == null) {
					// we create this localCustomization
					final Customization customizaion = CustomFactory.eINSTANCE
							.createCustomization();
					table.getLocalCustomizations().add(customizaion);
					alreadyDone.add(ePackage);
				}
			}
		}
		// we create localCustomization for FacetColumn
		for (FacetSet container : table.getFacetSets()) {
			if (!alreadyDone.contains(container)
					&& CustomizationUtils.findCustomizationExtendingEPackage(
							table.getLocalCustomizations(), container) == null) {
				// we create this localCustomization
				final Customization metamodelView = CustomFactory.eINSTANCE
						.createCustomization();
				table.getLocalCustomizations().add(metamodelView);
				alreadyDone.add(container);

			}
		}
		// we add all the local customization to the customization, at the
		// beginning of the list
		table.getCustomizations().addAll(0, table.getLocalCustomizations());
	}
}
