/*******************************************************************************
 * Copyright (c) 2009, 2010 Fabian Steeg. All rights reserved. This program and
 * the accompanying materials are made available under the terms of the Eclipse
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * <p/>
 * Contributors: Fabian Steeg - initial API and implementation; see bug 277380
 *******************************************************************************/

package org.eclipse.gef4.zest.internal.dot;

import java.io.File;
import java.util.Scanner;

import org.eclipse.draw2d.SWTGraphics;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef4.zest.core.widgets.Graph;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.eclipse.swt.graphics.Point;

/**
 * Transformation of Zest Graph instances to DOT strings or files, or via DOT to
 * an image generated by calling the 'dot' executable.
 * 
 * @author Fabian Steeg (fsteeg)
 */
public final class DotExport {
	public static final String DOT_BIN_DIR_KEY = "org.eclipse.gef4.zest.dot.bin.dir"; //$NON-NLS-1$
	private static final String DEFAULT_OUTPUT_FORMAT = "pdf"; //$NON-NLS-1$
	private String dotString;
	private String graphName = "Unnamed" + System.currentTimeMillis();
	private Graph graph;

	/**
	 * @param graph
	 *            The Zest graph to export to DOT
	 */
	public DotExport(final Graph graph) {
		this.dotString = graphToDot(graph);
		this.graphName = graph.getClass().getSimpleName();
		this.graph = graph;
	}

	/**
	 * @param dotString
	 *            The DOT graph to export to an image
	 */
	public DotExport(String dotString) {
		this.dotString = dotString;
	}

	/**
	 * Export a Zest Graph to a DOT string.
	 * 
	 * @return The DOT representation of the given Zest graph
	 */
	public String toDotString() {
		return dotString;
	}

	/**
	 * Export a Zest Graph to a DOT file.
	 * 
	 * @param destination
	 *            The file to store the DOT export
	 * @return The given file
	 */
	public File toDotFile(final File destination) {
		DotFileUtils.write(dotString, destination);
		return destination;
	}

	/**
	 * @param dotBinDir
	 *            The directory containing the local Graphviz 'dot' binary, e.g.
	 *            "/usr/bin"
	 * @param resultFile
	 *            The output image file name, e.g. "output.pdf" (can be null,
	 *            will result in export to temp result file with unpecified
	 *            default format)
	 * @return The image file generated by rendering the output of
	 *         {@link #toDotString()} with Graphviz, using the format of the
	 *         specified output file
	 */
	public File toImage(String dotBinDir, String resultFile) {
		return toImage(dotBinDir, resultFile == null ? DEFAULT_OUTPUT_FORMAT
				: resultFile.substring(resultFile.lastIndexOf('.') + 1),
				resultFile);
	}

	/**
	 * Experimental image export for Zest graphs in pure SWT.
	 * 
	 * @param resultFile
	 *            The location of the result image file
	 * @return The generated image file
	 */
	public File toImage(String resultFile) {
		Rectangle bounds = graph.getContents().getBounds();
		Point size = new Point(graph.getContents().getSize().width, graph
				.getContents().getSize().height);
		org.eclipse.draw2d.geometry.Point viewLocation = graph.getViewport()
				.getViewLocation();
		final Image image = new Image(null, size.x, size.y);
		GC gc = new GC(image);
		SWTGraphics swtGraphics = new SWTGraphics(gc);
		swtGraphics.translate(-1 * bounds.x + viewLocation.x, -1 * bounds.y
				+ viewLocation.y);
		graph.getViewport().paint(swtGraphics);
		gc.copyArea(image, 0, 0);
		gc.dispose();
		ImageLoader loader = new ImageLoader();
		loader.data = new ImageData[] { image.getImageData() };
		loader.save(resultFile, SWT.IMAGE_PNG);
		return new File(resultFile);
	}

	/**
	 * @param dotDir
	 *            The directory containing the 'dot' executable of the local
	 *            Graphviz installation
	 * @param format
	 *            The image format to export the graph to (e.g. 'pdf' or 'png')
	 * @param resultFile
	 * @return The image file exported via DOT for the given Zest graph, or null
	 */
	public File toImage(final String dotDir, final String format,
			final String resultFile) {
		File dotFile = DotFileUtils.write(dotString);
		File image = DotDrawer.renderImage(new File(dotDir), dotFile, format,
				resultFile);
		return image;
	}

	@Override
	public String toString() {
		/* The exact name 'Graph' is not valid for rendering with Graphviz: */
		return graphName.equals("Graph") ? "Zest" + graphName //$NON-NLS-1$//$NON-NLS-2$
		: graphName;
	}

	private static String graphToDot(final Graph graph) {
		String raw = new DotTemplate().generate(graph);
		raw = removeBlankLines(raw);
		return raw;
	}

	private static String removeBlankLines(final String raw) {
		Scanner scanner = new Scanner(raw);
		StringBuilder builder = new StringBuilder();
		while (scanner.hasNextLine()) {
			String line = scanner.nextLine();
			if (!line.trim().equals("")) { //$NON-NLS-1$
				builder.append(line + "\n"); //$NON-NLS-1$
			}
		}
		return builder.toString();
	}

}
