/**
 * Copyright (c) 2010-2012, Abel Hegedus, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.incquery.testing.core;

import com.google.common.base.Objects;
import java.util.Iterator;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.incquery.patternlanguage.emf.eMFPatternLanguage.PatternModel;
import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IQuerySpecification;
import org.eclipse.incquery.runtime.api.IncQueryEngine;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.runtime.extensibility.QuerySpecificationRegistry;
import org.eclipse.incquery.runtime.util.XmiModelUtil;
import org.eclipse.incquery.runtime.util.XmiModelUtilRunningOptionEnum;
import org.eclipse.incquery.snapshot.EIQSnapshot.IncQuerySnapshot;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchSetRecord;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Helper methods for loading models from files or URIs.
 */
@SuppressWarnings("all")
public class ModelLoadHelper {
  /**
   * Load an instance EMF model from the given file to a new resource set.
   */
  public Resource loadModelFromFile(final IFile file) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    Resource _loadModelFromUri = this.loadModelFromUri(_string);
    return _loadModelFromUri;
  }
  
  /**
   * Load an instance EMF model from the given platform URI to a new resource set.
   */
  public Resource loadModelFromUri(final String platformUri) {
    Resource _xblockexpression = null;
    {
      ResourceSetImpl _resourceSetImpl = new ResourceSetImpl();
      final ResourceSetImpl resourceSet = _resourceSetImpl;
      Resource _loadAdditionalResourceFromUri = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
      _xblockexpression = (_loadAdditionalResourceFromUri);
    }
    return _xblockexpression;
  }
  
  /**
   * Load an additional resource into the resource set from a given file.
   * Works for both pattern and target model resource sets.
   */
  public Resource loadAdditionalResourceFromFile(final ResourceSet resourceSet, final IFile file) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    Resource _loadAdditionalResourceFromUri = this.loadAdditionalResourceFromUri(resourceSet, _string);
    return _loadAdditionalResourceFromUri;
  }
  
  /**
   * Load an additional resource into the resource set from a given platform URI.
   * Works for both pattern and target model resource sets.
   */
  public Resource loadAdditionalResourceFromUri(final ResourceSet resourceSet, final String platformUri) {
    Resource _xblockexpression = null;
    {
      final URI modelURI = XmiModelUtil.resolvePlatformURI(XmiModelUtilRunningOptionEnum.BOTH, platformUri);
      Resource _xifexpression = null;
      boolean _notEquals = (!Objects.equal(modelURI, null));
      if (_notEquals) {
        Resource _resource = resourceSet.getResource(modelURI, true);
        _xifexpression = _resource;
      }
      _xblockexpression = (_xifexpression);
    }
    return _xblockexpression;
  }
  
  /**
   * Load a pattern model from the given file into a new resource set.
   */
  public PatternModel loadPatternModelFromFile(final IFile file) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    PatternModel _loadPatternModelFromUri = this.loadPatternModelFromUri(_string);
    return _loadPatternModelFromUri;
  }
  
  /**
   * Load a pattern model from the given platform URI into a new resource set.
   */
  public PatternModel loadPatternModelFromUri(final String platformUri) {
    PatternModel _xblockexpression = null;
    {
      ResourceSet _prepareXtextResource = XmiModelUtil.prepareXtextResource();
      final Resource resource = this.loadAdditionalResourceFromUri(_prepareXtextResource, platformUri);
      PatternModel _xifexpression = null;
      EList<EObject> _contents = resource.getContents();
      int _size = _contents.size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        PatternModel _xifexpression_1 = null;
        EList<EObject> _contents_1 = resource.getContents();
        EObject _get = _contents_1.get(0);
        if ((_get instanceof PatternModel)) {
          EList<EObject> _contents_2 = resource.getContents();
          EObject _get_1 = _contents_2.get(0);
          _xifexpression_1 = ((PatternModel) _get_1);
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = (_xifexpression);
    }
    return _xblockexpression;
  }
  
  /**
   * Initialize a matcher for the pattern with the given name from the pattern model on the selected EMF root.
   */
  public IncQueryMatcher<? extends IPatternMatch> initializeMatcherFromModel(final PatternModel model, final IncQueryEngine engine, final String patternName) {
    try {
      IncQueryMatcher<? extends IPatternMatch> _xblockexpression = null;
      {
        EList<Pattern> _patterns = model.getPatterns();
        final Function1<Pattern,Boolean> _function = new Function1<Pattern,Boolean>() {
            public Boolean apply(final Pattern it) {
              boolean _xifexpression = false;
              String _packageName = model.getPackageName();
              boolean _equals = Objects.equal(_packageName, null);
              if (_equals) {
                String _name = it.getName();
                boolean _equals_1 = _name.equals(patternName);
                _xifexpression = _equals_1;
              } else {
                String _packageName_1 = model.getPackageName();
                String _plus = (_packageName_1 + ".");
                String _name_1 = it.getName();
                String _plus_1 = (_plus + _name_1);
                boolean _equals_2 = _plus_1.equals(patternName);
                _xifexpression = _equals_2;
              }
              return Boolean.valueOf(_xifexpression);
            }
          };
        final Iterable<Pattern> patterns = IterableExtensions.<Pattern>filter(_patterns, _function);
        IncQueryMatcher<? extends IPatternMatch> _xifexpression = null;
        int _size = IterableExtensions.size(patterns);
        boolean _equals = (_size == 1);
        if (_equals) {
          Iterator<Pattern> _iterator = patterns.iterator();
          Pattern _next = _iterator.next();
          IncQueryMatcher<? extends IPatternMatch> _matcher = engine.getMatcher(_next);
          _xifexpression = _matcher;
        }
        _xblockexpression = (_xifexpression);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public IncQueryMatcher<? extends IPatternMatch> initializeMatcherFromModel(final PatternModel model, final Notifier emfRoot, final String patternName) {
    try {
      IncQueryMatcher<? extends IPatternMatch> _xblockexpression = null;
      {
        final IncQueryEngine engine = IncQueryEngine.on(emfRoot);
        IncQueryMatcher<? extends IPatternMatch> _initializeMatcherFromModel = this.initializeMatcherFromModel(model, engine, patternName);
        _xblockexpression = (_initializeMatcherFromModel);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Initialize a registered matcher for the pattern FQN on the selected EMF root.
   */
  public IncQueryMatcher<? extends IPatternMatch> initializeMatcherFromRegistry(final Notifier emfRoot, final String patternFQN) {
    try {
      IncQueryMatcher<? extends IPatternMatch> _xblockexpression = null;
      {
        final IQuerySpecification<? extends IncQueryMatcher<? extends IPatternMatch>> querySpecification = QuerySpecificationRegistry.getQuerySpecification(patternFQN);
        IncQueryEngine _on = IncQueryEngine.on(emfRoot);
        IncQueryMatcher<? extends IPatternMatch> _matcher = querySpecification.getMatcher(_on);
        _xblockexpression = (_matcher);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Load the recorded match set into an existing resource set form the given file.
   */
  public IncQuerySnapshot loadExpectedResultsFromFile(final ResourceSet resourceSet, final IFile file) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    IncQuerySnapshot _loadExpectedResultsFromUri = this.loadExpectedResultsFromUri(resourceSet, _string);
    return _loadExpectedResultsFromUri;
  }
  
  /**
   * Load the recorded match set into an existing resource set form the given platform URI.
   */
  public IncQuerySnapshot loadExpectedResultsFromUri(final ResourceSet resourceSet, final String platformUri) {
    IncQuerySnapshot _xblockexpression = null;
    {
      final Resource resource = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
      IncQuerySnapshot _xifexpression = null;
      boolean _notEquals = (!Objects.equal(resource, null));
      if (_notEquals) {
        IncQuerySnapshot _xifexpression_1 = null;
        EList<EObject> _contents = resource.getContents();
        int _size = _contents.size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          IncQuerySnapshot _xifexpression_2 = null;
          EList<EObject> _contents_1 = resource.getContents();
          EObject _get = _contents_1.get(0);
          if ((_get instanceof IncQuerySnapshot)) {
            EList<EObject> _contents_2 = resource.getContents();
            EObject _get_1 = _contents_2.get(0);
            _xifexpression_2 = ((IncQuerySnapshot) _get_1);
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = (_xifexpression);
    }
    return _xblockexpression;
  }
  
  /**
   * Load the recorded match set into a new resource set form the given file.
   */
  public IncQuerySnapshot loadExpectedResultsFromFile(final IFile file) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    IncQuerySnapshot _loadExpectedResultsFromUri = this.loadExpectedResultsFromUri(_string);
    return _loadExpectedResultsFromUri;
  }
  
  /**
   * Load the recorded match set into a new resource set form the given platform URI.
   */
  public IncQuerySnapshot loadExpectedResultsFromUri(final String platformUri) {
    IncQuerySnapshot _xblockexpression = null;
    {
      final Resource resource = this.loadModelFromUri(platformUri);
      IncQuerySnapshot _xifexpression = null;
      boolean _notEquals = (!Objects.equal(resource, null));
      if (_notEquals) {
        IncQuerySnapshot _xifexpression_1 = null;
        EList<EObject> _contents = resource.getContents();
        int _size = _contents.size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          IncQuerySnapshot _xifexpression_2 = null;
          EList<EObject> _contents_1 = resource.getContents();
          EObject _get = _contents_1.get(0);
          if ((_get instanceof IncQuerySnapshot)) {
            EList<EObject> _contents_2 = resource.getContents();
            EObject _get_1 = _contents_2.get(0);
            _xifexpression_2 = ((IncQuerySnapshot) _get_1);
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = (_xifexpression);
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the match set record for a given pattern name after it loads the snapshot from the given file.
   */
  public MatchSetRecord loadExpectedResultsForPatternFromFile(final ResourceSet resourceSet, final IFile file, final String patternFQN) {
    IPath _fullPath = file.getFullPath();
    String _string = _fullPath.toString();
    MatchSetRecord _loadExpectedResultsForPatternFromUri = this.loadExpectedResultsForPatternFromUri(resourceSet, _string, patternFQN);
    return _loadExpectedResultsForPatternFromUri;
  }
  
  /**
   * Returns the match set record for a given pattern name after it loads the snapshot from the given platform URI.
   */
  public MatchSetRecord loadExpectedResultsForPatternFromUri(final ResourceSet resourceSet, final String platformUri, final String patternFQN) {
    MatchSetRecord _xblockexpression = null;
    {
      Resource _loadAdditionalResourceFromUri = this.loadAdditionalResourceFromUri(resourceSet, platformUri);
      final IncQuerySnapshot snapshot = ((IncQuerySnapshot) _loadAdditionalResourceFromUri);
      EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
      final Function1<MatchSetRecord,Boolean> _function = new Function1<MatchSetRecord,Boolean>() {
          public Boolean apply(final MatchSetRecord it) {
            String _patternQualifiedName = it.getPatternQualifiedName();
            boolean _equals = _patternQualifiedName.equals(patternFQN);
            return Boolean.valueOf(_equals);
          }
        };
      final Iterable<MatchSetRecord> matchsetrecord = IterableExtensions.<MatchSetRecord>filter(_matchSetRecords, _function);
      MatchSetRecord _xifexpression = null;
      int _size = IterableExtensions.size(matchsetrecord);
      boolean _equals = (_size == 1);
      if (_equals) {
        Iterator<MatchSetRecord> _iterator = matchsetrecord.iterator();
        return _iterator.next();
      }
      _xblockexpression = (_xifexpression);
    }
    return _xblockexpression;
  }
}
