
package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Platform}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutablePlatform.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutablePlatform implements Platform {
  private final @Nullable String architecture;
  private final String os;

  private ImmutablePlatform(@Nullable String architecture, String os) {
    this.architecture = architecture;
    this.os = os;
  }

  /**
   * @return The value of the {@code architecture} attribute
   */
  @JsonProperty("Architecture")
  @Override
  public @Nullable String architecture() {
    return architecture;
  }

  /**
   * @return The value of the {@code os} attribute
   */
  @JsonProperty("OS")
  @Override
  public String os() {
    return os;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Platform#architecture() architecture} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for architecture (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePlatform withArchitecture(@Nullable String value) {
    if (Objects.equals(this.architecture, value)) return this;
    return new ImmutablePlatform(value, this.os);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Platform#os() os} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for os
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePlatform withOs(String value) {
    String newValue = Objects.requireNonNull(value, "os");
    if (this.os.equals(newValue)) return this;
    return new ImmutablePlatform(this.architecture, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutablePlatform} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutablePlatform
        && equalsByValue((ImmutablePlatform) another);
  }

  private boolean equalsByValue(ImmutablePlatform another) {
    return Objects.equals(architecture, another.architecture)
        && os.equals(another.os);
  }

  /**
   * Computes a hash code from attributes: {@code architecture}, {@code os}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(architecture);
    h += (h << 5) + os.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Platform} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Platform{"
        + "architecture=" + architecture
        + ", os=" + os
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Platform} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Platform instance
   */
  public static ImmutablePlatform copyOf(Platform instance) {
    if (instance instanceof ImmutablePlatform) {
      return (ImmutablePlatform) instance;
    }
    return ImmutablePlatform.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutablePlatform ImmutablePlatform}.
   * <pre>
   * ImmutablePlatform.builder()
   *    .architecture(String | null) // nullable {@link Platform#architecture() architecture}
   *    .os(String) // required {@link Platform#os() os}
   *    .build();
   * </pre>
   * @return A new ImmutablePlatform builder
   */
  public static ImmutablePlatform.Builder builder() {
    return new ImmutablePlatform.Builder();
  }

  /**
   * Builds instances of type {@link ImmutablePlatform ImmutablePlatform}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_OS = 0x1L;
    private long initBits = 0x1L;

    private String architecture;
    private String os;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Platform} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Platform instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String architectureValue = instance.architecture();
      if (architectureValue != null) {
        architecture(architectureValue);
      }
      this.os(instance.os());
      return this;
    }

    /**
     * Initializes the value for the {@link Platform#architecture() architecture} attribute.
     * @param architecture The value for architecture (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Architecture")
    public final Builder architecture(@Nullable String architecture) {
      this.architecture = architecture;
      return this;
    }

    /**
     * Initializes the value for the {@link Platform#os() os} attribute.
     * @param os The value for os 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("OS")
    public final Builder os(String os) {
      this.os = Objects.requireNonNull(os, "os");
      initBits &= ~INIT_BIT_OS;
      return this;
    }

    /**
     * Builds a new {@link ImmutablePlatform ImmutablePlatform}.
     * @return An immutable instance of Platform
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutablePlatform build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutablePlatform(architecture, os);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_OS) != 0) attributes.add("os");
      return "Cannot build Platform, some of required attributes are not set " + attributes;
    }
  }
}
