/*******************************************************************************
 * Copyright (c) 2008 Ecliptical Software Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Ecliptical Software Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.emf.mint.internal.genmodel;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.codegen.ecore.genmodel.GenBase;
import org.eclipse.emf.codegen.ecore.genmodel.GenParameter;
import org.eclipse.emf.codegen.ecore.genmodel.util.GenModelSwitch;
import org.eclipse.emf.codegen.util.CodeGenUtil;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreSwitch;
import org.eclipse.emf.edit.provider.ChangeNotifier;
import org.eclipse.emf.edit.provider.ComposeableAdapterFactory;
import org.eclipse.emf.edit.provider.IChangeNotifier;
import org.eclipse.emf.edit.provider.IDisposable;
import org.eclipse.emf.edit.provider.INotifyChangedListener;
import org.eclipse.emf.mint.IItemJavaElementDescriptor;
import org.eclipse.emf.mint.util.GroupAdapterImpl;
import org.eclipse.emf.mint.util.ItemJavaElementDescriptor;
import org.eclipse.emf.mint.util.JavaElementNotification;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IField;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.Signature;

public abstract class JavaElementItemProviderAdapter extends GroupAdapterImpl
		implements IDisposable, IChangeNotifier {

	protected static final String CAT_INTERFACE = Messages.JavaElementItemProviderAdapter_CategoryModelInterface;

	protected static final String CAT_IMPLEMENTATION = Messages.JavaElementItemProviderAdapter_CategoryModelImplementation;

	protected static final String CAT_META_DATA = Messages.JavaElementItemProviderAdapter_CategoryModelMetaData;

	protected static final String CAT_UTILS = Messages.JavaElementItemProviderAdapter_CategoryModelUtilities;

	protected static final String CAT_PROVIDER = Messages.JavaElementItemProviderAdapter_CategoryEditSupport;

	protected static final String CAT_PRESENTATION = Messages.JavaElementItemProviderAdapter_CategoryEditorSupport;

	protected static final String CAT_TEST = Messages.JavaElementItemProviderAdapter_CategoryTestSupport;

	protected static class GroupReferenceGenModelSwitch extends
			GenModelSwitch<List<EReference>> {

		protected GroupReferenceEcoreSwitch groupReferenceEcoreSwitch;

		@Override
		public List<EReference> defaultCase(EObject object) {
			GroupReferenceEcoreSwitch groupReferenceEcoreSwitch = getGroupReferenceEcoreSwitch();
			if (groupReferenceEcoreSwitch != null) {
				List<EReference> groupReferences = groupReferenceEcoreSwitch
						.doSwitch(object);
				if (groupReferences != null)
					return groupReferences;
			}

			return Collections.emptyList();
		}

		protected GroupReferenceEcoreSwitch getGroupReferenceEcoreSwitch() {
			if (groupReferenceEcoreSwitch == null)
				groupReferenceEcoreSwitch = createGroupReferenceEcoreSwitch();

			return groupReferenceEcoreSwitch;
		}

		protected GroupReferenceEcoreSwitch createGroupReferenceEcoreSwitch() {
			return new GroupReferenceEcoreSwitch();
		}
	}

	protected static class GroupReferenceEcoreSwitch extends
			EcoreSwitch<List<EReference>> {
		@Override
		public List<EReference> defaultCase(EObject object) {
			return Collections.emptyList();
		}
	}

	protected static class ObservedFeatureGenModelSwitch extends
			GenModelSwitch<List<EStructuralFeature>> {

		protected ObservedFeatureEcoreSwitch observedFeatureEcoreSwitch;

		@Override
		public List<EStructuralFeature> defaultCase(EObject object) {
			ObservedFeatureEcoreSwitch observedFeatureEcoreSwitch = getObservedFeatureEcoreSwitch();
			if (observedFeatureEcoreSwitch != null) {
				List<EStructuralFeature> observedFeatures = observedFeatureEcoreSwitch
						.doSwitch(object);
				if (observedFeatures != null)
					return observedFeatures;
			}

			return Collections.emptyList();
		}

		protected ObservedFeatureEcoreSwitch getObservedFeatureEcoreSwitch() {
			if (observedFeatureEcoreSwitch == null)
				observedFeatureEcoreSwitch = createObservedFeatureEcoreSwitch();

			return observedFeatureEcoreSwitch;
		}

		protected ObservedFeatureEcoreSwitch createObservedFeatureEcoreSwitch() {
			return new ObservedFeatureEcoreSwitch();
		}
	}

	protected static class ObservedFeatureEcoreSwitch extends
			EcoreSwitch<List<EStructuralFeature>> {
		@Override
		public List<EStructuralFeature> defaultCase(EObject object) {
			return Collections.emptyList();
		}
	}

	protected final AdapterFactory adapterFactory;

	protected IChangeNotifier changeNotifier;

	protected GroupReferenceGenModelSwitch groupReferenceGenModelSwitch;

	protected ObservedFeatureGenModelSwitch observedFeatureGenModelSwitch;

	public JavaElementItemProviderAdapter(AdapterFactory adapterFactory) {
		this.adapterFactory = adapterFactory;
	}

	@Override
	public boolean isAdapterForType(Object type) {
		return type == adapterFactory;
	}

	public AdapterFactory getAdapterFactory() {
		return adapterFactory;
	}

	protected AdapterFactory getRootAdapterFactory() {
		if (adapterFactory instanceof ComposeableAdapterFactory)
			return ((ComposeableAdapterFactory) adapterFactory)
					.getRootAdapterFactory();

		return adapterFactory;
	}

	public void addListener(INotifyChangedListener listener) {
		if (changeNotifier == null)
			changeNotifier = new ChangeNotifier();

		changeNotifier.addListener(listener);
	}

	public void removeListener(INotifyChangedListener listener) {
		if (changeNotifier != null)
			changeNotifier.removeListener(listener);
	}

	public void fireNotifyChanged(Notification notification) {
		if (changeNotifier != null)
			changeNotifier.fireNotifyChanged(notification);

		if (adapterFactory instanceof IChangeNotifier) {
			IChangeNotifier changeNotifier = (IChangeNotifier) adapterFactory;
			changeNotifier.fireNotifyChanged(notification);
		}
	}

	public List<IItemJavaElementDescriptor> getJavaElementDescriptors(
			Object object) {
		return Collections.emptyList();
	}

	@Override
	public final void notifyChanged(Notification msg) {
		super.notifyChanged(msg);
		doNotifyChanged(msg);
	}

	protected void doNotifyChanged(Notification msg) {
		Object notifier = msg.getNotifier();
		Object feature = msg.getFeature();
		if (notifier instanceof EObject
				&& feature instanceof EStructuralFeature) {
			if (isObserved((EObject) notifier, (EStructuralFeature) feature))
				fireNotifyChanged(new JavaElementNotification(msg,
						getRootTarget()));
		}
	}

	@Override
	protected List<EReference> getGroupReferences(EObject object) {
		GroupReferenceGenModelSwitch groupReferenceSwitch = getGroupReferenceGenModelSwitch();
		if (groupReferenceSwitch != null) {
			List<EReference> groupReferences = groupReferenceSwitch
					.doSwitch(object);
			if (groupReferences != null)
				return groupReferences;
		}

		return Collections.emptyList();
	}

	protected boolean isObserved(EObject object, EStructuralFeature feature) {
		if (feature instanceof EReference
				&& isGroupReference(object, (EReference) feature))
			return true;

		ObservedFeatureGenModelSwitch observedFeatureSwitch = getObservedFeatureGenModelSwitch();
		if (observedFeatureSwitch != null) {
			List<EStructuralFeature> observedFeatures = observedFeatureSwitch
					.doSwitch(object);
			if (observedFeatures != null)
				return observedFeatures.contains(feature);
		}

		return false;
	}

	protected GroupReferenceGenModelSwitch getGroupReferenceGenModelSwitch() {
		if (groupReferenceGenModelSwitch == null)
			groupReferenceGenModelSwitch = createGroupReferenceGenModelSwitch();

		return groupReferenceGenModelSwitch;
	}

	protected GroupReferenceGenModelSwitch createGroupReferenceGenModelSwitch() {
		return new GroupReferenceGenModelSwitch();
	}

	protected ObservedFeatureGenModelSwitch getObservedFeatureGenModelSwitch() {
		if (observedFeatureGenModelSwitch == null)
			observedFeatureGenModelSwitch = createObservedFeatureGenModelSwitch();

		return observedFeatureGenModelSwitch;
	}

	protected ObservedFeatureGenModelSwitch createObservedFeatureGenModelSwitch() {
		return new ObservedFeatureGenModelSwitch();
	}

	protected IPackageFragment getPackageFragment(String dir, String name) {
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IFolder src = root.getFolder(new Path(dir));
		IJavaProject project = JavaCore.create(src.getProject());
		IPackageFragmentRoot pkgRoot = project.getPackageFragmentRoot(src);
		return pkgRoot.getPackageFragment(name);
	}

	protected IType getType(IPackageFragment pkg, String className) {
		int i = className.indexOf('$');
		String cuName = i == -1 ? className : className.substring(0, i);
		ICompilationUnit cu = pkg.getCompilationUnit(cuName + ".java"); //$NON-NLS-1$
		IType type = cu.getType(cuName);
		if (i != -1)
			type = type.getType(className.substring(i + 1));

		return type;
	}

	protected void addPackage(String dir, String name, String category,
			String description, String displayName, Object feature,
			Collection<IItemJavaElementDescriptor> list) {
		if (!isBlank(dir) && !isBlank(name))
			list.add(new ItemJavaElementDescriptor(
					getPackageFragment(dir, name), category, description,
					displayName, feature));
	}

	protected void addType(String dir, String className, String category,
			String description, String displayName, Object feature,
			Collection<IItemJavaElementDescriptor> list) {
		if (!isBlank(dir) && !isBlank(className)) {
			String pkgName = CodeGenUtil.getPackageName(className);
			IPackageFragment pkg = getPackageFragment(dir, pkgName);
			IType type = getType(pkg, CodeGenUtil.getSimpleClassName(className));
			list.add(new ItemJavaElementDescriptor(type, category, description,
					displayName, feature));
		}
	}

	protected void addType(IJavaProject ctx, String name, String category,
			String description, String displayName, Object feature,
			Collection<IItemJavaElementDescriptor> list) {
		if (!isBlank(name))
			list.add(new ItemJavaElementDescriptor(ctx, name, category,
					description, displayName, feature));
	}

	protected void addMethod(String dir, String className, String methodName,
			String category, String description, String displayName,
			Object feature, Collection<IItemJavaElementDescriptor> list) {
		addMethod(dir, className, methodName, (String[]) null, category,
				description, displayName, feature, list);
	}

	protected void addMethod(String dir, String className, String methodName,
			String paramType, String category, String description,
			String displayName, Object feature,
			Collection<IItemJavaElementDescriptor> list) {
		String[] paramSigs;
		if (paramType == null)
			paramSigs = null;
		else {
			String simpleTypeName = Signature.getSimpleName(paramType);
			paramSigs = new String[] { Signature.createTypeSignature(
					simpleTypeName, false) };
		}

		addMethod(dir, className, methodName, paramSigs, category, description,
				displayName, feature, list);
	}

	protected void addMethod(String dir, String className, String methodName,
			List<GenParameter> params, String category, String description,
			String displayName, Object feature,
			Collection<IItemJavaElementDescriptor> list) {
		addMethod(dir, className, methodName, params == null ? null
				: createParamSignatures(params), category, description,
				displayName, feature, list);
	}

	protected void addMethod(String dir, String className, String methodName,
			String[] paramSigs, String category, String description,
			String displayName, Object feature,
			Collection<IItemJavaElementDescriptor> list) {
		if (!isBlank(dir) && !isBlank(className) && !isBlank(methodName)) {
			String pkgName = CodeGenUtil.getPackageName(className);
			IPackageFragment pkg = getPackageFragment(dir, pkgName);
			IType type = getType(pkg, CodeGenUtil.getSimpleClassName(className));
			IMethod method = type.getMethod(methodName, paramSigs);
			list.add(new ItemJavaElementDescriptor(method, category,
					description, displayName, feature));
		}
	}

	protected String[] createParamSignatures(List<GenParameter> params) {
		String[] sigs = new String[params.size()];
		for (int i = 0; i < params.size(); ++i) {
			String typeName = params.get(i).getType(null);
			String simpleTypeName = Signature.getSimpleName(typeName);
			sigs[i] = Signature.createTypeSignature(simpleTypeName, false);
		}

		return sigs;
	}

	protected void addField(String dir, String className, String fieldName,
			String category, String description, String displayName,
			Object feature, Collection<IItemJavaElementDescriptor> list) {
		if (!isBlank(dir) && !isBlank(className) && !isBlank(fieldName)) {
			String pkgName = CodeGenUtil.getPackageName(className);
			IPackageFragment pkg = getPackageFragment(dir, pkgName);
			IType type = getType(pkg, CodeGenUtil.getSimpleClassName(className));
			IField field = type.getField(fieldName);
			list.add(new ItemJavaElementDescriptor(field, category,
					description, displayName, feature));
		}
	}

	protected static boolean isBlank(String dir) {
		return dir == null || dir.trim().length() == 0;
	}

	protected static boolean isFullyResolved(GenBase genBase) {
		if (genBase == null || genBase.eIsProxy())
			return false;

		EModelElement modelElement = genBase.getEcoreModelElement();
		if (modelElement == null || modelElement.eIsProxy())
			return false;

		return true;
	}

	protected static <T> List<T> appendList(List<T> base, T... elements) {
		ArrayList<T> result = new ArrayList<T>(base);
		result.addAll(Arrays.asList(elements));
		return result;
	}
}
