/**
 * Copyright (c) 2014 openHAB UG (haftungsbeschraenkt) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.smarthome.model.thing.internal;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smarthome.config.core.Configuration;
import org.eclipse.smarthome.core.common.registry.AbstractProvider;
import org.eclipse.smarthome.core.i18n.LocaleProvider;
import org.eclipse.smarthome.core.thing.Bridge;
import org.eclipse.smarthome.core.thing.Channel;
import org.eclipse.smarthome.core.thing.ChannelUID;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingProvider;
import org.eclipse.smarthome.core.thing.ThingTypeUID;
import org.eclipse.smarthome.core.thing.ThingUID;
import org.eclipse.smarthome.core.thing.binding.ThingHandlerFactory;
import org.eclipse.smarthome.core.thing.binding.builder.BridgeBuilder;
import org.eclipse.smarthome.core.thing.binding.builder.ChannelBuilder;
import org.eclipse.smarthome.core.thing.binding.builder.GenericThingBuilder;
import org.eclipse.smarthome.core.thing.binding.builder.ThingBuilder;
import org.eclipse.smarthome.core.thing.type.ChannelDefinition;
import org.eclipse.smarthome.core.thing.type.ChannelType;
import org.eclipse.smarthome.core.thing.type.ChannelTypeUID;
import org.eclipse.smarthome.core.thing.type.ThingType;
import org.eclipse.smarthome.core.thing.type.ThingTypeRegistry;
import org.eclipse.smarthome.core.thing.type.TypeResolver;
import org.eclipse.smarthome.core.thing.util.ThingHelper;
import org.eclipse.smarthome.model.core.ModelRepository;
import org.eclipse.smarthome.model.core.ModelRepositoryChangeListener;
import org.eclipse.smarthome.model.thing.thing.ModelBridge;
import org.eclipse.smarthome.model.thing.thing.ModelChannel;
import org.eclipse.smarthome.model.thing.thing.ModelProperty;
import org.eclipse.smarthome.model.thing.thing.ModelPropertyContainer;
import org.eclipse.smarthome.model.thing.thing.ModelThing;
import org.eclipse.smarthome.model.thing.thing.ThingModel;
import org.eclipse.xtend.lib.Data;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * {@link ThingProvider} implementation which computes *.things files.
 * 
 * @author Oliver Libutzki - Initial contribution
 * @author niehues - Fix ESH Bug 450236
 *         https://bugs.eclipse.org/bugs/show_bug.cgi?id=450236 - Considering
 *         ThingType Description
 * @author Simon Kaufmann - Added asynchronous retry in case the handler
 *         factory cannot load a thing yet (bug 470368)
 * @author Markus Rathgeb - Add locale provider support
 */
@SuppressWarnings("all")
public class GenericThingProvider extends AbstractProvider<Thing> implements ThingProvider, ModelRepositoryChangeListener {
  @Data
  private static final class QueueContent {
    private final ThingTypeUID _thingTypeUID;
    
    private final String _label;
    
    private final Configuration _configuration;
    
    private final ThingUID _thingUID;
    
    private final ThingUID _bridgeUID;
    
    private final ThingHandlerFactory _thingHandlerFactory;
    
    public QueueContent(final ThingTypeUID thingTypeUID, final String label, final Configuration configuration, final ThingUID thingUID, final ThingUID bridgeUID, final ThingHandlerFactory thingHandlerFactory) {
      super();
      this._thingTypeUID = thingTypeUID;
      this._label = label;
      this._configuration = configuration;
      this._thingUID = thingUID;
      this._bridgeUID = bridgeUID;
      this._thingHandlerFactory = thingHandlerFactory;
    }
    
    @Override
    @Pure
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((this._thingTypeUID== null) ? 0 : this._thingTypeUID.hashCode());
      result = prime * result + ((this._label== null) ? 0 : this._label.hashCode());
      result = prime * result + ((this._configuration== null) ? 0 : this._configuration.hashCode());
      result = prime * result + ((this._thingUID== null) ? 0 : this._thingUID.hashCode());
      result = prime * result + ((this._bridgeUID== null) ? 0 : this._bridgeUID.hashCode());
      result = prime * result + ((this._thingHandlerFactory== null) ? 0 : this._thingHandlerFactory.hashCode());
      return result;
    }
    
    @Override
    @Pure
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      GenericThingProvider.QueueContent other = (GenericThingProvider.QueueContent) obj;
      if (this._thingTypeUID == null) {
        if (other._thingTypeUID != null)
          return false;
      } else if (!this._thingTypeUID.equals(other._thingTypeUID))
        return false;
      if (this._label == null) {
        if (other._label != null)
          return false;
      } else if (!this._label.equals(other._label))
        return false;
      if (this._configuration == null) {
        if (other._configuration != null)
          return false;
      } else if (!this._configuration.equals(other._configuration))
        return false;
      if (this._thingUID == null) {
        if (other._thingUID != null)
          return false;
      } else if (!this._thingUID.equals(other._thingUID))
        return false;
      if (this._bridgeUID == null) {
        if (other._bridgeUID != null)
          return false;
      } else if (!this._bridgeUID.equals(other._bridgeUID))
        return false;
      if (this._thingHandlerFactory == null) {
        if (other._thingHandlerFactory != null)
          return false;
      } else if (!this._thingHandlerFactory.equals(other._thingHandlerFactory))
        return false;
      return true;
    }
    
    @Override
    @Pure
    public String toString() {
      String result = new ToStringHelper().toString(this);
      return result;
    }
    
    @Pure
    public ThingTypeUID getThingTypeUID() {
      return this._thingTypeUID;
    }
    
    @Pure
    public String getLabel() {
      return this._label;
    }
    
    @Pure
    public Configuration getConfiguration() {
      return this._configuration;
    }
    
    @Pure
    public ThingUID getThingUID() {
      return this._thingUID;
    }
    
    @Pure
    public ThingUID getBridgeUID() {
      return this._bridgeUID;
    }
    
    @Pure
    public ThingHandlerFactory getThingHandlerFactory() {
      return this._thingHandlerFactory;
    }
  }
  
  private LocaleProvider localeProvider;
  
  private ModelRepository modelRepository;
  
  private ThingTypeRegistry thingTypeRegistry;
  
  private Map<String, Collection<Thing>> thingsMap = new ConcurrentHashMap<String, Collection<Thing>>();
  
  private List<ThingHandlerFactory> thingHandlerFactories = new CopyOnWriteArrayList<ThingHandlerFactory>();
  
  private final List<GenericThingProvider.QueueContent> queue = new CopyOnWriteArrayList<GenericThingProvider.QueueContent>();
  
  private Thread lazyRetryThread = null;
  
  private final static Logger logger = LoggerFactory.getLogger(GenericThingProvider.class);
  
  public void activate() {
    Iterable<String> _allModelNamesOfType = this.modelRepository.getAllModelNamesOfType("things");
    final Procedure1<String> _function = new Procedure1<String>() {
      @Override
      public void apply(final String it) {
        GenericThingProvider.this.createThingsFromModel(it);
      }
    };
    IterableExtensions.<String>forEach(_allModelNamesOfType, _function);
  }
  
  @Override
  public Collection<Thing> getAll() {
    Collection<Collection<Thing>> _values = this.thingsMap.values();
    Iterable<Thing> _flatten = Iterables.<Thing>concat(_values);
    return IterableExtensions.<Thing>toList(_flatten);
  }
  
  private void createThingsFromModel(final String modelName) {
    GenericThingProvider.logger.debug("Read things from model \'{}\'", modelName);
    final ArrayList<Thing> things = CollectionLiterals.<Thing>newArrayList();
    boolean _notEquals = (!Objects.equal(this.modelRepository, null));
    if (_notEquals) {
      EObject _model = this.modelRepository.getModel(modelName);
      final ThingModel model = ((ThingModel) _model);
      boolean _notEquals_1 = (!Objects.equal(model, null));
      if (_notEquals_1) {
        EList<ModelThing> _things = model.getThings();
        final Procedure1<ModelThing> _function = new Procedure1<ModelThing>() {
          @Override
          public void apply(final ModelThing it) {
            GenericThingProvider.this.createThing(it, null, things);
          }
        };
        IterableExtensions.<ModelThing>forEach(_things, _function);
      }
    }
    this.thingsMap.put(modelName, things);
  }
  
  private void createThing(final ModelThing modelThing, final Bridge parentBridge, final Collection<Thing> thingList) {
    this.createThing(modelThing, parentBridge, thingList, null);
  }
  
  private void createThing(final ModelThing modelThing, final Bridge parentBridge, final Collection<Thing> thingList, final ThingHandlerFactory thingHandlerFactory) {
    ThingTypeUID thingTypeUID = null;
    ThingUID thingUID = null;
    ThingUID bridgeUID = null;
    boolean _notEquals = (!Objects.equal(parentBridge, null));
    if (_notEquals) {
      ThingTypeUID _thingTypeUID = parentBridge.getThingTypeUID();
      final String bindingId = _thingTypeUID.getBindingId();
      final String thingTypeId = modelThing.getThingTypeId();
      final String thingId = modelThing.getThingId();
      ThingTypeUID _thingTypeUID_1 = new ThingTypeUID(bindingId, thingTypeId);
      thingTypeUID = _thingTypeUID_1;
      List<String> _parentPath = this.getParentPath(parentBridge);
      ThingUID _thingUID = new ThingUID(thingTypeUID, thingId, ((String[])Conversions.unwrapArray(_parentPath, String.class)));
      thingUID = _thingUID;
      ThingUID _uID = parentBridge.getUID();
      bridgeUID = _uID;
    } else {
      String _id = modelThing.getId();
      ThingUID _thingUID_1 = new ThingUID(_id);
      thingUID = _thingUID_1;
      String _bindingId = thingUID.getBindingId();
      String _thingTypeId = thingUID.getThingTypeId();
      ThingTypeUID _thingTypeUID_2 = new ThingTypeUID(_bindingId, _thingTypeId);
      thingTypeUID = _thingTypeUID_2;
      boolean _and = false;
      String _bridgeUID = modelThing.getBridgeUID();
      boolean _notEquals_1 = (!Objects.equal(_bridgeUID, null));
      if (!_notEquals_1) {
        _and = false;
      } else {
        String _bridgeUID_1 = modelThing.getBridgeUID();
        boolean _isEmpty = _bridgeUID_1.isEmpty();
        boolean _not = (!_isEmpty);
        _and = _not;
      }
      if (_and) {
        String _bridgeUID_2 = modelThing.getBridgeUID();
        ThingUID _thingUID_2 = new ThingUID(_bridgeUID_2);
        bridgeUID = _thingUID_2;
      }
    }
    boolean _isSupportedByThingHandlerFactory = this.isSupportedByThingHandlerFactory(thingTypeUID, thingHandlerFactory);
    boolean _not_1 = (!_isSupportedByThingHandlerFactory);
    if (_not_1) {
      return;
    }
    GenericThingProvider.logger.trace("Creating thing for type \'{}\' with UID \'{}.", thingTypeUID, thingUID);
    final Configuration configuration = this.createConfiguration(modelThing);
    final ThingUID uid = thingUID;
    final Function1<Thing, Boolean> _function = new Function1<Thing, Boolean>() {
      @Override
      public Boolean apply(final Thing it) {
        ThingUID _uID = it.getUID();
        return Boolean.valueOf(_uID.equals(uid));
      }
    };
    boolean _exists = IterableExtensions.<Thing>exists(thingList, _function);
    if (_exists) {
      String _string = uid.toString();
      GenericThingProvider.logger.debug("Thing already exists {}", _string);
      return;
    }
    final ThingType thingType = this.getThingType(thingTypeUID);
    String _xifexpression = null;
    String _label = modelThing.getLabel();
    boolean _notEquals_2 = (!Objects.equal(_label, null));
    if (_notEquals_2) {
      _xifexpression = modelThing.getLabel();
    } else {
      String _label_1 = null;
      if (thingType!=null) {
        _label_1=thingType.getLabel();
      }
      _xifexpression = _label_1;
    }
    final String label = _xifexpression;
    final Thing thingFromHandler = this.getThingFromThingHandlerFactories(thingTypeUID, label, configuration, thingUID, bridgeUID, thingHandlerFactory);
    GenericThingBuilder<?> _xifexpression_1 = null;
    if ((modelThing instanceof ModelBridge)) {
      _xifexpression_1 = BridgeBuilder.create(thingTypeUID, thingUID);
    } else {
      _xifexpression_1 = ThingBuilder.create(thingTypeUID, thingUID);
    }
    final GenericThingBuilder<?> thingBuilder = _xifexpression_1;
    thingBuilder.withConfiguration(configuration);
    thingBuilder.withBridge(bridgeUID);
    thingBuilder.withLabel(label);
    EList<ModelChannel> _channels = modelThing.getChannels();
    List<ChannelDefinition> _elvis = null;
    List<ChannelDefinition> _channelDefinitions = null;
    if (thingType!=null) {
      _channelDefinitions=thingType.getChannelDefinitions();
    }
    if (_channelDefinitions != null) {
      _elvis = _channelDefinitions;
    } else {
      ArrayList<ChannelDefinition> _newArrayList = CollectionLiterals.<ChannelDefinition>newArrayList();
      _elvis = _newArrayList;
    }
    final List<Channel> channels = this.createChannels(thingTypeUID, thingUID, _channels, _elvis);
    thingBuilder.withChannels(channels);
    Thing thing = thingBuilder.build();
    boolean _notEquals_3 = (!Objects.equal(thingFromHandler, null));
    if (_notEquals_3) {
      this.merge(thingFromHandler, thing);
    }
    Thing _elvis_1 = null;
    if (thingFromHandler != null) {
      _elvis_1 = thingFromHandler;
    } else {
      _elvis_1 = thing;
    }
    thingList.add(_elvis_1);
    if ((modelThing instanceof ModelBridge)) {
      Thing _elvis_2 = null;
      if (thingFromHandler != null) {
        _elvis_2 = thingFromHandler;
      } else {
        _elvis_2 = thing;
      }
      final Bridge bridge = ((Bridge) _elvis_2);
      EList<ModelThing> _things = ((ModelBridge)modelThing).getThings();
      final Procedure1<ModelThing> _function_1 = new Procedure1<ModelThing>() {
        @Override
        public void apply(final ModelThing it) {
          GenericThingProvider.this.createThing(it, ((Bridge) bridge), thingList, thingHandlerFactory);
        }
      };
      IterableExtensions.<ModelThing>forEach(_things, _function_1);
    }
  }
  
  private boolean isSupportedByThingHandlerFactory(final ThingTypeUID thingTypeUID, final ThingHandlerFactory specific) {
    if ((specific != null)) {
      return specific.supportsThingType(thingTypeUID);
    }
    for (final ThingHandlerFactory thingHandlerFactory : this.thingHandlerFactories) {
      boolean _supportsThingType = thingHandlerFactory.supportsThingType(thingTypeUID);
      if (_supportsThingType) {
        return true;
      }
    }
    return false;
  }
  
  private Thing getThingFromThingHandlerFactories(final ThingTypeUID thingTypeUID, final String label, final Configuration configuration, final ThingUID thingUID, final ThingUID bridgeUID, final ThingHandlerFactory specific) {
    Object _xblockexpression = null;
    {
      boolean _and = false;
      boolean _notEquals = (!Objects.equal(specific, null));
      if (!_notEquals) {
        _and = false;
      } else {
        boolean _supportsThingType = specific.supportsThingType(thingTypeUID);
        _and = _supportsThingType;
      }
      if (_and) {
        GenericThingProvider.logger.trace("Creating thing from specific ThingHandlerFactory {} for thingType {}", specific, thingTypeUID);
        return this.getThingFromThingHandlerFactory(thingTypeUID, label, configuration, thingUID, bridgeUID, specific);
      }
      for (final ThingHandlerFactory thingHandlerFactory : this.thingHandlerFactories) {
        {
          GenericThingProvider.logger.trace("Searching thingHandlerFactory for thingType: {}", thingTypeUID);
          boolean _supportsThingType_1 = thingHandlerFactory.supportsThingType(thingTypeUID);
          if (_supportsThingType_1) {
            return this.getThingFromThingHandlerFactory(thingTypeUID, label, configuration, thingUID, bridgeUID, thingHandlerFactory);
          }
        }
      }
      _xblockexpression = null;
    }
    return ((Thing)_xblockexpression);
  }
  
  private Thing getThingFromThingHandlerFactory(final ThingTypeUID thingTypeUID, final String label, final Configuration configuration, final ThingUID thingUID, final ThingUID bridgeUID, final ThingHandlerFactory thingHandlerFactory) {
    final Thing thing = thingHandlerFactory.createThing(thingTypeUID, configuration, thingUID, bridgeUID);
    boolean _equals = Objects.equal(thing, null);
    if (_equals) {
      Class<? extends ThingHandlerFactory> _class = thingHandlerFactory.getClass();
      String _simpleName = _class.getSimpleName();
      String _asString = thingTypeUID.getAsString();
      GenericThingProvider.logger.debug(
        "ThingHandlerFactory \'{}\' claimed it can handle \'{}\' type but actually did not. Queued for later refresh.", _simpleName, _asString);
      GenericThingProvider.QueueContent _queueContent = new GenericThingProvider.QueueContent(thingTypeUID, label, configuration, thingUID, bridgeUID, thingHandlerFactory);
      this.queue.add(_queueContent);
      boolean _or = false;
      boolean _equals_1 = Objects.equal(this.lazyRetryThread, null);
      if (_equals_1) {
        _or = true;
      } else {
        boolean _isAlive = this.lazyRetryThread.isAlive();
        boolean _not = (!_isAlive);
        _or = _not;
      }
      if (_or) {
        Thread _thread = new Thread(this.lazyRetryRunnable);
        this.lazyRetryThread = _thread;
        this.lazyRetryThread.start();
      }
    } else {
      thing.setLabel(label);
    }
    return thing;
  }
  
  protected void _merge(final Thing targetThing, final Thing sourceThing) {
    ThingUID _bridgeUID = sourceThing.getBridgeUID();
    targetThing.setBridgeUID(_bridgeUID);
    Configuration _configuration = targetThing.getConfiguration();
    Configuration _configuration_1 = sourceThing.getConfiguration();
    this.merge(_configuration, _configuration_1);
    List<Channel> _channels = sourceThing.getChannels();
    this.merge(targetThing, _channels);
  }
  
  protected void _merge(final Configuration target, final Configuration source) {
    Set<String> _keySet = source.keySet();
    final Procedure1<String> _function = new Procedure1<String>() {
      @Override
      public void apply(final String it) {
        Object _get = source.get(it);
        target.put(it, _get);
      }
    };
    IterableExtensions.<String>forEach(_keySet, _function);
  }
  
  protected void _merge(final Thing targetThing, final List<Channel> source) {
    final List<Channel> channelsToAdd = CollectionLiterals.<Channel>newArrayList();
    final Procedure1<Channel> _function = new Procedure1<Channel>() {
      @Override
      public void apply(final Channel sourceChannel) {
        List<Channel> _channels = targetThing.getChannels();
        final Function1<Channel, Boolean> _function = new Function1<Channel, Boolean>() {
          @Override
          public Boolean apply(final Channel it) {
            ChannelUID _uID = it.getUID();
            ChannelUID _uID_1 = sourceChannel.getUID();
            return Boolean.valueOf(_uID.equals(_uID_1));
          }
        };
        final Iterable<Channel> targetChannels = IterableExtensions.<Channel>filter(_channels, _function);
        final Procedure1<Channel> _function_1 = new Procedure1<Channel>() {
          @Override
          public void apply(final Channel it) {
            GenericThingProvider.this.merge(it, sourceChannel);
          }
        };
        IterableExtensions.<Channel>forEach(targetChannels, _function_1);
        boolean _isEmpty = IterableExtensions.isEmpty(targetChannels);
        if (_isEmpty) {
          channelsToAdd.add(sourceChannel);
        }
      }
    };
    IterableExtensions.<Channel>forEach(source, _function);
    ThingHelper.addChannelsToThing(targetThing, channelsToAdd);
  }
  
  protected void _merge(final Channel target, final Channel source) {
    Configuration _configuration = target.getConfiguration();
    Configuration _configuration_1 = source.getConfiguration();
    this.merge(_configuration, _configuration_1);
  }
  
  private List<String> getParentPath(final Bridge bridge) {
    ThingUID _uID = bridge.getUID();
    List<String> bridgeIds = _uID.getBridgeIds();
    ThingUID _uID_1 = bridge.getUID();
    String _id = _uID_1.getId();
    bridgeIds.add(_id);
    return bridgeIds;
  }
  
  private List<Channel> createChannels(final ThingTypeUID thingTypeUID, final ThingUID thingUID, final List<ModelChannel> modelChannels, final List<ChannelDefinition> channelDefinitions) {
    List<Channel> _xblockexpression = null;
    {
      final Set<String> addedChannelIds = CollectionLiterals.<String>newHashSet();
      final List<Channel> channels = CollectionLiterals.<Channel>newArrayList();
      final Procedure1<ModelChannel> _function = new Procedure1<ModelChannel>() {
        @Override
        public void apply(final ModelChannel it) {
          String _id = it.getId();
          boolean _add = addedChannelIds.add(_id);
          if (_add) {
            String _id_1 = it.getId();
            ChannelUID _channelUID = new ChannelUID(thingTypeUID, thingUID, _id_1);
            String _type = it.getType();
            ChannelBuilder _create = ChannelBuilder.create(_channelUID, _type);
            Configuration _createConfiguration = GenericThingProvider.this.createConfiguration(it);
            ChannelBuilder _withConfiguration = _create.withConfiguration(_createConfiguration);
            Channel _build = _withConfiguration.build();
            channels.add(_build);
          }
        }
      };
      IterableExtensions.<ModelChannel>forEach(modelChannels, _function);
      final Procedure1<ChannelDefinition> _function_1 = new Procedure1<ChannelDefinition>() {
        @Override
        public void apply(final ChannelDefinition it) {
          String _id = it.getId();
          boolean _add = addedChannelIds.add(_id);
          if (_add) {
            ChannelTypeUID _channelTypeUID = it.getChannelTypeUID();
            final ChannelType channelType = TypeResolver.resolve(_channelTypeUID);
            boolean _notEquals = (!Objects.equal(channelType, null));
            if (_notEquals) {
              String _id_1 = it.getId();
              ChannelUID _channelUID = new ChannelUID(thingTypeUID, thingUID, _id_1);
              String _itemType = channelType.getItemType();
              ChannelBuilder _create = ChannelBuilder.create(_channelUID, _itemType);
              ChannelTypeUID _channelTypeUID_1 = it.getChannelTypeUID();
              ChannelBuilder _withType = _create.withType(_channelTypeUID_1);
              Channel _build = _withType.build();
              channels.add(_build);
            } else {
              String _id_2 = it.getId();
              ChannelTypeUID _channelTypeUID_2 = it.getChannelTypeUID();
              GenericThingProvider.logger.warn(
                "Could not create channel \'{}\' for thing \'{}\', because channel type \'{}\' could not be found.", _id_2, thingUID, _channelTypeUID_2);
            }
          }
        }
      };
      IterableExtensions.<ChannelDefinition>forEach(channelDefinitions, _function_1);
      _xblockexpression = channels;
    }
    return _xblockexpression;
  }
  
  private Configuration createConfiguration(final ModelPropertyContainer propertyContainer) {
    Configuration _xblockexpression = null;
    {
      final Configuration configuration = new Configuration();
      EList<ModelProperty> _properties = propertyContainer.getProperties();
      final Procedure1<ModelProperty> _function = new Procedure1<ModelProperty>() {
        @Override
        public void apply(final ModelProperty it) {
          String _key = it.getKey();
          Object _value = it.getValue();
          configuration.put(_key, _value);
        }
      };
      IterableExtensions.<ModelProperty>forEach(_properties, _function);
      _xblockexpression = configuration;
    }
    return _xblockexpression;
  }
  
  private ThingType getThingType(final ThingTypeUID thingTypeUID) {
    ThingType _thingType = null;
    if (this.thingTypeRegistry!=null) {
      Locale _locale = this.localeProvider.getLocale();
      _thingType=this.thingTypeRegistry.getThingType(thingTypeUID, _locale);
    }
    return _thingType;
  }
  
  protected void setModelRepository(final ModelRepository modelRepository) {
    this.modelRepository = modelRepository;
    modelRepository.addModelRepositoryChangeListener(this);
  }
  
  protected void unsetModelRepository(final ModelRepository modelRepository) {
    modelRepository.removeModelRepositoryChangeListener(this);
    this.modelRepository = null;
  }
  
  @Override
  public void modelChanged(final String modelName, final org.eclipse.smarthome.model.core.EventType type) {
    boolean _endsWith = modelName.endsWith("things");
    if (_endsWith) {
      if (type != null) {
        switch (type) {
          case ADDED:
            this.createThingsFromModel(modelName);
            Collection<Thing> _elvis = null;
            Collection<Thing> _get = this.thingsMap.get(modelName);
            if (_get != null) {
              _elvis = _get;
            } else {
              ArrayList<Thing> _newArrayList = CollectionLiterals.<Thing>newArrayList();
              _elvis = _newArrayList;
            }
            final Collection<Thing> things = _elvis;
            final Procedure1<Thing> _function = new Procedure1<Thing>() {
              @Override
              public void apply(final Thing it) {
                GenericThingProvider.this.notifyListenersAboutAddedElement(it);
              }
            };
            IterableExtensions.<Thing>forEach(things, _function);
            break;
          case MODIFIED:
            Collection<Thing> _elvis_1 = null;
            Collection<Thing> _get_1 = this.thingsMap.get(modelName);
            if (_get_1 != null) {
              _elvis_1 = _get_1;
            } else {
              ArrayList<Thing> _newArrayList_1 = CollectionLiterals.<Thing>newArrayList();
              _elvis_1 = _newArrayList_1;
            }
            final Collection<Thing> oldThings = _elvis_1;
            final Function1<Thing, ThingUID> _function_1 = new Function1<Thing, ThingUID>() {
              @Override
              public ThingUID apply(final Thing it) {
                return it.getUID();
              }
            };
            Iterable<ThingUID> _map = IterableExtensions.<Thing, ThingUID>map(oldThings, _function_1);
            final List<ThingUID> oldThingsUIDs = IterableExtensions.<ThingUID>toList(_map);
            this.createThingsFromModel(modelName);
            Collection<Thing> _elvis_2 = null;
            Collection<Thing> _get_2 = this.thingsMap.get(modelName);
            if (_get_2 != null) {
              _elvis_2 = _get_2;
            } else {
              ArrayList<Thing> _newArrayList_2 = CollectionLiterals.<Thing>newArrayList();
              _elvis_2 = _newArrayList_2;
            }
            final Collection<Thing> currentThings = _elvis_2;
            final Function1<Thing, ThingUID> _function_2 = new Function1<Thing, ThingUID>() {
              @Override
              public ThingUID apply(final Thing it) {
                return it.getUID();
              }
            };
            Iterable<ThingUID> _map_1 = IterableExtensions.<Thing, ThingUID>map(currentThings, _function_2);
            final List<ThingUID> currentThingUIDs = IterableExtensions.<ThingUID>toList(_map_1);
            final Function1<Thing, Boolean> _function_3 = new Function1<Thing, Boolean>() {
              @Override
              public Boolean apply(final Thing it) {
                ThingUID _uID = it.getUID();
                boolean _contains = currentThingUIDs.contains(_uID);
                return Boolean.valueOf((!_contains));
              }
            };
            final Iterable<Thing> removedThings = IterableExtensions.<Thing>filter(oldThings, _function_3);
            final Function1<Thing, Boolean> _function_4 = new Function1<Thing, Boolean>() {
              @Override
              public Boolean apply(final Thing it) {
                ThingUID _uID = it.getUID();
                boolean _contains = oldThingsUIDs.contains(_uID);
                return Boolean.valueOf((!_contains));
              }
            };
            final Iterable<Thing> addedThings = IterableExtensions.<Thing>filter(currentThings, _function_4);
            final Procedure1<Thing> _function_5 = new Procedure1<Thing>() {
              @Override
              public void apply(final Thing it) {
                GenericThingProvider.this.notifyListenersAboutRemovedElement(it);
              }
            };
            IterableExtensions.<Thing>forEach(removedThings, _function_5);
            final Procedure1<Thing> _function_6 = new Procedure1<Thing>() {
              @Override
              public void apply(final Thing it) {
                GenericThingProvider.this.notifyListenersAboutAddedElement(it);
              }
            };
            IterableExtensions.<Thing>forEach(addedThings, _function_6);
            final Procedure1<Thing> _function_7 = new Procedure1<Thing>() {
              @Override
              public void apply(final Thing newThing) {
                final Function1<Thing, Boolean> _function = new Function1<Thing, Boolean>() {
                  @Override
                  public Boolean apply(final Thing it) {
                    ThingUID _uID = it.getUID();
                    ThingUID _uID_1 = newThing.getUID();
                    return Boolean.valueOf(Objects.equal(_uID, _uID_1));
                  }
                };
                final Thing oldThing = IterableExtensions.<Thing>findFirst(oldThings, _function);
                boolean _notEquals = (!Objects.equal(oldThing, null));
                if (_notEquals) {
                  boolean _equals = ThingHelper.equals(oldThing, newThing);
                  boolean _not = (!_equals);
                  if (_not) {
                    GenericThingProvider.this.notifyListenersAboutUpdatedElement(oldThing, newThing);
                  }
                }
              }
            };
            IterableExtensions.<Thing>forEach(currentThings, _function_7);
            break;
          case REMOVED:
            Collection<Thing> _elvis_3 = null;
            Collection<Thing> _remove = this.thingsMap.remove(modelName);
            if (_remove != null) {
              _elvis_3 = _remove;
            } else {
              ArrayList<Thing> _newArrayList_3 = CollectionLiterals.<Thing>newArrayList();
              _elvis_3 = _newArrayList_3;
            }
            final Collection<Thing> things_1 = _elvis_3;
            final Procedure1<Thing> _function_8 = new Procedure1<Thing>() {
              @Override
              public void apply(final Thing it) {
                GenericThingProvider.this.notifyListenersAboutRemovedElement(it);
              }
            };
            IterableExtensions.<Thing>forEach(things_1, _function_8);
            break;
          default:
            break;
        }
      }
    }
  }
  
  protected void setLocaleProvider(final LocaleProvider localeProvider) {
    this.localeProvider = localeProvider;
  }
  
  protected void unsetLocaleProvider(final LocaleProvider localeProvider) {
    this.localeProvider = null;
  }
  
  protected void setThingTypeRegistry(final ThingTypeRegistry thingTypeRegistry) {
    this.thingTypeRegistry = thingTypeRegistry;
  }
  
  protected void unsetThingTypeRegistry(final ThingTypeRegistry thingTypeRegistry) {
    this.thingTypeRegistry = null;
  }
  
  protected void addThingHandlerFactory(final ThingHandlerFactory thingHandlerFactory) {
    GenericThingProvider.logger.debug("ThingHandlerFactory added {}", thingHandlerFactory);
    this.thingHandlerFactories.add(thingHandlerFactory);
    this.thingHandlerFactoryAdded(thingHandlerFactory);
  }
  
  protected void removeThingHandlerFactory(final ThingHandlerFactory thingHandlerFactory) {
    this.thingHandlerFactories.remove(thingHandlerFactory);
    this.thingHandlerFactoryRemoved();
  }
  
  private Object thingHandlerFactoryRemoved() {
    return null;
  }
  
  private void thingHandlerFactoryAdded(final ThingHandlerFactory thingHandlerFactory) {
    Set<String> _keySet = this.thingsMap.keySet();
    final Procedure1<String> _function = new Procedure1<String>() {
      @Override
      public void apply(final String it) {
        GenericThingProvider.this.createThingsFromModelForThingHandlerFactory(it, thingHandlerFactory);
      }
    };
    IterableExtensions.<String>forEach(_keySet, _function);
  }
  
  private void createThingsFromModelForThingHandlerFactory(final String modelName, final ThingHandlerFactory factory) {
    Collection<Thing> _get = this.thingsMap.get(modelName);
    final Thing[] oldThings = ((Thing[])Conversions.unwrapArray(_get, Thing.class)).clone();
    final ArrayList<Thing> newThings = CollectionLiterals.<Thing>newArrayList();
    boolean _notEquals = (!Objects.equal(this.modelRepository, null));
    if (_notEquals) {
      EObject _model = this.modelRepository.getModel(modelName);
      final ThingModel model = ((ThingModel) _model);
      boolean _notEquals_1 = (!Objects.equal(model, null));
      if (_notEquals_1) {
        EList<ModelThing> _things = model.getThings();
        final Procedure1<ModelThing> _function = new Procedure1<ModelThing>() {
          @Override
          public void apply(final ModelThing it) {
            GenericThingProvider.this.createThing(it, null, newThings, factory);
          }
        };
        IterableExtensions.<ModelThing>forEach(_things, _function);
      }
    }
    final Procedure1<Thing> _function_1 = new Procedure1<Thing>() {
      @Override
      public void apply(final Thing newThing) {
        final Function1<Thing, Boolean> _function = new Function1<Thing, Boolean>() {
          @Override
          public Boolean apply(final Thing it) {
            ThingUID _uID = it.getUID();
            ThingUID _uID_1 = newThing.getUID();
            return Boolean.valueOf(Objects.equal(_uID, _uID_1));
          }
        };
        final Thing oldThing = IterableExtensions.<Thing>findFirst(((Iterable<Thing>)Conversions.doWrapArray(oldThings)), _function);
        boolean _notEquals = (!Objects.equal(oldThing, null));
        if (_notEquals) {
        } else {
          ThingUID _uID = newThing.getUID();
          GenericThingProvider.logger.debug("Adding thing \'{}\' from model \'{}.", _uID, modelName);
          Collection<Thing> _get = GenericThingProvider.this.thingsMap.get(modelName);
          _get.add(newThing);
          GenericThingProvider.this.notifyListenersAboutAddedElement(newThing);
        }
      }
    };
    IterableExtensions.<Thing>forEach(newThings, _function_1);
  }
  
  private final Runnable lazyRetryRunnable = new Runnable() {
    @Override
    public void run() {
      try {
        GenericThingProvider.logger.debug("Starting lazy retry thread");
        while ((!GenericThingProvider.this.queue.isEmpty())) {
          {
            boolean _isEmpty = GenericThingProvider.this.queue.isEmpty();
            boolean _not = (!_isEmpty);
            if (_not) {
              final ArrayList<Thing> newThings = new ArrayList<Thing>();
              final Procedure1<GenericThingProvider.QueueContent> _function = new Procedure1<GenericThingProvider.QueueContent>() {
                @Override
                public void apply(final GenericThingProvider.QueueContent qc) {
                  ThingTypeUID _thingTypeUID = qc.getThingTypeUID();
                  GenericThingProvider.logger.trace("Searching thingHandlerFactory for thingType: {}", _thingTypeUID);
                  ThingHandlerFactory _thingHandlerFactory = qc.getThingHandlerFactory();
                  ThingTypeUID _thingTypeUID_1 = qc.getThingTypeUID();
                  Configuration _configuration = qc.getConfiguration();
                  ThingUID _thingUID = qc.getThingUID();
                  ThingUID _bridgeUID = qc.getBridgeUID();
                  final Thing thing = _thingHandlerFactory.createThing(_thingTypeUID_1, _configuration, _thingUID, _bridgeUID);
                  boolean _notEquals = (!Objects.equal(thing, null));
                  if (_notEquals) {
                    GenericThingProvider.this.queue.remove(qc);
                    String _label = qc.getLabel();
                    thing.setLabel(_label);
                    ThingUID _thingUID_1 = qc.getThingUID();
                    GenericThingProvider.logger.debug("Successfully loaded \'{}\' during retry", _thingUID_1);
                    newThings.add(thing);
                  }
                }
              };
              IterableExtensions.<GenericThingProvider.QueueContent>forEach(GenericThingProvider.this.queue, _function);
              boolean _isEmpty_1 = newThings.isEmpty();
              boolean _not_1 = (!_isEmpty_1);
              if (_not_1) {
                final Procedure1<Thing> _function_1 = new Procedure1<Thing>() {
                  @Override
                  public void apply(final Thing newThing) {
                    Set<String> _keySet = GenericThingProvider.this.thingsMap.keySet();
                    final Function1<String, Boolean> _function = new Function1<String, Boolean>() {
                      @Override
                      public Boolean apply(final String mName) {
                        Collection<Thing> _get = GenericThingProvider.this.thingsMap.get(mName);
                        final Function1<Thing, Boolean> _function = new Function1<Thing, Boolean>() {
                          @Override
                          public Boolean apply(final Thing it) {
                            ThingUID _uID = it.getUID();
                            ThingUID _uID_1 = newThing.getUID();
                            return Boolean.valueOf(Objects.equal(_uID, _uID_1));
                          }
                        };
                        Iterable<Thing> _filter = IterableExtensions.<Thing>filter(_get, _function);
                        boolean _isEmpty = IterableExtensions.isEmpty(_filter);
                        return Boolean.valueOf((!_isEmpty));
                      }
                    };
                    final String modelName = IterableExtensions.<String>findFirst(_keySet, _function);
                    Collection<Thing> _get = GenericThingProvider.this.thingsMap.get(modelName);
                    final Function1<Thing, Boolean> _function_1 = new Function1<Thing, Boolean>() {
                      @Override
                      public Boolean apply(final Thing it) {
                        ThingUID _uID = it.getUID();
                        ThingUID _uID_1 = newThing.getUID();
                        return Boolean.valueOf(Objects.equal(_uID, _uID_1));
                      }
                    };
                    final Thing oldThing = IterableExtensions.<Thing>findFirst(_get, _function_1);
                    boolean _notEquals = (!Objects.equal(oldThing, null));
                    if (_notEquals) {
                      Collection<Thing> _get_1 = GenericThingProvider.this.thingsMap.get(modelName);
                      _get_1.remove(oldThing);
                      Collection<Thing> _get_2 = GenericThingProvider.this.thingsMap.get(modelName);
                      _get_2.add(newThing);
                      ThingUID _uID = newThing.getUID();
                      GenericThingProvider.logger.debug("Refreshing thing \'{}\' after successful retry", _uID);
                      boolean _equals = ThingHelper.equals(oldThing, newThing);
                      boolean _not = (!_equals);
                      if (_not) {
                        GenericThingProvider.this.notifyListenersAboutUpdatedElement(oldThing, newThing);
                      }
                    } else {
                      ThingUID _uID_1 = newThing.getUID();
                      String _format = String.format("Item %s not yet known", _uID_1);
                      throw new IllegalStateException(_format);
                    }
                  }
                };
                IterableExtensions.<Thing>forEach(newThings, _function_1);
              }
            }
            boolean _isEmpty_2 = GenericThingProvider.this.queue.isEmpty();
            boolean _not_2 = (!_isEmpty_2);
            if (_not_2) {
              Thread.sleep(1000);
            }
          }
        }
        GenericThingProvider.logger.debug("Lazy retry thread ran out of work. Good bye.");
      } catch (Throwable _e) {
        throw Exceptions.sneakyThrow(_e);
      }
    }
  };
  
  public void merge(final Object targetThing, final Object source) {
    if (targetThing instanceof Thing
         && source instanceof List) {
      _merge((Thing)targetThing, (List<Channel>)source);
      return;
    } else if (targetThing instanceof Configuration
         && source instanceof Configuration) {
      _merge((Configuration)targetThing, (Configuration)source);
      return;
    } else if (targetThing instanceof Channel
         && source instanceof Channel) {
      _merge((Channel)targetThing, (Channel)source);
      return;
    } else if (targetThing instanceof Thing
         && source instanceof Thing) {
      _merge((Thing)targetThing, (Thing)source);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(targetThing, source).toString());
    }
  }
}
