/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.core.thing.firmware;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.eclipse.smarthome.config.core.validation.ConfigDescriptionValidator;
import org.eclipse.smarthome.config.core.validation.ConfigValidationException;
import org.eclipse.smarthome.core.common.SafeCaller;
import org.eclipse.smarthome.core.common.ThreadPoolManager;
import org.eclipse.smarthome.core.events.Event;
import org.eclipse.smarthome.core.events.EventFilter;
import org.eclipse.smarthome.core.events.EventPublisher;
import org.eclipse.smarthome.core.events.EventSubscriber;
import org.eclipse.smarthome.core.i18n.LocaleProvider;
import org.eclipse.smarthome.core.i18n.TranslationProvider;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingStatus;
import org.eclipse.smarthome.core.thing.ThingUID;
import org.eclipse.smarthome.core.thing.binding.firmware.Firmware;
import org.eclipse.smarthome.core.thing.binding.firmware.FirmwareUID;
import org.eclipse.smarthome.core.thing.binding.firmware.FirmwareUpdateBackgroundTransferHandler;
import org.eclipse.smarthome.core.thing.binding.firmware.FirmwareUpdateHandler;
import org.eclipse.smarthome.core.thing.events.ThingStatusInfoChangedEvent;
import org.eclipse.smarthome.core.thing.firmware.FirmwareEventFactory;
import org.eclipse.smarthome.core.thing.firmware.FirmwareRegistry;
import org.eclipse.smarthome.core.thing.firmware.FirmwareStatus;
import org.eclipse.smarthome.core.thing.firmware.FirmwareStatusInfo;
import org.eclipse.smarthome.core.thing.firmware.ProgressCallbackImpl;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(immediate=true, service={EventSubscriber.class, FirmwareUpdateService.class})
public final class FirmwareUpdateService
implements EventSubscriber {
    private static final String THREAD_POOL_NAME = FirmwareUpdateService.class.getSimpleName();
    private static final Set<String> SUPPORTED_TIME_UNITS = ImmutableSet.of((Object)TimeUnit.SECONDS.name(), (Object)TimeUnit.MINUTES.name(), (Object)TimeUnit.HOURS.name(), (Object)TimeUnit.DAYS.name());
    protected static final String PERIOD_CONFIG_KEY = "period";
    protected static final String DELAY_CONFIG_KEY = "delay";
    protected static final String TIME_UNIT_CONFIG_KEY = "timeUnit";
    private static final String CONFIG_DESC_URI_KEY = "system:firmware-status-info-job";
    private final Logger logger = LoggerFactory.getLogger(FirmwareUpdateService.class);
    private int firmwareStatusInfoJobPeriod = 3600;
    private int firmwareStatusInfoJobDelay = 3600;
    private TimeUnit firmwareStatusInfoJobTimeUnit = TimeUnit.SECONDS;
    private ScheduledFuture<?> firmwareStatusInfoJob;
    protected int timeout = 1800000;
    private final Set<String> subscribedEventTypes = ImmutableSet.of((Object)ThingStatusInfoChangedEvent.TYPE);
    private final Map<ThingUID, FirmwareStatusInfo> firmwareStatusInfoMap = new ConcurrentHashMap<ThingUID, FirmwareStatusInfo>();
    private final Map<ThingUID, ProgressCallbackImpl> progressCallbackMap = new ConcurrentHashMap<ThingUID, ProgressCallbackImpl>();
    private final List<FirmwareUpdateHandler> firmwareUpdateHandlers = new CopyOnWriteArrayList<FirmwareUpdateHandler>();
    private FirmwareRegistry firmwareRegistry;
    private EventPublisher eventPublisher;
    private TranslationProvider i18nProvider;
    private LocaleProvider localeProvider;
    private SafeCaller safeCaller;
    private final Runnable firmwareStatusRunnable = new Runnable(){

        @Override
        public void run() {
            FirmwareUpdateService.this.logger.debug("Running firmware status check.");
            for (FirmwareUpdateHandler firmwareUpdateHandler : FirmwareUpdateService.this.firmwareUpdateHandlers) {
                try {
                    FirmwareUpdateService.this.logger.debug("Executing firmware status check for thing with UID {}.", (Object)firmwareUpdateHandler.getThing().getUID());
                    Firmware latestFirmware = FirmwareUpdateService.this.getLatestSuitableFirmware(firmwareUpdateHandler.getThing());
                    FirmwareStatusInfo newFirmwareStatusInfo = FirmwareUpdateService.this.getFirmwareStatusInfo(firmwareUpdateHandler, latestFirmware);
                    FirmwareUpdateService.this.processFirmwareStatusInfo(firmwareUpdateHandler, newFirmwareStatusInfo, latestFirmware);
                }
                catch (Exception e) {
                    FirmwareUpdateService.this.logger.debug("Exception occurred during firmware status check.", (Throwable)e);
                }
            }
        }
    };

    @Activate
    protected void activate(Map<String, Object> config) {
        this.modified(config);
    }

    @Modified
    protected synchronized void modified(Map<String, Object> config) {
        this.logger.debug("Modifying the configuration of the firmware update service.");
        if (!this.isValid(config)) {
            return;
        }
        this.cancelFirmwareUpdateStatusInfoJob();
        this.firmwareStatusInfoJobPeriod = config.containsKey(PERIOD_CONFIG_KEY) ? (Integer)config.get(PERIOD_CONFIG_KEY) : this.firmwareStatusInfoJobPeriod;
        this.firmwareStatusInfoJobDelay = config.containsKey(DELAY_CONFIG_KEY) ? (Integer)config.get(DELAY_CONFIG_KEY) : this.firmwareStatusInfoJobDelay;
        TimeUnit timeUnit = this.firmwareStatusInfoJobTimeUnit = config.containsKey(TIME_UNIT_CONFIG_KEY) ? TimeUnit.valueOf((String)config.get(TIME_UNIT_CONFIG_KEY)) : this.firmwareStatusInfoJobTimeUnit;
        if (!this.firmwareUpdateHandlers.isEmpty()) {
            this.createFirmwareUpdateStatusInfoJob();
        }
    }

    @Deactivate
    protected void deactivate() {
        this.cancelFirmwareUpdateStatusInfoJob();
        this.firmwareStatusInfoMap.clear();
        this.progressCallbackMap.clear();
    }

    public FirmwareStatusInfo getFirmwareStatusInfo(ThingUID thingUID) {
        Preconditions.checkNotNull((Object)thingUID, (Object)"Thing UID must not be null.");
        FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
        if (firmwareUpdateHandler == null) {
            this.logger.trace("No firmware update handler available for thing with UID {}.", (Object)thingUID);
            return null;
        }
        Firmware latestFirmware = this.getLatestSuitableFirmware(firmwareUpdateHandler.getThing());
        FirmwareStatusInfo firmwareStatusInfo = this.getFirmwareStatusInfo(firmwareUpdateHandler, latestFirmware);
        this.processFirmwareStatusInfo(firmwareUpdateHandler, firmwareStatusInfo, latestFirmware);
        return firmwareStatusInfo;
    }

    public void updateFirmware(ThingUID thingUID, FirmwareUID firmwareUID, Locale locale) {
        Preconditions.checkNotNull((Object)thingUID, (Object)"Thing UID must not be null.");
        Preconditions.checkNotNull((Object)firmwareUID, (Object)"Firmware UID must not be null.");
        FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
        if (firmwareUpdateHandler == null) {
            throw new IllegalArgumentException(String.format("There is no firmware update handler for thing with UID %s.", thingUID));
        }
        Firmware firmware = this.getFirmware(firmwareUID);
        this.validateFirmwareUpdateConditions(firmware, firmwareUpdateHandler);
        Locale loc = locale != null ? locale : this.localeProvider.getLocale();
        ProgressCallbackImpl progressCallback = new ProgressCallbackImpl(firmwareUpdateHandler, this.eventPublisher, this.i18nProvider, thingUID, firmwareUID, loc);
        this.progressCallbackMap.put(thingUID, progressCallback);
        this.logger.debug("Starting firmware update for thing with UID {} and firmware with UID {}", (Object)thingUID, (Object)firmwareUID);
        ((FirmwareUpdateHandler)this.safeCaller.create((Object)firmwareUpdateHandler).withTimeout((long)this.timeout).withAsync().onTimeout(() -> {
            this.logger.error("Timeout occurred for firmware update of thing with UID {} and firmware with UID {}.", (Object)thingUID, (Object)firmwareUID);
            progressCallback.failedInternal("timeout-error");
        }).onException(e -> {
            this.logger.error("Unexpected exception occurred for firmware update of thing with UID {} and firmware with UID {}.", new Object[]{thingUID, firmwareUID, e.getCause()});
            progressCallback.failedInternal("unexpected-handler-error");
        }).build()).updateFirmware(firmware, progressCallback);
    }

    public void cancelFirmwareUpdate(ThingUID thingUID) {
        Preconditions.checkNotNull((Object)thingUID, (Object)"Thing UID must not be null.");
        FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
        if (firmwareUpdateHandler == null) {
            throw new IllegalArgumentException(String.format("There is no firmware update handler for thing with UID %s.", thingUID));
        }
        ProgressCallbackImpl progressCallback = this.getProgressCallback(thingUID);
        this.logger.debug("Cancelling firmware update for thing with UID {}.", (Object)thingUID);
        ((FirmwareUpdateHandler)this.safeCaller.create((Object)firmwareUpdateHandler).withTimeout((long)this.timeout).withAsync().onTimeout(() -> {
            this.logger.error("Timeout occurred while cancelling firmware update of thing with UID {}.", (Object)thingUID);
            progressCallback.failedInternal("timeout-error-during-cancel");
        }).onException(e -> {
            this.logger.error("Unexpected exception occurred while cancelling firmware update of thing with UID {}.", (Object)thingUID, (Object)e.getCause());
            progressCallback.failedInternal("unexpected-handler-error-during-cancel");
        }).build()).cancel();
    }

    public Set<String> getSubscribedEventTypes() {
        return this.subscribedEventTypes;
    }

    public EventFilter getEventFilter() {
        return null;
    }

    public void receive(Event event) {
        if (event instanceof ThingStatusInfoChangedEvent) {
            ThingStatusInfoChangedEvent changedEvent = (ThingStatusInfoChangedEvent)event;
            if (changedEvent.getStatusInfo().getStatus() != ThingStatus.ONLINE) {
                return;
            }
            ThingUID thingUID = changedEvent.getThingUID();
            FirmwareUpdateHandler firmwareUpdateHandler = this.getFirmwareUpdateHandler(thingUID);
            if (firmwareUpdateHandler != null && !this.firmwareStatusInfoMap.containsKey(thingUID)) {
                this.initializeFirmwareStatus(firmwareUpdateHandler);
            }
        }
    }

    private ProgressCallbackImpl getProgressCallback(ThingUID thingUID) {
        if (!this.progressCallbackMap.containsKey(thingUID)) {
            throw new IllegalStateException(String.format("No ProgressCallback available for thing with UID %s.", thingUID));
        }
        return this.progressCallbackMap.get(thingUID);
    }

    private Firmware getLatestSuitableFirmware(Thing thing) {
        return this.firmwareRegistry.getFirmwares(thing.getThingTypeUID()).stream().filter(firmware -> firmware.isSuitableFor(thing)).findFirst().orElse(null);
    }

    private FirmwareStatusInfo getFirmwareStatusInfo(FirmwareUpdateHandler firmwareUpdateHandler, Firmware latestFirmware) {
        String thingFirmwareVersion = this.getThingFirmwareVersion(firmwareUpdateHandler);
        if (latestFirmware == null || thingFirmwareVersion == null) {
            return FirmwareStatusInfo.createUnknownInfo();
        }
        if (latestFirmware.isSuccessorVersion(thingFirmwareVersion)) {
            if (firmwareUpdateHandler.isUpdateExecutable()) {
                return FirmwareStatusInfo.createUpdateExecutableInfo(latestFirmware.getUID());
            }
            return FirmwareStatusInfo.createUpdateAvailableInfo();
        }
        return FirmwareStatusInfo.createUpToDateInfo();
    }

    private synchronized void processFirmwareStatusInfo(FirmwareUpdateHandler firmwareUpdateHandler, FirmwareStatusInfo newFirmwareStatusInfo, Firmware latestFirmware) {
        ThingUID thingUID = firmwareUpdateHandler.getThing().getUID();
        FirmwareStatusInfo previousFirmwareStatusInfo = this.firmwareStatusInfoMap.put(thingUID, newFirmwareStatusInfo);
        if (previousFirmwareStatusInfo == null || !previousFirmwareStatusInfo.equals(newFirmwareStatusInfo)) {
            this.eventPublisher.post((Event)FirmwareEventFactory.createFirmwareStatusInfoEvent(newFirmwareStatusInfo, thingUID));
            if (newFirmwareStatusInfo.getFirmwareStatus() == FirmwareStatus.UPDATE_AVAILABLE && firmwareUpdateHandler instanceof FirmwareUpdateBackgroundTransferHandler && !firmwareUpdateHandler.isUpdateExecutable()) {
                this.transferLatestFirmware((FirmwareUpdateBackgroundTransferHandler)firmwareUpdateHandler, latestFirmware, previousFirmwareStatusInfo);
            }
        }
    }

    private void transferLatestFirmware(final FirmwareUpdateBackgroundTransferHandler fubtHandler, final Firmware latestFirmware, final FirmwareStatusInfo previousFirmwareStatusInfo) {
        FirmwareUpdateService.getPool().submit(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                try {
                    fubtHandler.transferFirmware(latestFirmware);
                }
                catch (Exception e) {
                    FirmwareUpdateService.this.logger.error("Exception occurred during background firmware transfer.", (Throwable)e);
                    2 var2_2 = this;
                    synchronized (var2_2) {
                        FirmwareUpdateService.this.firmwareStatusInfoMap.put(fubtHandler.getThing().getUID(), previousFirmwareStatusInfo);
                    }
                }
            }
        });
    }

    private void validateFirmwareUpdateConditions(Firmware firmware, FirmwareUpdateHandler firmwareUpdateHandler) {
        if (!firmwareUpdateHandler.isUpdateExecutable()) {
            throw new IllegalStateException(String.format("The firmware update of thing with UID %s is not executable.", firmwareUpdateHandler.getThing().getUID()));
        }
        this.validateFirmwareSuitability(firmware, firmwareUpdateHandler);
    }

    private void validateFirmwareSuitability(Firmware firmware, FirmwareUpdateHandler firmwareUpdateHandler) {
        Thing thing = firmwareUpdateHandler.getThing();
        if (!firmware.isSuitableFor(thing)) {
            throw new IllegalArgumentException(String.format("Firmware with UID %s is not suitable for thing with UID %s.", firmware.getUID(), thing.getUID()));
        }
        String firmwareVersion = this.getThingFirmwareVersion(firmwareUpdateHandler);
        if (firmware.getPrerequisiteVersion() != null && !firmware.isPrerequisiteVersion(firmwareVersion)) {
            throw new IllegalArgumentException(String.format("Firmware with UID %s requires at least firmware version %s to get installed. But the current firmware version of the thing with UID %s is %s.", firmware.getUID(), firmware.getPrerequisiteVersion(), thing.getUID(), firmwareVersion));
        }
    }

    private Firmware getFirmware(FirmwareUID firmwareUID) {
        Firmware firmware = this.firmwareRegistry.getFirmware(firmwareUID);
        if (firmware == null) {
            throw new IllegalArgumentException(String.format("Firmware with UID %s was not found.", firmwareUID));
        }
        return firmware;
    }

    private FirmwareUpdateHandler getFirmwareUpdateHandler(ThingUID thingUID) {
        for (FirmwareUpdateHandler firmwareUpdateHandler : this.firmwareUpdateHandlers) {
            if (!thingUID.equals(firmwareUpdateHandler.getThing().getUID())) continue;
            return firmwareUpdateHandler;
        }
        return null;
    }

    private String getThingFirmwareVersion(FirmwareUpdateHandler firmwareUpdateHandler) {
        return firmwareUpdateHandler.getThing().getProperties().get("firmwareVersion");
    }

    private void createFirmwareUpdateStatusInfoJob() {
        if (this.firmwareStatusInfoJob == null || this.firmwareStatusInfoJob.isCancelled()) {
            this.logger.debug("Creating firmware status info job. [delay:{}, period:{}, time unit: {}]", new Object[]{this.firmwareStatusInfoJobDelay, this.firmwareStatusInfoJobPeriod, this.firmwareStatusInfoJobTimeUnit});
            this.firmwareStatusInfoJob = FirmwareUpdateService.getPool().scheduleAtFixedRate(this.firmwareStatusRunnable, this.firmwareStatusInfoJobDelay, this.firmwareStatusInfoJobPeriod, this.firmwareStatusInfoJobTimeUnit);
        }
    }

    private void cancelFirmwareUpdateStatusInfoJob() {
        if (this.firmwareStatusInfoJob != null && !this.firmwareStatusInfoJob.isCancelled()) {
            this.logger.debug("Cancelling firmware status info job.");
            this.firmwareStatusInfoJob.cancel(true);
            this.firmwareStatusInfoJob = null;
        }
    }

    private boolean isValid(Map<String, Object> config) {
        if (!SUPPORTED_TIME_UNITS.contains(config.get(TIME_UNIT_CONFIG_KEY))) {
            this.logger.debug("Given time unit {} is not supported. Will keep current configuration.", config.get(TIME_UNIT_CONFIG_KEY));
            return false;
        }
        try {
            ConfigDescriptionValidator.validate(config, (URI)new URI(CONFIG_DESC_URI_KEY));
        }
        catch (URISyntaxException | ConfigValidationException e) {
            this.logger.debug("Validation of new configuration values failed. Will keep current configuration.", e);
            return false;
        }
        return true;
    }

    private void initializeFirmwareStatus(final FirmwareUpdateHandler firmwareUpdateHandler) {
        FirmwareUpdateService.getPool().submit(new Runnable(){

            @Override
            public void run() {
                ThingUID thingUID = firmwareUpdateHandler.getThing().getUID();
                FirmwareStatusInfo info = FirmwareUpdateService.this.getFirmwareStatusInfo(thingUID);
                FirmwareUpdateService.this.logger.debug("Firmware status {} for thing {} initialized.", (Object)info.getFirmwareStatus(), (Object)thingUID);
                FirmwareUpdateService.this.firmwareStatusInfoMap.put(thingUID, info);
            }
        });
    }

    private static ScheduledExecutorService getPool() {
        return ThreadPoolManager.getScheduledPool((String)THREAD_POOL_NAME);
    }

    protected int getFirmwareStatusInfoJobPeriod() {
        return this.firmwareStatusInfoJobPeriod;
    }

    protected int getFirmwareStatusInfoJobDelay() {
        return this.firmwareStatusInfoJobDelay;
    }

    protected TimeUnit getFirmwareStatusInfoJobTimeUnit() {
        return this.firmwareStatusInfoJobTimeUnit;
    }

    @Reference(cardinality=ReferenceCardinality.MULTIPLE, policy=ReferencePolicy.DYNAMIC)
    protected synchronized void addFirmwareUpdateHandler(FirmwareUpdateHandler firmwareUpdateHandler) {
        if (this.firmwareUpdateHandlers.isEmpty()) {
            this.createFirmwareUpdateStatusInfoJob();
        }
        this.firmwareUpdateHandlers.add(firmwareUpdateHandler);
    }

    protected synchronized void removeFirmwareUpdateHandler(FirmwareUpdateHandler firmwareUpdateHandler) {
        this.firmwareStatusInfoMap.remove(firmwareUpdateHandler.getThing().getUID());
        this.firmwareUpdateHandlers.remove(firmwareUpdateHandler);
        if (this.firmwareUpdateHandlers.isEmpty()) {
            this.cancelFirmwareUpdateStatusInfoJob();
        }
        this.progressCallbackMap.remove(firmwareUpdateHandler.getThing().getUID());
    }

    @Reference
    protected void setFirmwareRegistry(FirmwareRegistry firmwareRegistry) {
        this.firmwareRegistry = firmwareRegistry;
    }

    protected void unsetFirmwareRegistry(FirmwareRegistry firmwareRegistry) {
        this.firmwareRegistry = null;
    }

    @Reference
    protected void setEventPublisher(EventPublisher eventPublisher) {
        this.eventPublisher = eventPublisher;
    }

    protected void unsetEventPublisher(EventPublisher eventPublisher) {
        this.eventPublisher = null;
    }

    @Reference
    protected void setTranslationProvider(TranslationProvider i18nProvider) {
        this.i18nProvider = i18nProvider;
    }

    protected void unsetTranslationProvider(TranslationProvider i18nProvider) {
        this.i18nProvider = null;
    }

    @Reference
    protected void setLocaleProvider(LocaleProvider localeProvider) {
        this.localeProvider = localeProvider;
    }

    protected void unsetLocaleProvider(LocaleProvider localeProvider) {
        this.localeProvider = null;
    }

    @Reference
    protected void setSafeCaller(SafeCaller safeCaller) {
        this.safeCaller = safeCaller;
    }

    protected void unsetSafeCaller(SafeCaller safeCaller) {
        this.safeCaller = null;
    }
}

