/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.binding.astro.handler;

import java.lang.invoke.MethodHandles;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateFormatUtils;
import org.eclipse.smarthome.binding.astro.internal.config.AstroChannelConfig;
import org.eclipse.smarthome.binding.astro.internal.config.AstroThingConfig;
import org.eclipse.smarthome.binding.astro.internal.job.Job;
import org.eclipse.smarthome.binding.astro.internal.job.PositionalJob;
import org.eclipse.smarthome.binding.astro.internal.model.Planet;
import org.eclipse.smarthome.binding.astro.internal.util.PropertyUtils;
import org.eclipse.smarthome.core.scheduler.CronScheduler;
import org.eclipse.smarthome.core.scheduler.ScheduledCompletableFuture;
import org.eclipse.smarthome.core.scheduler.SchedulerRunnable;
import org.eclipse.smarthome.core.thing.Channel;
import org.eclipse.smarthome.core.thing.ChannelUID;
import org.eclipse.smarthome.core.thing.Thing;
import org.eclipse.smarthome.core.thing.ThingStatus;
import org.eclipse.smarthome.core.thing.binding.BaseThingHandler;
import org.eclipse.smarthome.core.thing.type.ChannelKind;
import org.eclipse.smarthome.core.types.Command;
import org.eclipse.smarthome.core.types.RefreshType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AstroThingHandler
extends BaseThingHandler {
    private static final String DAILY_MIDNIGHT = "30 0 0 * * ? *";
    protected final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private final CronScheduler cronScheduler;
    private int linkedPositionalChannels = 0;
    protected AstroThingConfig thingConfig;
    private final Lock monitor = new ReentrantLock();
    private ScheduledCompletableFuture dailyJob;
    private final Set<ScheduledFuture<?>> scheduledFutures = new HashSet();

    public AstroThingHandler(Thing thing, CronScheduler scheduler) {
        super(thing);
        this.cronScheduler = scheduler;
    }

    public void initialize() {
        this.logger.debug("Initializing thing {}", (Object)this.getThing().getUID());
        String thingUid = this.getThing().getUID().toString();
        this.thingConfig = (AstroThingConfig)this.getConfigAs(AstroThingConfig.class);
        this.thingConfig.setThingUid(thingUid);
        boolean validConfig = true;
        if (StringUtils.trimToNull((String)this.thingConfig.getGeolocation()) == null) {
            this.logger.error("Astro parameter geolocation is mandatory and must be configured, disabling thing '{}'", (Object)thingUid);
            validConfig = false;
        } else {
            this.thingConfig.parseGeoLocation();
        }
        if (this.thingConfig.getLatitude() == null || this.thingConfig.getLongitude() == null) {
            this.logger.error("Astro parameters geolocation could not be split into latitude and longitude, disabling thing '{}'", (Object)thingUid);
            validConfig = false;
        }
        if (this.thingConfig.getInterval() == null || this.thingConfig.getInterval() < 1 || this.thingConfig.getInterval() > 86400) {
            this.logger.error("Astro parameter interval must be in the range of 1-86400, disabling thing '{}'", (Object)thingUid);
            validConfig = false;
        }
        if (validConfig) {
            this.logger.debug("{}", (Object)this.thingConfig);
            this.updateStatus(ThingStatus.ONLINE);
            this.restartJobs();
        } else {
            this.updateStatus(ThingStatus.OFFLINE);
        }
        this.logger.debug("Thing {} initialized {}", (Object)this.getThing().getUID(), (Object)this.getThing().getStatus());
    }

    public void dispose() {
        this.logger.debug("Disposing thing {}", (Object)this.getThing().getUID());
        this.stopJobs();
        this.logger.debug("Thing {} disposed", (Object)this.getThing().getUID());
    }

    public void handleCommand(ChannelUID channelUID, Command command) {
        if (RefreshType.REFRESH == command) {
            this.logger.debug("Refreshing {}", (Object)channelUID);
            this.publishChannelIfLinked(channelUID);
        } else {
            this.logger.warn("The Astro-Binding is a read-only binding and can not handle commands");
        }
    }

    public void publishPlanet() {
        this.logger.debug("Publishing planet {} for thing {}", (Object)this.getPlanet().getClass().getSimpleName(), (Object)this.getThing().getUID());
        for (Channel channel : this.getThing().getChannels()) {
            if (channel.getKind() == ChannelKind.TRIGGER) continue;
            this.publishChannelIfLinked(channel.getUID());
        }
    }

    public void publishChannelIfLinked(ChannelUID channelUID) {
        if (this.isLinked(channelUID.getId()) && this.getPlanet() != null) {
            Channel channel = this.getThing().getChannel(channelUID.getId());
            if (channel == null) {
                this.logger.error("Cannot find channel for {}", (Object)channelUID);
                return;
            }
            try {
                AstroChannelConfig config = (AstroChannelConfig)channel.getConfiguration().as(AstroChannelConfig.class);
                this.updateState(channelUID, PropertyUtils.getState(channelUID, config, this.getPlanet()));
            }
            catch (Exception ex) {
                this.logger.error("Can't update state for channel {} : {}", new Object[]{channelUID, ex.getMessage(), ex});
            }
        }
    }

    private void restartJobs() {
        this.logger.debug("Restarting jobs for thing {}", (Object)this.getThing().getUID());
        this.monitor.lock();
        try {
            this.stopJobs();
            if (this.getThing().getStatus() == ThingStatus.ONLINE) {
                String thingUID = this.getThing().getUID().toString();
                if (this.cronScheduler == null) {
                    this.logger.warn("Thread Pool Executor is not available");
                    return;
                }
                Job runnable = this.getDailyJob();
                this.dailyJob = this.cronScheduler.schedule((SchedulerRunnable)runnable, DAILY_MIDNIGHT);
                this.logger.debug("Scheduled {} at midnight", (Object)this.dailyJob);
                runnable.run();
                if (this.isPositionalChannelLinked()) {
                    PositionalJob positionalJob = new PositionalJob(thingUID);
                    ScheduledFuture<?> future = this.scheduler.scheduleAtFixedRate(positionalJob, 0L, this.thingConfig.getInterval().intValue(), TimeUnit.SECONDS);
                    this.scheduledFutures.add(future);
                    this.logger.info("Scheduled {} every {} seconds", (Object)positionalJob, (Object)this.thingConfig.getInterval());
                }
            }
        }
        finally {
            this.monitor.unlock();
        }
    }

    private void stopJobs() {
        this.logger.debug("Stopping scheduled jobs for thing {}", (Object)this.getThing().getUID());
        this.monitor.lock();
        try {
            try {
                if (this.cronScheduler != null) {
                    if (this.dailyJob != null) {
                        this.dailyJob.cancel(true);
                    }
                    this.dailyJob = null;
                }
                for (ScheduledFuture<?> future : this.scheduledFutures) {
                    if (future.isDone()) continue;
                    future.cancel(true);
                }
                this.scheduledFutures.clear();
            }
            catch (Exception ex) {
                this.logger.error("{}", (Object)ex.getMessage(), (Object)ex);
                this.monitor.unlock();
            }
        }
        finally {
            this.monitor.unlock();
        }
    }

    public void channelLinked(ChannelUID channelUID) {
        this.linkedChannelChange(channelUID, 1);
        this.publishChannelIfLinked(channelUID);
    }

    public void channelUnlinked(ChannelUID channelUID) {
        this.linkedChannelChange(channelUID, -1);
    }

    private void linkedChannelChange(ChannelUID channelUID, int step) {
        if (ArrayUtils.contains((Object[])this.getPositionalChannelIds(), (Object)channelUID.getId())) {
            int oldValue = this.linkedPositionalChannels;
            this.linkedPositionalChannels += step;
            if (oldValue == 0 && this.linkedPositionalChannels > 0 || oldValue > 0 && this.linkedPositionalChannels == 0) {
                this.restartJobs();
            }
        }
    }

    private boolean isPositionalChannelLinked() {
        for (Channel channel : this.getThing().getChannels()) {
            if (!ArrayUtils.contains((Object[])this.getPositionalChannelIds(), (Object)channel.getUID().getId()) || !this.isLinked(channel.getUID().getId())) continue;
            return true;
        }
        return false;
    }

    public void triggerEvent(String channelId, String event) {
        Channel channel = this.getThing().getChannel(channelId);
        if (channel == null) {
            this.logger.warn("Event {} in thing {} does not exist, please recreate the thing", (Object)event, (Object)this.getThing().getUID());
            return;
        }
        this.triggerChannel(channel.getUID(), event);
    }

    public void schedule(Job job, Calendar eventAt) {
        long sleepTime;
        this.monitor.lock();
        try {
            this.tidyScheduledFutures();
            sleepTime = eventAt.getTimeInMillis() - new Date().getTime();
            ScheduledFuture<?> future = this.scheduler.schedule(job, sleepTime, TimeUnit.MILLISECONDS);
            this.scheduledFutures.add(future);
        }
        finally {
            this.monitor.unlock();
        }
        if (this.logger.isDebugEnabled()) {
            String formattedDate = DateFormatUtils.ISO_DATETIME_FORMAT.format(eventAt);
            this.logger.debug("Scheduled {} in {}ms (at {})", new Object[]{job, sleepTime, formattedDate});
        }
    }

    private void tidyScheduledFutures() {
        Iterator<ScheduledFuture<?>> iterator = this.scheduledFutures.iterator();
        while (iterator.hasNext()) {
            ScheduledFuture<?> future = iterator.next();
            if (!future.isDone()) continue;
            this.logger.trace("Tidying up done future {}", future);
            iterator.remove();
        }
    }

    public abstract void publishDailyInfo();

    public abstract void publishPositionalInfo();

    public abstract Planet getPlanet();

    protected abstract String[] getPositionalChannelIds();

    protected abstract Job getDailyJob();
}

