/**
 * Copyright (c) 2014,2019 Contributors to the Eclipse Foundation
 * 
 * See the NOTICE file(s) distributed with this work for additional
 * information regarding copyright ownership.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.smarthome.model.validation;

import com.google.common.base.Objects;
import java.math.BigDecimal;
import org.eclipse.emf.common.util.EList;
import org.eclipse.smarthome.model.sitemap.Frame;
import org.eclipse.smarthome.model.sitemap.LinkableWidget;
import org.eclipse.smarthome.model.sitemap.Setpoint;
import org.eclipse.smarthome.model.sitemap.Sitemap;
import org.eclipse.smarthome.model.sitemap.SitemapPackage;
import org.eclipse.smarthome.model.sitemap.Widget;
import org.eclipse.smarthome.model.validation.AbstractSitemapValidator;
import org.eclipse.xtext.validation.Check;

/**
 * Custom validation rules.
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#validation
 */
@SuppressWarnings("all")
public class SitemapValidator extends AbstractSitemapValidator {
  @Check
  public void checkFramesInFrame(final Frame frame) {
    EList<Widget> _children = frame.getChildren();
    for (final Widget w : _children) {
      if ((w instanceof Frame)) {
        this.error("Frames must not contain other frames", 
          SitemapPackage.Literals.FRAME.getEStructuralFeature(SitemapPackage.FRAME__CHILDREN));
        return;
      }
    }
  }
  
  @Check
  public void checkFramesInWidgetList(final Sitemap sitemap) {
    boolean containsFrames = false;
    boolean containsOtherWidgets = false;
    EList<Widget> _children = sitemap.getChildren();
    for (final Widget w : _children) {
      {
        if ((w instanceof Frame)) {
          containsFrames = true;
        } else {
          containsOtherWidgets = true;
        }
        if ((containsFrames && containsOtherWidgets)) {
          this.error("Sitemap should contain either only frames or none at all", 
            SitemapPackage.Literals.SITEMAP.getEStructuralFeature(SitemapPackage.SITEMAP__NAME));
          return;
        }
      }
    }
  }
  
  @Check
  public void checkFramesInWidgetList(final LinkableWidget widget) {
    if ((widget instanceof Frame)) {
      return;
    }
    boolean containsFrames = false;
    boolean containsOtherWidgets = false;
    EList<Widget> _children = widget.getChildren();
    for (final Widget w : _children) {
      {
        if ((w instanceof Frame)) {
          containsFrames = true;
        } else {
          containsOtherWidgets = true;
        }
        if ((containsFrames && containsOtherWidgets)) {
          this.error("Linkable widget should contain either only frames or none at all", 
            SitemapPackage.Literals.FRAME.getEStructuralFeature(SitemapPackage.LINKABLE_WIDGET__CHILDREN));
          return;
        }
      }
    }
  }
  
  @Check
  public void checkSetpoints(final Setpoint sp) {
    BigDecimal _step = sp.getStep();
    boolean _equals = Objects.equal(BigDecimal.ZERO, _step);
    if (_equals) {
      String _item = sp.getItem();
      String _plus = ("Setpoint on item \'" + _item);
      String _plus_1 = (_plus + "\' has step size of 0");
      this.error(_plus_1, 
        SitemapPackage.Literals.SETPOINT.getEStructuralFeature(SitemapPackage.SETPOINT__STEP));
    }
    if (((sp.getStep() != null) && (sp.getStep().compareTo(BigDecimal.ZERO) < 0))) {
      String _item_1 = sp.getItem();
      String _plus_2 = ("Setpoint on item \'" + _item_1);
      String _plus_3 = (_plus_2 + "\' has negative step size");
      this.error(_plus_3, 
        SitemapPackage.Literals.SETPOINT.getEStructuralFeature(SitemapPackage.SETPOINT__STEP));
    }
    if ((((sp.getMinValue() != null) && (sp.getMaxValue() != null)) && (sp.getMinValue().compareTo(sp.getMaxValue()) > 0))) {
      String _item_2 = sp.getItem();
      String _plus_4 = ("Setpoint on item \'" + _item_2);
      String _plus_5 = (_plus_4 + "\' has larger minValue than maxValue");
      this.error(_plus_5, 
        SitemapPackage.Literals.SETPOINT.getEStructuralFeature(SitemapPackage.SETPOINT__MIN_VALUE));
    }
  }
}
