/**
 * Copyright (c) 2017 Technische Hochschule Ulm, Servicerobotics Ulm, Germany
 * headed by Prof. Dr. Christian Schlegel
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Alex Lotz, Dennis Stampfer, Matthias Lutz
 */
package org.eclipse.smartmdsd.xtext.system.componentArchitecture.ui.contentassist;

import com.google.inject.Inject;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.smartmdsd.ecore.component.componentDefinition.ComponentDefinitionPackage;
import org.eclipse.smartmdsd.ecore.system.componentArchitecture.ComponentInstance;
import org.eclipse.smartmdsd.ecore.system.systemParameter.ComponentParameterInstance;
import org.eclipse.smartmdsd.ecore.system.systemParameter.ParameterStructInstance;
import org.eclipse.smartmdsd.ecore.system.systemParameter.SystemParameterPackage;
import org.eclipse.smartmdsd.xtext.indexer.XtextResourceIndex;
import org.eclipse.xtext.Assignment;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.ui.editor.contentassist.ContentAssistContext;
import org.eclipse.xtext.ui.editor.contentassist.ICompletionProposalAcceptor;

/**
 * See https://www.eclipse.org/Xtext/documentation/304_ide_concepts.html#content-assist
 * on how to customize the content assistant.
 */
@SuppressWarnings("all")
public class ComponentArchitectureProposalProvider extends AbstractComponentArchitectureProposalProvider {
  @Inject
  private XtextResourceIndex index;
  
  @Inject
  private IQualifiedNameProvider fqn_provider;
  
  @Override
  public void completeSystemComponentArchitecture_Name(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    super.completeSystemComponentArchitecture_Name(model, assignment, context, acceptor);
    final URI uri = model.eResource().getURI();
    boolean _isPlatform = uri.isPlatform();
    if (_isPlatform) {
      final String projectName = uri.segment(1);
      acceptor.accept(this.createCompletionProposal(projectName, (projectName + " - ProjectName"), this.getImage(model), context));
    }
  }
  
  @Override
  public void completeComponentInstance_Name(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    super.completeComponentInstance_Name(model, assignment, context, acceptor);
    final Iterable<IEObjectDescription> components = this.index.getVisibleEObjectDescriptions(model, ComponentDefinitionPackage.eINSTANCE.getComponentDefinition());
    final Consumer<IEObjectDescription> _function = (IEObjectDescription component) -> {
      final String name = component.getName().toString();
      final String text = (((name + " instantiates ") + name) + " {\n\n\t}");
      acceptor.accept(this.createCompletionProposal(text, (name + " - ComponentDefinition"), this.getImage(model), context));
    };
    components.forEach(_function);
  }
  
  @Override
  public void completeParameterStructInstance_Parameter(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    if ((model instanceof ParameterStructInstance)) {
      final EObject compInstance = ((ParameterStructInstance)model).eContainer();
      if ((compInstance instanceof ComponentInstance)) {
        final Iterable<IEObjectDescription> componentParams = this.index.getVisibleEObjectDescriptions(model, SystemParameterPackage.eINSTANCE.getComponentParameterInstance());
        for (final IEObjectDescription componentParam : componentParams) {
          {
            final EObject componentParamObj = componentParam.getEObjectOrProxy();
            boolean _eIsProxy = componentParamObj.eIsProxy();
            if (_eIsProxy) {
              final EObject object = EcoreUtil.resolve(componentParamObj, compInstance);
              if ((object instanceof ComponentParameterInstance)) {
                boolean _equals = ((ComponentParameterInstance)object).getComponentInstance().equals(compInstance);
                if (_equals) {
                  final String text = this.fqn_provider.getFullyQualifiedName(object).toString();
                  acceptor.accept(this.createCompletionProposal(text, (text + " - ComponentParameterInstance"), this.getImage(model), context));
                }
              }
            }
          }
        }
      }
    }
  }
}
