/*******************************************************************************
 * Copyright (c) 2004, 2010 John Krasnay and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     John Krasnay - initial API and implementation
 *     Florian Thienel - generics inferred
 *******************************************************************************/
package org.eclipse.vex.core.internal.core;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.EventObject;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A collection of listener objects. The main point of this class is the fireEvent method, which takes care of the
 * tedium of iterating over the collection and catching exceptions generated by listeners.
 * 
 * @param <L>
 *            the type of listeners maintained by this list
 * @param <E>
 *            the type of events to fire
 */
public class ListenerList<L, E extends EventObject> {

	private final Class<L> listenerClass;
	private final List<L> listeners = new ArrayList<L>();

	/** Mapping: method name => method object */
	private final Map<String, Method> methods = new HashMap<String, Method>();

	/**
	 * Class constructor.
	 * 
	 * @param listenerClass
	 *            Class of the listener interface.
	 */
	public ListenerList(final Class<L> listenerClass) {
		this.listenerClass = listenerClass;
	}

	/**
	 * Adds a listener to the list. Rejects listeners that are not subclasses of the listener class passed to the
	 * constructor.
	 * 
	 * @param listener
	 *            Listener to be added.
	 */
	public void add(final L listener) {
		this.listeners.add(listener);
	}

	/**
	 * Removes a listener from the list.
	 * 
	 * @param listener
	 *            Listener to remove.
	 */
	public void remove(final L listener) {
		this.listeners.remove(listener);
	}

	/**
	 * Called from fireEvent whenever a called listener method throws an exception, or if there is a problem looking up
	 * the listener method by reflection. By default, simply prints the stack trace to stdout. Clients may override this
	 * method to provide a more suitable implementation.
	 * 
	 * @param e
	 *            Exception thrown by the listener method.
	 */
	public void handleException(final Exception e) {
		e.printStackTrace();
	}

	/**
	 * Calls the given method on each registered listener. Any exception thrown from one of the called methods is passed
	 * to handleException, as is any introspection error, e.g. if the given method doesn't exist.
	 * 
	 * @param methodName
	 *            Listener method to call.
	 * @param event
	 *            Event to be passed to each call.
	 */
	public void fireEvent(final String methodName, final E event) {
		final Method method = getMethod(methodName, event.getClass());
		if (method == null) {
			return; // Exception handling already done by getMethod
		}

		for (final L listener : listeners) {
			try {
				method.invoke(listener, event);
			} catch (final IllegalArgumentException e) {
				handleException(e);
			} catch (final IllegalAccessException e) {
				handleException(e);
			} catch (final InvocationTargetException e) {
				handleException(e);
			}
		}
	}

	private Method getMethod(final String methodName, final Class<?> eventClass) {
		if (this.methods.containsKey(methodName)) {
			return this.methods.get(methodName);
		}

		try {
			final Method method = listenerClass.getMethod(methodName, eventClass);
			this.methods.put(methodName, method);
			return method;
		} catch (final SecurityException e) {
			handleException(e);
		} catch (final NoSuchMethodException e) {
			handleException(e);
		}
		return null;
	}

}
