﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudtrail/CloudTrailRequest.h>
#include <aws/cloudtrail/CloudTrail_EXPORTS.h>
#include <aws/cloudtrail/model/QueryStatus.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudTrail {
namespace Model {

/**
 */
class ListQueriesRequest : public CloudTrailRequest {
 public:
  AWS_CLOUDTRAIL_API ListQueriesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListQueries"; }

  AWS_CLOUDTRAIL_API Aws::String SerializePayload() const override;

  AWS_CLOUDTRAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ARN (or the ID suffix of the ARN) of an event data store on which queries
   * were run.</p>
   */
  inline const Aws::String& GetEventDataStore() const { return m_eventDataStore; }
  inline bool EventDataStoreHasBeenSet() const { return m_eventDataStoreHasBeenSet; }
  template <typename EventDataStoreT = Aws::String>
  void SetEventDataStore(EventDataStoreT&& value) {
    m_eventDataStoreHasBeenSet = true;
    m_eventDataStore = std::forward<EventDataStoreT>(value);
  }
  template <typename EventDataStoreT = Aws::String>
  ListQueriesRequest& WithEventDataStore(EventDataStoreT&& value) {
    SetEventDataStore(std::forward<EventDataStoreT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token you can use to get the next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListQueriesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of queries to show on a page.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListQueriesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use with <code>EndTime</code> to bound a <code>ListQueries</code> request,
   * and limit its results to only those queries run within a specified time
   * period.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListQueriesRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use with <code>StartTime</code> to bound a <code>ListQueries</code> request,
   * and limit its results to only those queries run within a specified time
   * period.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListQueriesRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of queries that you want to return in results. Valid values for
   * <code>QueryStatus</code> include <code>QUEUED</code>, <code>RUNNING</code>,
   * <code>FINISHED</code>, <code>FAILED</code>, <code>TIMED_OUT</code>, or
   * <code>CANCELLED</code>.</p>
   */
  inline QueryStatus GetQueryStatus() const { return m_queryStatus; }
  inline bool QueryStatusHasBeenSet() const { return m_queryStatusHasBeenSet; }
  inline void SetQueryStatus(QueryStatus value) {
    m_queryStatusHasBeenSet = true;
    m_queryStatus = value;
  }
  inline ListQueriesRequest& WithQueryStatus(QueryStatus value) {
    SetQueryStatus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_eventDataStore;
  bool m_eventDataStoreHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  QueryStatus m_queryStatus{QueryStatus::NOT_SET};
  bool m_queryStatusHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudTrail
}  // namespace Aws
