﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/connect/ConnectRequest.h>
#include <aws/connect/Connect_EXPORTS.h>
#include <aws/connect/model/CreatedByInfo.h>
#include <aws/connect/model/FileUseCaseType.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace Connect {
namespace Model {

/**
 */
class StartAttachedFileUploadRequest : public ConnectRequest {
 public:
  AWS_CONNECT_API StartAttachedFileUploadRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartAttachedFileUpload"; }

  AWS_CONNECT_API Aws::String SerializePayload() const override;

  AWS_CONNECT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If not provided, the Amazon Web Services SDK
   * populates this field. For more information about idempotency, see <a
   * href="https://aws.amazon.com/builders-library/making-retries-safe-with-idempotent-APIs/">Making
   * retries safe with idempotent APIs</a>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartAttachedFileUploadRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the Amazon Connect instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  StartAttachedFileUploadRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A case-sensitive name of the attached file being uploaded.</p>
   */
  inline const Aws::String& GetFileName() const { return m_fileName; }
  inline bool FileNameHasBeenSet() const { return m_fileNameHasBeenSet; }
  template <typename FileNameT = Aws::String>
  void SetFileName(FileNameT&& value) {
    m_fileNameHasBeenSet = true;
    m_fileName = std::forward<FileNameT>(value);
  }
  template <typename FileNameT = Aws::String>
  StartAttachedFileUploadRequest& WithFileName(FileNameT&& value) {
    SetFileName(std::forward<FileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The size of the attached file in bytes.</p>
   */
  inline long long GetFileSizeInBytes() const { return m_fileSizeInBytes; }
  inline bool FileSizeInBytesHasBeenSet() const { return m_fileSizeInBytesHasBeenSet; }
  inline void SetFileSizeInBytes(long long value) {
    m_fileSizeInBytesHasBeenSet = true;
    m_fileSizeInBytes = value;
  }
  inline StartAttachedFileUploadRequest& WithFileSizeInBytes(long long value) {
    SetFileSizeInBytes(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional override for the expiry of the pre-signed S3 URL in seconds. The
   * default value is 300.</p>
   */
  inline int GetUrlExpiryInSeconds() const { return m_urlExpiryInSeconds; }
  inline bool UrlExpiryInSecondsHasBeenSet() const { return m_urlExpiryInSecondsHasBeenSet; }
  inline void SetUrlExpiryInSeconds(int value) {
    m_urlExpiryInSecondsHasBeenSet = true;
    m_urlExpiryInSeconds = value;
  }
  inline StartAttachedFileUploadRequest& WithUrlExpiryInSeconds(int value) {
    SetUrlExpiryInSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The use case for the file.</p>  <p> Only <code>ATTACHMENTS</code>
   * are supported.</p>
   */
  inline FileUseCaseType GetFileUseCaseType() const { return m_fileUseCaseType; }
  inline bool FileUseCaseTypeHasBeenSet() const { return m_fileUseCaseTypeHasBeenSet; }
  inline void SetFileUseCaseType(FileUseCaseType value) {
    m_fileUseCaseTypeHasBeenSet = true;
    m_fileUseCaseType = value;
  }
  inline StartAttachedFileUploadRequest& WithFileUseCaseType(FileUseCaseType value) {
    SetFileUseCaseType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The resource to which the attached file is (being) uploaded to. The supported
   * resources are <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/cases.html">Cases</a>
   * and <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/setup-email-channel.html">Email</a>.</p>
   *  <p>This value must be a valid ARN.</p>
   */
  inline const Aws::String& GetAssociatedResourceArn() const { return m_associatedResourceArn; }
  inline bool AssociatedResourceArnHasBeenSet() const { return m_associatedResourceArnHasBeenSet; }
  template <typename AssociatedResourceArnT = Aws::String>
  void SetAssociatedResourceArn(AssociatedResourceArnT&& value) {
    m_associatedResourceArnHasBeenSet = true;
    m_associatedResourceArn = std::forward<AssociatedResourceArnT>(value);
  }
  template <typename AssociatedResourceArnT = Aws::String>
  StartAttachedFileUploadRequest& WithAssociatedResourceArn(AssociatedResourceArnT&& value) {
    SetAssociatedResourceArn(std::forward<AssociatedResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Represents the identity that created the file.</p>
   */
  inline const CreatedByInfo& GetCreatedBy() const { return m_createdBy; }
  inline bool CreatedByHasBeenSet() const { return m_createdByHasBeenSet; }
  template <typename CreatedByT = CreatedByInfo>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = CreatedByInfo>
  StartAttachedFileUploadRequest& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource. For
   * example, <code>{ "Tags": {"key1":"value1", "key2":"value2"} }</code>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  StartAttachedFileUploadRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  StartAttachedFileUploadRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_instanceId;
  bool m_instanceIdHasBeenSet = false;

  Aws::String m_fileName;
  bool m_fileNameHasBeenSet = false;

  long long m_fileSizeInBytes{0};
  bool m_fileSizeInBytesHasBeenSet = false;

  int m_urlExpiryInSeconds{0};
  bool m_urlExpiryInSecondsHasBeenSet = false;

  FileUseCaseType m_fileUseCaseType{FileUseCaseType::NOT_SET};
  bool m_fileUseCaseTypeHasBeenSet = false;

  Aws::String m_associatedResourceArn;
  bool m_associatedResourceArnHasBeenSet = false;

  CreatedByInfo m_createdBy;
  bool m_createdByHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Connect
}  // namespace Aws
