/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Text;

namespace LocaleInpector
{

internal sealed class Strings
{

private Strings()
{
}

public static int CountSubstrings(string text, string substring)
{
    int qty = text.Replace(substring, string.Empty).Length;
    return (text.Length - qty) / substring.Length;
}

public static string FindEol(string source)
{
    int unknown = CountSubstrings(source, "\r\r\n");            // 0d 0d 0a    CR CR LF
    int dos     = CountSubstrings(source, "\r\n") - unknown;    // 0d 0a       CR LF
    int mac     = CountSubstrings(source, "\r")   - dos;        // 0d          CR
    int unix    = CountSubstrings(source, "\n")   - dos;        // 0a          LF

    int maxQty = 0;
    string eol = string.Empty;

    if (unknown > maxQty) {
        maxQty = unknown;
        eol = "\r\r\n";
    }
    if (dos > maxQty) {
        maxQty = dos;
        eol = "\r\n";
    }
    if (mac > maxQty) {
        maxQty = mac;
        eol = "\r";
    }
    if (unix > maxQty) {
        maxQty = unix;
        eol = "\n";
    }

    return eol;
}

public static List<string> SplitLines(string source)
{
    string[] lines;
    string eol = FindEol(source);
    switch (eol) {
        case "":
            lines = new string[] { source };
            break;
        case "\n":
            lines = source.Split('\n');
            break;
        case "\r":
            lines = source.Split('\r');
            break;
        default:
            Regex regex = new Regex(eol, RegexOptions.Singleline);
            lines = regex.Split(source);
            break;
    }

    List<string> all = new List<string>();
    all.AddRange(lines);
    return all;
}

public static string RemoveWhite(string txt)
{
    return System.Text.RegularExpressions.Regex.Replace(txt, @"\s+", " ");
}

private static readonly char[][] m_chars = new char[][] {
    new char[] { 'a', '', '', '', '', '' },
    new char[] { 'o', '', '', '', '', '' },
    new char[] { 'e', '', '', '', '' },
    new char[] { 'i', '', '', '', '' },
    new char[] { 'u', '', '', '', '' },
    new char[] { 'y', '',           '' },
    new char[] { 'c', '' },
    new char[] { 'n', '' }
};

public static string ReplaceHighChars(string txt)
{
    System.Text.StringBuilder buffer = new System.Text.StringBuilder(txt);

    char c;
    char lowerChar;
    char[] ch;

    for (int idx = 0; idx < buffer.Length; idx++) {
        c = buffer[idx];
        if (char.IsLetter(c) == false) {
            continue;
        }

        lowerChar = char.ToLower(c);

        for (int i = 0; i < m_chars.Length; i++) {

            ch = m_chars[i];
            for (int j = 1; j < ch.Length; j++) {
                if (ch[j] == lowerChar) {
                    buffer[idx] = char.IsLower(c) ? ch[0] : char.ToUpper(ch[0]);
                    continue;
                }
            }
        }
    }
    return buffer.ToString();
}

}//class
}//ns