/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System.IO;
using System.Collections.Generic;
using System.Text.RegularExpressions;

namespace LocaleInpector
{

internal abstract class NameValueFile
{


protected bool m_isTranslated = false;

protected System.Text.Encoding m_encoding;
public    string m_path;
protected string m_content;
protected string m_eol;

protected NameValueFile(string path)
{
    m_path = path;
    m_encoding = FindEncoding();
    this.LoadContent();
}

protected virtual System.Text.Encoding FindEncoding()
{
    return BPIO.UTF8;
}

public static NameValueFile Load(string file)
{
    if (BPIO.LoadCache.ContainsKey(file)) {
        return BPIO.LoadCache[file];
    }
    NameValueFile obj = LoadCore(file);
    BPIO.LoadCache[file] = obj;
    return obj;

}

public static NameValueFile LoadTranslated(string file)
{
    NameValueFile dic = Load(file);
    if (dic != null) {
        dic.m_isTranslated = true;
    }
    return dic;
}



private static NameValueFile LoadCore(string file)
{
    if (file.EndsWith(".dtd")) {
        return new DtdFile(file);
    }
    if (file.EndsWith(".ent")) {
        return new DtdFile(file);
    }
    if (file.EndsWith(".properties")) {
        return new PropertiesFile(file);
    }
    if (file.EndsWith(".ini")) {
        return new IniFile(file);
    }
    if (file.EndsWith(".it")) {
        return new IniFile(file);
    }
    if (file.EndsWith(".inc")) {
        NameValueFile dic;
        dic = new IniFile(file);
        List<string> sec = (dic as IniFile).LoadSections();
        if (sec.Count > 0) {
            return dic;
        }

        dic = new PropertiesFile(file);
        Dictionary<string, string> ht = dic.LoadHashtable();
        if (ht.Count > 0) {
            return dic;
        }

        dic = new DirectiveFile(file);
        ht = dic.LoadHashtable();
        if (ht.Count > 0) {
            return dic;
        }
    }
    if (file.EndsWith(".po")) {
        return new PoFile(file);
    }
    if (file.EndsWith(".pot")) {
        return new PoFile(file);
    }

    // undefined
    return null;
}

public string Content
{
    get {
        return m_content;
    }
}

public string Eol
{
    get {
        return m_eol;
    }
}

public string Path
{
    get {
        return m_path;
    }
}

protected string LoadText()
{
    string txt = BPIO.LoadText(m_path, m_encoding);
    m_eol = Strings.FindEol(txt);
    return txt;
}

public NameValueFile LoadFrom(string path)
{
    NameValueFile clone = this.MemberwiseClone() as NameValueFile; // translated+encoding
    clone.m_path = path;
    clone.LoadContent();
    return clone;
}

// this ==> merged file
public bool SaveTo(string txt)
{
    bool save = true;
    if (this.Exists == false) {
        this.SaveCore(txt);
        return true;
    }

    if (txt == this.Content) {
        if (BPIO.HasBom(m_path) == false) {
            save = false;
        }
    } else {
        //Console.WriteLine(nameNew + "!= " + txt.Length + " *** " + txtCurrent.Length);
        //Console.WriteLine("[new[[" + txt + "]]]");
        //Console.WriteLine("[current[[" + txtCurrent + "]]]");
        //nameNew += ".txt";
    }

    if (save) {
        this.SaveCore(txt);
        //System.Console.WriteLine("SAVING: " + m_path);
        return true;
    }
    return false;
}

private void SaveCore(string txt)
{
    BPIO.Save(txt, m_path, m_encoding);
}

// mailing.zipmailinglistheader-0.1.xpi|chrome/messenger-mailinglistheader.jar|locale/en/messenger-mailinglistheader/messenger-mailinglistheader.dtd
// mailing.zipmailinglistheader-0.1.xpi
//     chrome/messenger-mailinglistheader.jar
//         locale/en/messenger-mailinglistheader/messenger-mailinglistheader.dtd
public void AddToArchive(string txt, ICSharpCode.SharpZipLib.Zip.ZipOutputStream zipStream, string path)
{
    List<string> arr2 = new List<string>(path.Split('/'));
    if (arr2[0] as string == "locale") {
        arr2.RemoveAt(0);
        if (arr2.Count > 1) {
            arr2.RemoveAt(0);
        }
        path = string.Join("/", arr2.ToArray());
    }

    ICSharpCode.SharpZipLib.Zip.ZipEntry entry = new ICSharpCode.SharpZipLib.Zip.ZipEntry(path);

    MemoryStream streamTxt = new MemoryStream();
    byte[] dataTxt;
    using (StreamWriter sw = new StreamWriter(streamTxt, m_encoding)) {
        sw.Write(txt);
    }
    dataTxt = streamTxt.ToArray();
    entry.Size = dataTxt.Length;
    ICSharpCode.SharpZipLib.Checksums.Crc32 crc = new ICSharpCode.SharpZipLib.Checksums.Crc32();
    crc.Update(dataTxt);
    entry.Crc = crc.Value;
    zipStream.PutNextEntry(entry);
    zipStream.Write(dataTxt, 0, dataTxt.Length);
    streamTxt.Close();
}

public abstract void LoadContent();

public abstract Dictionary<string, string> LoadHashtableLogged();

public abstract Dictionary<string, string> LoadHashtable();

public virtual Dictionary<string, string> LoadHashtableDecoded()
{
    return this.LoadHashtable();
}

public virtual string Sanitize(string txt)
{
    return txt;
}

public abstract Merge GetMerge();

public bool Exists
{
    get {
        return BPIO.IsFile(m_path);
    }
}

// not used by PoFile
protected static void AddToHashtable(Dictionary<string, string> table, string key, string val, string file)
{
    if (table.ContainsKey(key)) {
        if (file != null) {
            Output.WriteWarning("Further tests will be unreliable. Key redeclarated at " + file + " => " + key + ".");
        }
        return;
    }
    table[key] = val;
}

}//class
}//ns

