/** Mozile Script Resource Test Script
 * @fileoverview This file contains JSUnit test functions for testing the MozileScriptResource class. 
 * Requires MozileResourcesUtils.js
 * 
 * @link http://mozile.mozdev.org 
 * @author James A. Overton <james@overton.ca>
 * @version 0.7.2
 */


var category = "script";
var id = "testScript";
var source = "http://source.com/";
var defaultType = "application/x-javascript";
var type = "text/javascript";
var defaultNamespace = XHTMLNS;
var namespace = "http://namespace.com/";
var content = "Test content.";


/** Test - Constructor
 * Make sure that the MozileScriptResource constructor is working. Use the default namespace.
 */
function testConstructor() {
	var resource = new MozileScriptResource(id, source);
	checkResource(resource);
	checkScriptResource(resource);
}


/** Test - Constructor
 * Make sure that the MozileScriptResource constructor is working. Define a type and a namespace.
 */
function testConstructorNS() {
	var resource = new MozileScriptResource(id, source, type, namespace);
	checkResourceNS(resource);
	checkScriptResource(resource);
}


/** Test - Create Element
 * Create an element using the default namespace.
 */
function testCreateElement() {
	var resource = new MozileScriptResource(id, source);

		// getElement creates a new element	
	var element1 = resource.getElement();
	checkElement(element1);
	checkScriptElement(element1);

		// createElement creates a new element to replace the old one
	var element2 = resource.createElement();
	checkElement(element2);
	checkScriptElement(element2);

		// getElement should get element2
	var element3 = resource.getElement();
	checkElement(element3);
	checkScriptElement(element3);
	
	compareElements(element1, element2, element3);
}


/** Test - Create Element
 * Create an element, specifying a type and a namespace.
 */
function testCreateElementNS() {
	var resource = new MozileScriptResource(id, source, type, namespace);

		// getElement creates a new element	
	var element1 = resource.getElement();
	checkElementNS(element1);
	checkScriptElement(element1);

		// createElement creates a new element to replace the old one
	var element2 = resource.createElement();
	checkElementNS(element2);
	checkScriptElement(element2);

		// getElement should get element2
	var element3 = resource.getElement();
	checkElementNS(element3);
	checkScriptElement(element3);
	
	compareElements(element1, element2, element3);
}


/** Test - Content
 * Test the content methods.
 */
function testContent() {
	var resource = new MozileScriptResource(id, source);
	checkResource(resource);
	
	assertNotNull("The content should be not be null.", resource.getContent());
	assertUndefined("The content should be undefined.", resource.getContent());
	
	var textNode = document.createTextNode(content);
	var contentNode = resource.setContent(content);
	assertEquals("setContent should return a text node.", textNode.nodeType, contentNode.nodeType);
	assertEquals("getContent should return a text node.", textNode.nodeType, resource.getContent().nodeType);
	assertTrue("contentNode and getContent() should be identical", contentNode === resource.getContent());
	
	assertEquals("getContent().textContent should be content.", content, resource.getContent().textContent);
	
	var element = resource.getElement();
	checkElement(element);
	
	assertEquals("element should have exactly one child.", 1, element.childNodes.length);
	assertTrue("element.firstChild shouldbe identical to contentNode", element.firstChild === contentNode);
	assertEquals("element.firstChild.textContent should equal content", content, element.firstChild.textContent);
}


/** Test - Load and Unload
 * Test loading the resource into the document.
 */
function testLoadUnload() {
	var count1 = document.getElementsByTagName(category).length;
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
	var resource = new MozileScriptResource(id, source);
	resource.load();
	var count2 = document.getElementsByTagName(category).length;
	assertEquals("There should be one more element now.", count2, count1+1);
	assertEquals("There should be exactly one element with the matching id", 1, countElements(category, id));
	
	var resource2 = new MozileScriptResource(id+"1", source+"1");
	resource2.load();
	assertEquals("There should still be exactly one element with the matching id", 1, countElements(category, id));
	assertEquals("There should be exactly one element with the matching id", 1, countElements(category, id+"1"));
	resource2.unload();
	
	resource.unload();
	var count3 = document.getElementsByTagName(category).length;
	assertEquals("There should be one fewer element now.", count3, count2-1);
	assertEquals("There should be the same number of elements as when we started.", count3, count1);
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
	resource.load();
	resource.unload(document.documentElement);
	var count4 = document.getElementsByTagName(category).length;
	assertEquals("There should be the same number of elements as when we started.", count4, count1);
	assertEquals("There should be no elements with the matching id", 0, countElements(category, id));
	
}





/** Check Script Resource
 * Check the particulars of a script resource.
 */
function checkScriptResource(resource) {
	assertEquals("getSource() should return the source", source, resource.getSource());
}


/** Check Script Element
 * Check the particulars of a script element.
 */
function checkScriptElement(element) {
	assertEquals("The src of element should equal source", source, element.getAttribute("src"));
}

