/***
  * The Developer of this Code is HJ van Rantwijk. Copyright
  * (C) 2004-2006 by HJ van Rantwijk.  All Rights Reserved.
  */

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cr = Components.results;

const STATE_IDLE = 0;
const STATE_BUSY = 1;

const DEFAULT_LIMIT = 5;
const DEFAULT_INTERVAL = 5; // minutes

const VERSION_INFO = "mzWebFeeds version 1.0.i";

function debug(aMsg) {
  if (mzWebFeedService.jsConsoleMessages) {
    aMsg = ("mzWebFeed Service: " + aMsg).replace(/\S{80}/g, "$&\n");
    Cc["@mozilla.org/consoleservice;1"].getService(Ci.nsIConsoleService).logStringMessage(aMsg);
  }
}
/* :::::::: The Web Feed Service ::::::::::::::: */

const mzWebFeedService = {
  initialized: false,
  mTimer: null, // need to hold on to timer ref
  updateQueue: new Array(),
  pendingUpdates: null,
  subscriptionsDS: null,
  RDF: null,
  RDFCUtils: null,
  NS : "http://multizilla.mozdev.org/rdf#",
  prefBranch : null,
  refreshInterval: DEFAULT_INTERVAL,
  refreshLimit: DEFAULT_LIMIT,
  currentState: STATE_IDLE,
  _webFeedsFile: null,
  _webFeedsFileBackup: null,
  _bundle: null,
  _nsIWindowMediator: null,
  _nsIObserverService: null,
  feedCache: null,
  categoryCache: null,
  refreshStateCache: null,
  failedRefreshCache: null,
  userAgent: "The MultiZilla Feed Reader/Viewer",
  lastUpdateCheck: 0,
  jsConsoleMessages: false,

  onProfileStartup: function(aProfileName)
  {
    this._nsIObserverService = Cc["@mozilla.org/observer-service;1"].getService(Ci.nsIObserverService);
    this._nsIObserverService.addObserver(this, "domwindowopened", false);
    this._nsIWindowMediator = Cc["@mozilla.org/appshell/window-mediator;1"].getService(Ci.nsIWindowMediator);
  },

  /**
   * Handle notifications
   */
  observe: function(aSubject, aTopic, aData)
  {
    switch (aTopic) {
      case "domwindowopened":
        this.mTimer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
        this.RDF = Cc["@mozilla.org/rdf/rdf-service;1"].getService(Ci.nsIRDFService);
        this.RDFCUtils = Cc["@mozilla.org/rdf/container-utils;1"].getService(Ci.nsIRDFContainerUtils);
        this._bundle = Cc["@mozilla.org/intl/stringbundle;1"].getService(Ci.nsIStringBundleService)
                                                             .createBundle("chrome://multiviews/locale/multizilla.properties");
        // we are no longer interested in the ``domwindowopened'' topic
        this._nsIObserverService.removeObserver(this, "domwindowopened");
        // remove our timer reference on XPCOM shutdown to avoid a leak.
        this._nsIObserverService.addObserver(this, "xpcom-shutdown", false);
        this.prefBranch = Cc["@mozilla.org/preferences-service;1"].getService(Ci.nsIPrefService).getBranch("multizilla.feeds.refresh.");
        this.prefBranch.QueryInterface(Ci.nsIPrefBranch2);
        this.lastUpdateCheck = (this._getPref("lastcheck", 0) * 1000);
        this.refreshInterval = (this._getPref("interval", DEFAULT_INTERVAL) * 60000);

        if (this.refreshInterval < 60000)
          this.refreshInterval = 60000;
        else if (this.refreshInterval > 3600000)
          this.refreshInterval = 3600000;

        this.jsConsoleMessages = this._getPref("debug_messages");
        this.refreshLimit = this._getPref("limit", DEFAULT_LIMIT);

        if (this.refreshLimit < 1)
          this.refreshLimit = 1;

        this.prefBranch.addObserver("debug_messages", this, false);
        this.prefBranch.addObserver("limit", this, false);
        this.prefBranch.addObserver("interval", this, false);

        if (this.getMostRecentBrowserWindow() != null)
          this.init();
        else
          this.mTimer.init(this, 2500, Ci.nsITimer.TYPE_ONE_SHOT);

        break;

      case "timer-callback":
        if (!this.initialized)
          this.init();
        else
          this.checkFeedsForUpdates();
        break;

      case "nsPref:changed":
        switch(aData) {
          case "debug_messages":
            this.jsConsoleMessages = this._getPref(aData, this.jsConsoleMessages);
            debug("JS Console messages: " + (this.jsConsoleMessages ? "enabled" : "disabled"));
            break;
          case "limit":
            this.refreshLimit = this._getPref(aData, this.refreshLimit);

            if (this.refreshLimit < 1)
              this.refreshLimit = 1;

            debug("Refresh limit set to: " + this.refreshLimit);
            break;
          case "interval":
            this.refreshInterval = (this._getPref(aData, this.refreshInterval) * 60000);

            if (this.refreshInterval < 60000)
              this.refreshInterval = 60000;
            else if (this.refreshInterval > 3600000)
              this.refreshInterval = 3600000;

            this.mTimer.init(this, this.refreshInterval, Ci.nsITimer.TYPE_ONE_SHOT);
            debug("Refresh interval set to: " + this.refreshInterval + "ms delay");
            break;
          }
        break;

      case "xpcom-shutdown":
        /* 
         * We need to drop our timer reference here to avoid a leak
         * since the timer keeps a reference to the observer! 
         */
        this.mTimer = null;
        break;
    } // switch (aTopic) {
  },

  init: function()
  {
    this.getFeedCategories(); // initializes the 'categoryCache'
    this.getFeedSubscriptions(); // initializes the 'feedCache'
    this.getGlobalRefreshState(); // initializes the 'refreshStateCache'
    this.failedRefreshCache = new Array();

    var refreshDelay = 2500;
    var now = new Date().getTime();

    if (this.lastUpdateCheck > 0) {
      var nextUpdate = (this.lastUpdateCheck + this.refreshInterval);

      if (nextUpdate > (now + 2500))
        refreshDelay = (nextUpdate - now);
    }
    // this._nsIObserverService.notifyObservers(null, "WebFeedService", "NextUpdateCheck:" + nextUpdate);
    this.mTimer.init(this, refreshDelay, Ci.nsITimer.TYPE_ONE_SHOT);
    debug("Initialized with a " + this.refreshInterval + "ms delay");

    if (this.feedCache.length)
      debug("Next refresh: " + new Date(now + refreshDelay));

    this.initialized = true;
  },

  initDataSource: function()
  {
    if (!this.subscriptionsDS)
      this.subscriptionsDS = this.getMostRecentBrowserWindow().mzGetNewOrCurrentDSByFilename("mzWebFeeds.rdf", true, null, false);

    return this.subscriptionsDS
  },

  checkFeedsForUpdates: function()
  {
    if (this.feedCache.length) {
      debug("Checking idle state...");

      if (this.currentState == STATE_IDLE) {
        debug("Current state: STATE_IDLE");
        this.updateAllFeeds(false);
      }
    }
    else
      this.mTimer.init(this, this.refreshInterval, Ci.nsITimer.TYPE_ONE_SHOT);
  },

  updateFeeds: function(aFeedResourcesString, aForcedCheckFlag)
  {
    if (aFeedResourcesString) {
      debug("Building feed list...")
      var feedResourceList = aFeedResourcesString.split(',');
      this.refreshChannels(feedResourceList, aForcedCheckFlag);
    }
    else
      this.mTimer.init(this, this.refreshInterval, Ci.nsITimer.TYPE_ONE_SHOT);
  },

  updateAllFeeds: function(aForcedCheckFlag)
  {
    var feedSubscriptions = this.getFeedSubscriptions();

    if (feedSubscriptions) {
      debug("Building feed list...")
      this.lastUpdateCheck = new Date().getTime();
      debug("Updating last refresh time...");
      this.prefBranch.setIntPref("lastcheck", (this.lastUpdateCheck / 1000));
      this.refreshChannels(feedSubscriptions, true);
    }
    else
      this.mTimer.init(this, this.refreshInterval, Ci.nsITimer.TYPE_ONE_SHOT);
  },

  refreshChannels: function(aFeedList, aForcedCheckFlag)
  {
    this.currentState == STATE_BUSY;
    this.pendingUpdates = new Array();

    try {

      if (aFeedList.length > 0) {
        var dataSource = this.initDataSource();
        debug("Checking local feed storage...");
        for (var i = 0; i < aFeedList.length; i++) {
          var urn = (typeof aFeedList[i] == "object") ? aFeedList[i].urn : aFeedList[i];
          dump("\n\ncurrentResource[ " + i + " ]: " + urn);
          var currentResource = this.RDF.GetResource(urn);
          var feedContainer = this.RDFCUtils.MakeSeq(dataSource, currentResource);

          if (feedContainer) {
            var dataObj = new Object();
            dataObj.urn = urn;

            var resource = dataSource.GetTarget(currentResource, this.RDF.GetResource(this.NS + "state"), true);
            dump("\nChecking 'state': " + resource);
            if (resource)
              dataObj.state = resource.QueryInterface(Ci.nsIRDFLiteral).Value;
            else
              dump("\nNo 'state' for: " + urn);

            resource = dataSource.GetTarget(currentResource, this.RDF.GetResource(this.NS + "url"), true);
            dump("\nChecking 'url': " + resource);
            if (resource)
              dataObj.url = resource.QueryInterface(Ci.nsIRDFLiteral).Value;
            else
              dump("\nNo 'url' for: " + urn);

            resource = dataSource.GetTarget(currentResource, this.RDF.GetResource(this.NS + "date"), true);
            dump("\nChecking 'date': " + resource);
            if (resource)
              dataObj.date = resource.QueryInterface(Ci.nsIRDFLiteral).Value;
            else
              dump("\nNo 'date' for: " + urn);

            resource = dataSource.GetTarget(currentResource, this.RDF.GetResource(this.NS + "lastModified"), true);
            dump("\nChecking 'lastmodified': " + resource);
            if (resource instanceof Ci.nsIRDFNode)
              dataObj.rdfDate = resource.QueryInterface(Ci.nsIRDFLiteral).Value;
            else
              dataObj.rdfDate = dataObj.date;

            if (aForcedCheckFlag) {
              dataObj.lastModifiedDate = 0;
              dataObj.rdfDate = -1;
            }
            this.pendingUpdates.push(dataObj);
          }
          else {
            debug("Refresh failed for: " + urn);
            var category = this.getCategoryByFeed(urn);
            var categoryContainer = this.RDFCUtils.MakeSeq(dataSource, this.RDF.GetResource(category));

            if (categoryContainer) {
              categoryContainer.RemoveElement(currentResource, true);
              this.flushDataSource(dataSource);
            }
            aFeedList = aFeedList.splice(i ,1);
          }
        }
      }
      if (this.pendingUpdates.length) {
        // this.updateQueue = new Array();
        debug(aFeedList.length + " feed(s) pending for a refresh...");
        this.checkPendingUpdates(aForcedCheckFlag);
      }
      else {
        this.currentState = STATE_IDLE;
        // The update check has finished, forced a re-init of the global refresh state
        var currentState = this.getGlobalRefreshState(true);
        this._nsIObserverService.notifyObservers(null, "WebFeedService", (currentState) ? "hasNewContent" : "noNewContent");
        dump("\nrefreshInterval: " + this.refreshInterval);
        this.mTimer.init(this, this.refreshInterval, Ci.nsITimer.TYPE_ONE_SHOT);
        debug("No new feed items available");
        debug("Next refresh: " + new Date(this.lastUpdateCheck + this.refreshInterval));
      }
    } catch(ex) { debug(ex); }
  },

  checkPendingUpdates: function(aForcedCheckFlag)
  {
    if (this.pendingUpdates && this.pendingUpdates.length && 
        (this.updateQueue.length <= this.refreshLimit)) {
      dump("\ncheckPendingUpdate() " + this.pendingUpdates.length + " item(s) left...");
      var _this = this; // for event handlers
      var dataObj = this.pendingUpdates[0];
      // debug("Checking feed: " + dataObj.urn);
      this.addOrChangeProperty(this.subscriptionsDS, this.RDF.GetResource(dataObj.urn), this.RDF.GetResource(this.NS + "state"), "checking");
      var xmlRequest = Cc["@mozilla.org/xmlextras/xmlhttprequest;1"].createInstance(Ci.nsIXMLHttpRequest);
      // xmlRequest.onprogress = function(event) { onXMLProgress(event); };
      xmlRequest.open("HEAD", dataObj.url, true);
      xmlRequest.setRequestHeader("User-Agent", this.userAgent);
      xmlRequest.onerror = function(aEvent)
      {
        var request, status;

        try {
          request = aEvent.target;
          // the following may throw (e.g. a local file or timeout)
          status = request.status;
        }
        catch(ex) {
          request = aEvent.target.channel.QueryInterface(Ci.nsIRequest);
          status = request.status;
        }
        xmlRequest.abort();
        _this.addOrChangeProperty(_this.subscriptionsDS, _this.RDF.GetResource(dataObj.urn), _this.RDF.GetResource(_this.NS + "state"), "");
        _this.hasNewArticles(dataObj.urn, true, null);
        _this.removeFromPendingQueue(dataObj.urn);
      }

      xmlRequest.onload = function()
      {
        // dump("\nxmlRequest.readyState: " + xmlRequest.readyState);
        // dump("\nonload: " + xmlRequest.status);
        if (xmlRequest.readyState == 4) {
          // dump("\n\nxmlRequest.status: " + xmlRequest.status);
          // dump("\ngetAllResponseHeaders: " + xmlRequest.getAllResponseHeaders());
          if (xmlRequest.status == 200) {
            var lastModifiedDate = xmlRequest.getResponseHeader("Last-Modified");
            dump("\nlastModifiedDate: " + lastModifiedDate);
            dataObj.lastModifiedDate = new Date(lastModifiedDate).getTime();
            /* dump("\nChannel urn: " + dataObj.urn +
                    "\nlastModified send by server   : " + dataObj.lastModifiedDate +
                    "\ndate stored in mzWebFeeds.rdf : " + dataObj.rdfDate); */
            var lastMonth = new Date().getTime() - 2592000000;

            if (aForcedCheckFlag || (dataObj.lastModifiedDate < lastMonth) || (dataObj.lastModifiedDate > dataObj.rdfDate)) {
              /***
                * The 'Last-Modified' send by the server is more recent, but that  
                * doesn't necessarily means that the document itself is updated,
                * but callback function updateStageTwo will take care of that.
                */
              _this.updateStageTwo(dataObj);
            }
            else {
              // _this.hasNewArticles(dataObj.urn, true, null);
              /* var newState = (dataObj.state == "checking") ? "" : dataObj.state;  
              this.addOrChangeProperty(this.subscriptionsDS, this.RDF.GetResource(dataObj.urn), this.RDF.GetResource(this.NS + "state"), newState); */
              _this.removeFromPendingQueue(dataObj.urn);
            }
          }
          else {
            // dump("\nERROR");
            // _this.addOrChangeProperty(_this.subscriptionsDS, _this.RDF.GetResource(dataObj.urn), _this.RDF.GetResource(_this.NS + "state"), "");
            // _this.hasNewArticles(dataObj.urn, true, null);
            // _this.removeFromPendingQueue(dataObj.urn);
          }
        }
      }
      xmlRequest.send(null);
    }
    else {
      var failedFeeds = this.failedRefreshCache.length;

      if (failedFeeds) {
        debug(failedFeeds + " feed(s) failed. Retrying...");
        var failedFeeds = this.failedRefreshCache;
        this.failedRefreshCache = new Array();
        this.refreshChannels(failedFeeds, true);
      }
      else {
        this.currentState = STATE_IDLE;
        // The update check has finished, forced a re-init of the global refresh state
        var currentState = this.getGlobalRefreshState(true);
        this._nsIObserverService.notifyObservers(null, "WebFeedService", (currentState) ? "hasNewContent" : "noNewContent");
        this.mTimer.init(this, this.refreshInterval, Ci.nsITimer.TYPE_ONE_SHOT);
        debug("Feed refresh (check) finished");
        debug("Next refresh: " + new Date(this.lastUpdateCheck + this.refreshInterval));
      }
    }
  },

  updateStageTwo: function(aDataObj)
  {
    // dump("\nupdateStageTwo()");
    var _this = this;
    var feedURN = aDataObj.urn;
    var xmlRequest = Cc["@mozilla.org/xmlextras/xmlhttprequest;1"].createInstance(Ci.nsIXMLHttpRequest);
    xmlRequest.overrideMimeType("application/xml"); // "application/rss+xml"
    xmlRequest.onprogress = function(aEvent)
    {
      var percentage = (aEvent.position * 100) / aEvent.totalSize;
      // dump("\npercentage: " + percentage);
      // statusMeter.value = percentage;
    }
    xmlRequest.open("GET", aDataObj.url);
    xmlRequest.setRequestHeader("User-Agent", this.userAgent);
    xmlRequest.setRequestHeader("Cache-Control", "no-cache");
    xmlRequest.overrideMimeType("application/xml");

    xmlRequest.onerror = function(aEvent)
    {
      var request, status;

      try {
        request = aEvent.target;
        // the following may throw (e.g. a local file or timeout)
        status = request.status;
      }
      catch(ex) {
        request = aEvent.target.channel.QueryInterface(Ci.nsIRequest);
        status = request.status;
      }
      xmlRequest.abort();

      if (_this.failedRefreshCache.indexOf(feedURN) == -1)
        _this.failedRefreshCache.push(feedURN);

      _this.removeFromPendingQueue(feedURN);
    }

    xmlRequest.onload = function()
    {
      var xmldoc = xmlRequest.responseXML;

      if (xmldoc) {
        var dataCollection = _this.getFeedInfoFromDocument(xmldoc);
        /* dump("\ndate inside remote document   : " + dataCollection[7] +
                "\ndate stored in mzWebFeeds.rdf : " + aDataObj.rdfDate +
                "\nlastModified send by server   : " + aDataObj.lastModifiedDate); */
        if ((dataCollection[7] > aDataObj.rdfDate) || 
          (aDataObj.lastModifiedDate > dataCollection[7]) || (aDataObj.lastModifiedDate > aDataObj.rdfDate))
        {
          dataCollection[1] = aDataObj.url;
          dataCollection[8] = aDataObj.urn;
          dataCollection = _this.getDataCollectionFromDocument(xmldoc, dataCollection);
          // dump("\nUpdate available for: " + aDataObj.urn);
          /***
            * We have a new update available and callback function updateStageThree
            * will remove all current items, and add this update to the update queue
            * which will later be used by callback function finalizeUpdateProcess.
            */
            _this.updateStageThree(dataCollection);
        }
        else {
          // _this.hasNewArticles(aDataObj.urn, true, null);
          /* var newState = (aDataObj.state == "checking") ? "" : aDataObj.state;
          this.addOrChangeProperty(this.subscriptionsDS, this.RDF.GetResource(dataObj.urn), this.RDF.GetResource(this.NS + "state"), newState); */
          // dump("\nNo update available for: " + aDataObj.urn);
          _this.removeFromPendingQueue(aDataObj.urn);
        }
      }
    }
    xmlRequest.send(null);
  },

  updateStageThree: function(aDataCollection)
  {
    var urn = aDataCollection[0][8];
    // debug("Refreshing feed: " + urn);
    var category = this.getCategoryByFeed(urn);
    var resource = this.RDF.GetResource(urn);
    var dataSource = this.initDataSource();

    this.removeResources(dataSource, urn); // remove article container
    this.removeDataResources(dataSource, resource, false); // remove articles
    this.addFeedFromDataCollection(aDataCollection, true, category);
    this.removeFromPendingQueue(urn);
  },

  removeFromPendingQueue: function(aTargetURN, aSkipCaller)
  {
    for (i in this.pendingUpdates) {
      var target = (this.pendingUpdates[i].urn == undefined) ? this.pendingUpdates[i][8] : this.pendingUpdates[i].urn;

      if (target == aTargetURN) {
        // dump("\nRemoving: " + aTargetURN + " from pending queue...");
        this.pendingUpdates.splice(i, 1);
      }
    }
    if (this.pendingUpdates.length == 0)
      this.currentState = STATE_IDLE;

    this.checkPendingUpdates();
  },

  hasNewArticles: function(aResource, aUpdateChannelFlag, aTargetState)
  {
    /* dump("\n\nhasNewArticles - aResource         : " + aResource +
           "\nhasNewArticles - aUpdateChannelFlag: " + aUpdateChannelFlag); */
    var dataSource = this.initDataSource();
    var state = (aTargetState == null) ? false : aTargetState;
    var resource = aResource.replace(/:article-\d*/, '');
    resource = this.RDF.GetResource(resource);

    if (aTargetState == null) {
      var channels = this.getResourcesFromContainer(resource);

      if (channels && channels.length) {
        var property = this.RDF.GetResource(this.NS + "state");
        // var targetVisited = this.RDF.GetLiteral("visited");
        // var targetUnread = this.RDF.GetLiteral("unread");

        for (i in channels) {
          // dump("\nchecking: " + channels[i]);
          var feedResource = this.RDF.GetResource(channels[i]);
           /* if (dataSource.HasAssertion(feedResource, property, targetUnread, true) || 
              dataSource.HasAssertion(feedResource, property, targetVisited, false)) {
            state = true;
            break;
          } */
          var data = this.subscriptionsDS.GetTarget(feedResource, property, true);

          if (data) {
            var currentState = data.QueryInterface(Ci.nsIRDFLiteral).Value;
            // dump("\ncurrentState: " + currentState);

            if (currentState == "unread") {
              // dump("\nstate is 'unread'");
              state = true;
              break;
            }
          }
          else {
            // dump("\nno state node: " + data);
            state = true;
            break;
          }
        }
      }
      state = (state) ? "new" : "";
    }
    /* dump("\nhasNewArticles - state             : " + state +
         "\nhasNewArticles - resource          : " + resource.Value); */
    if (aUpdateChannelFlag)
      this.setVisitedState(resource.Value, state);

    // The check has finished, forced a re-init of the global refresh state
    var currentState = this.getGlobalRefreshState(true);
    this._nsIObserverService.notifyObservers(null, "WebFeedService", (currentState) ? "hasNewContent" : "noNewContent");
    this.flushDataSource(dataSource);
    return state;
  },

  addFeedFromDataCollection: function(aDataCollection, aRefreshFlag, aCategory)
  {
    this.getMostRecentBrowserWindow().mzFeedHandler.addFeedFromDataCollection(aDataCollection, aRefreshFlag, aCategory);
  },

  addFeedCategory: function(aCategoryName, aTargetResource)
  {
    dump("\naddFeedCategory: " + aTargetResource);
    try {
      var dataSource = this.initDataSource();
      var rootResource = this.RDF.GetResource("urn:feeds:root");
      var rootContainer = this.RDFCUtils.MakeSeq(dataSource, rootResource);

      /* if (!rootContainer) {
        var newContainer = Cc["@mozilla.org/rdf/container;1"].createInstance(Ci.nsIRDFContainer);
        rootContainer = newContainer.Init(dataSource, rootResource);
        // this.flushDataSource(dataSource);
      }
      else { */
        var propertiesArray = [ "name", "group", "date", "state" ];
        var category = (aCategoryName) ? aCategoryName : "New Feed Group";
        var valuesArray = [ category, "true", new Date().getTime(), "undetermined" ];
        var anonymousResource = this.RDF.GetAnonymousResource();
        var resource = this.RDF.GetResource("urn:feedgroup:" + anonymousResource.Value.replace(/^rdf:/, ''));

        for (i in propertiesArray) {
          var property = this.RDF.GetResource(this.NS + propertiesArray[i]);
          dataSource.Assert(resource, property, this.RDF.GetLiteral(valuesArray[i]), true);
        }
        var targetIndex = (aTargetResource == undefined) ? 0 : rootContainer.IndexOf(this.RDF.GetResource(aTargetResource));

        if (targetIndex > 0 && targetIndex < rootContainer.GetCount())
          rootContainer.InsertElementAt(resource, targetIndex, true);
        else
          rootContainer.AppendElement(resource);

        this.RDFCUtils.MakeSeq(dataSource, resource);
        this.flushDataSource(dataSource);
        this.categoryCache.push(resource.Value)
        return resource.Value;
      // }
    }
    catch(ex) { 
      dump("\naddFeedCategory - ex: " + ex);
    }
    return null;
  },

  setVisitedState: function(aResourceString, aState)
  {
    // dump("\nsetVisitedState - aState: " + aState);
    var dataSource = this.initDataSource();
    var resource = this.RDF.GetResource(aResourceString);

    if (aState != "")
      this.addOrChangeProperty(dataSource, resource, this.RDF.GetResource(this.NS + "state"), aState);
    else { // if (this.getVisitedState(aResourceString) && aState == "") {
      var property = this.RDF.GetResource(this.NS + "state");
      var stateResource = dataSource.GetTarget(resource, property, true);

      if (stateResource instanceof Ci.nsIRDFNode)
        dataSource.Unassert(resource, property, stateResource);
    }
    // if (aSkipParent == undefined)
      // this.hasNewArticles(aResourceString, true, null);
  },

  getVisitedState: function(aResourceString)
  {
    // dump("\ngetVisitedState - aResourceString: " + aResourceString);
    var dataSource = this.initDataSource();
    return (dataSource.HasAssertion(this.RDF.GetResource(aResourceString), 
                                    this.RDF.GetResource(this.NS + "state"), 
                                    this.RDF.GetLiteral("visited"), 
                                    true));
  },

  getGlobalRefreshState: function(aForcedRefreshFlag)
  {
    if (aForcedRefreshFlag || (this.refreshStateCache == null)) {
      var dataSource = this.initDataSource();
      var feedSubscriptions = this.getFeedSubscriptions();

      for (i in feedSubscriptions) {
        if (dataSource.HasAssertion(this.RDF.GetResource(feedSubscriptions[i].urn), 
                                    this.RDF.GetResource(this.NS + "state"), this.RDF.GetLiteral("new"), true)) {
          
          this.refreshStateCache = true;
          return true;
        }
      }
      this.refreshStateCache = false;
    }
    return this.refreshStateCache;
  },

  getFeedInfoFromDocument: function(aXMLDocument)
  {
    return this.getMostRecentBrowserWindow().mzFeedHandler.getFeedInfoFromDocument(aXMLDocument);
  },

  getDataCollectionFromDocument: function(aXMLDocument, aDataCollection)
  {
    return this.getMostRecentBrowserWindow().mzFeedHandler.getDataCollectionFromDocument(aXMLDocument, aDataCollection);
  },

  getFeedCategoryName: function(aCategory)
  {
    var dataSource = this.initDataSource();
    var data = dataSource.GetTarget(this.RDF.GetResource(aCategory), 
                                    this.RDF.GetResource(this.NS + "name"), 
                                    true);

    if (data instanceof Ci.nsIRDFLiteral)
      return data.QueryInterface(Ci.nsIRDFLiteral).Value;
    return null;
  },

  renameFeedCategory: function(aCategoryResource, aNewName)
  {
    /* dump("\n\naCategoryResource: " + aCategoryResource +
         "\nName             : " + aNewName); */
    if (/^urn:feedgroup:/.test(aCategoryResource)) {
      var dataSource = this.subscriptionsDS;
      var property = this.RDF.GetResource(this.NS + "name");
      this.addOrChangeProperty(dataSource, this.RDF.GetResource(aCategoryResource), property, aNewName)
      return aNewName;
    }
    else
      debug("RenameFeedCategory() failed for " + aCategoryResource);
    return null;
  },

  removeFeedCategory: function(aResource, aPurgeFlag)
  {
    /* dump("\n\naResource : " + aResource +
         "\naPurgeFlag: " + aPurgeFlag); */
    if (aResource && /^urn:feedgroup:/.test(aResource)) {
      if (aPurgeFlag == undefined)
        aPurgeFlag = false;

      var dataSource = this.initDataSource();
      var resource = this.RDF.GetResource(aResource);

      if (resource instanceof Ci.nsIRDFResource) {
        var categoryContainer = this.RDFCUtils.MakeSeq(dataSource, resource);

        if (categoryContainer) {
          var checkbox = { value:aPurgeFlag };
          var checkboxText = null;
          var feedCount = categoryContainer.GetCount();
          var hasFeeds = (feedCount > 0);

          if (hasFeeds)
            checkboxText = this._bundle.GetStringFromName("removeFeedConfirmationCheckboxText");

          var promptService = Cc["@mozilla.org/embedcomp/prompt-service;1"].getService(Ci.nsIPromptService);
          var navigatorWindow = this.getMostRecentBrowserWindow();
          var categoryName = this.getFeedCategoryName(aResource);
          var confirmationDialogTitle = this._bundle.GetStringFromName("removeFeedConfirmationTitle");
          var confirmationDialogText = this._bundle.formatStringFromName("removeFeedConfirmationText", [categoryName], 1);
          var confirmResult = promptService.confirmEx(navigatorWindow, confirmationDialogTitle, confirmationDialogText,
                                                      (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_YES) + 
                                                      (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_CANCEL),
                                                      null, null, null, checkboxText, checkbox);

          if (confirmResult == 0) { // Yes returns 0 / No, Esc and X returns 1
            if (hasFeeds) {
              var feeds = this.getResourcesFromContainer(resource);

              if (checkbox.value) {
                for (i in feeds) {
                  this.moveFeed(feeds[i], aResource, "urn:feeds:root", -1);
                }
              }
              else {
                for (i in feeds) {
                  this.removeResources(dataSource, feeds[i]); // remove articles
                  this.removeDataResources(dataSource, feeds[i], true); // remove article container
                  this.removeFeedFromFeedCache(feeds[i]);
                }
              }
            }
            this.removeDataResources(dataSource, resource, true); // remove category
            var rootContainer = this.RDFCUtils.MakeSeq(dataSource, this.RDF.GetResource("urn:feeds:root"));
            rootContainer.RemoveElement(resource, true);
            this.flushDataSource(dataSource);
            var categoryCacheIndex = this.categoryCache.indexOf(aResource);

            if (categoryCacheIndex != -1)
              this.categoryCache.splice(categoryCacheIndex, 1);
            return true;
          }
        }
      }
    }
    return false;
  },

  removeResources: function(aDataSource, aResource)
  {
    // see also mzRemoveResources() in multiviewsRDF.js
    // dump("\nremoveResources: " + aResource);
    var resource = this._GetResource(aResource);
    var container = this.RDFCUtils.MakeSeq(aDataSource, resource).GetElements();

    while (container.hasMoreElements()) {
      var currentResource = container.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
      var labels = aDataSource.ArcLabelsOut(currentResource);

      while (labels.hasMoreElements()) {
        var label = labels.getNext().QueryInterface(Ci.nsIRDFResource);
        var data = aDataSource.GetTarget(currentResource, label, true);

        if (data instanceof Ci.nsIRDFNode)
          aDataSource.Unassert(currentResource, label, data);
      }
    }
    this.flushDataSource(aDataSource);
  },

  removeDataResources: function(aDataSource, aResource, aUnassertLiteralsFlag)
  {
    // see also mzRemoveDataResources() in multiviewsRDF.js
    // dump("\nremoveDataResources");
    var resource = this._GetResource(aResource);
    var labels = aDataSource.ArcLabelsOut(resource);

    while (labels.hasMoreElements()) {
      var label = labels.getNext().QueryInterface(Ci.nsIRDFResource);
      var data = aDataSource.GetTarget(resource, label, true);

      if (data instanceof Ci.nsIRDFResource) 
        aDataSource.Unassert(resource, label, data);
      else if (aUnassertLiteralsFlag && data instanceof Ci.nsIRDFLiteral)
        aDataSource.Unassert(resource, label, data);
    }
    this.flushDataSource(aDataSource);
  },

  addFeedToCategory: function(aFeedResource, aCategoryResource)
  {
    // dump("\naddFeedToCategory: " + aCategoryResource);
    this.moveFeed(aFeedResource, "urn:feeds:root", aCategoryResource, -1);
  },

  removeFeedFromCategory: function(aFeedResource, aCategoryResource)
  {
    // dump("\naddFeedToCategory: " + aCategoryResource);
    this.moveFeed(aFeedResource, aCategoryResource, "urn:feeds:root", -1);
  },

  isNewFeedURL: function(aURL)
  {
    // dump("\nChecking " + aURL + " in isNewFeedURL()");
    var dataSource = this.initDataSource();
    var rootResource = this.RDF.GetResource("urn:feeds:root")
    var rootContainer = this.RDFCUtils.MakeSeq(dataSource, rootResource);

    if (!rootContainer) {
      var newContainer = Cc["@mozilla.org/rdf/container;1"].createInstance(Ci.nsIRDFContainer);
      rootContainer = newContainer.Init(dataSource, rootResource);
      return true;
    }
    else {
      var _nsIIOService = Cc["@mozilla.org/network/io-service;1"].getService(Ci.nsIIOService);
      aURL = aURL.replace(/www\./,'');
      var uri = _nsIIOService.newURI(aURL, null, null);
      var urn = "urn:" + uri.host + ":feed-";
      var property = this.RDF.GetResource(this.NS + "url");
      var feedSubscriptions = this.getFeedSubscriptions();

      for (i in feedSubscriptions) {
        if (aURL.indexOf(feedSubscriptions[i].url) == 0) {
          // dump("\nThis feed is already filed!");
          return false;
        }
      }
    }
    // dump("\nThis feed isn't filed yet!");
    return true;
  },

  removeFeed: function(aCategoryResource, aFeedResource)
  {
    /* dump("\nremoveFeed - aCategoryResource: " + aCategoryResource + 
         "\nremoveFeed - aFeedResource    : " + aFeedResource); */
    var dataSource = this.initDataSource();
    var categoryResource = this._GetResource(aCategoryResource);
    var feedResource = this._GetResource(aFeedResource);
    var feedContainer = this.RDFCUtils.MakeSeq(dataSource, categoryResource);

    if (feedContainer) {
      feedContainer.Init(dataSource, categoryResource);
      feedContainer.RemoveElement(feedResource, true);
      this.flushDataSource(dataSource);
      this.removeFeedFromFeedCache(aFeedResource);
      return true;
    }
    return false;
  },

  moveFeed: function(aFeedResource, aFromContainerResource, aToContainerResource, aTargetIndex)
  {
    /* dump("\n\nmoveFeed     : " + aFeedResource + 
         "\nfrom container: " + aFromContainerResource +
         "\nto container  : " + aToContainerResource +
         "\nTarget index  : " + aTargetIndex); */
    try {

    var dataSource = this.initDataSource();
    var feedResource = this.RDF.GetResource(aFeedResource);
    var fromContainer = this.RDFCUtils.MakeSeq(dataSource, this.RDF.GetResource(aFromContainerResource));
    var toContainer;

    if (aFromContainerResource == aToContainerResource)
      toContainer = fromContainer;
    else 
      toContainer = this.RDFCUtils.MakeSeq(dataSource, this.RDF.GetResource(aToContainerResource));

    fromContainer.RemoveElement(feedResource, true);

    var itemCount = toContainer.GetCount();

    if (aTargetIndex < 1 || aTargetIndex > itemCount)
      toContainer.AppendElement(feedResource);
    else
      toContainer.InsertElementAt(feedResource, aTargetIndex, true);

    this.flushDataSource(dataSource);
    return aFeedResource;

    } catch(ex) {
      dump("\nmoveFeed - ex: " + ex);
    }
    return null;
  },

  addFeedToFeedCache: function(aFeed, aURL)
  {
    /* dump("\n\naddURNToFeedCache - aFeed: " + aFeed +
         "\naddURNToFeedCache - aURL : " + aURL); */
    if (this.feedCache == null)
      this.feedCache = new Array();

    for (i in this.feedCache) {
      if (this.feedCache[i].urn == aFeed)
        return;
    }
    this.feedCache.push({urn:aFeed, url:aURL});
  },

  removeFeedFromFeedCache: function(aFeed)
  {
    // dump("\nremoveFeedFromFeedCache: " + aFeed);
    for (var i = 0; i < this.feedCache.length; i++ ) {
      if (this.feedCache[i].urn == aFeed) {
        this.feedCache.splice(i, 1);
        return true;
      }
    }
    return false;
  },

  getFeedInfoFromRDF: function(aInfoArray)
  {
    // dump("\ngetFeedInfoFromRDF: " + aFeedResource);
    var names = [ "urn", "feed", "url", "title", "link", "description", "date", "lastModified", 
                  "image", "imageUrl", "imageTitle", "imageWidth", "imageHeight" ];
    var dataSource = this.initDataSource();
    var resource = this._GetResource(aInfoArray[0])
    var feedContainer = this.RDFCUtils.MakeSeq(dataSource, resource);

    if (feedContainer) {
      var lastItem = names.length;

      for (var i = 1; i < lastItem; i++) {
        var data = dataSource.GetTarget(resource, this._GetResource(this.NS + names[i]), true);      
        aInfoArray.push("");

        if (data instanceof Ci.nsIRDFLiteral) {
          var textValue = data.QueryInterface(Ci.nsIRDFLiteral).Value;
          // dump("\nnames[" + i + "]: " + textValue);
          if (i == 1)
            aInfoArray[i] = (textValue.match(/^rss/)) ? "rss" : textValue;
          else if (i == 6 || i == 7)
            aInfoArray[i] = (textValue == "") ? "" : new Date().toString(textValue);
          else
            aInfoArray[i] = textValue;
        }
        if ((data == null) && (i == 8)) {
          // dump("\ndata[" + i + "] : " + data);
          lastItem = names.length;
        }
      }
      return aInfoArray;
    }
    return null;
  },

  getNextURNForHost: function(aHost)
  {
    aHost = aHost.replace(/^urn:/, '').replace(/:feed-(\d*)/, '');
    var indexNumber = 1;
    var currentList = new Array();

    for (i in this.feedCache) {
      if (this.feedCache[i].urn.indexOf(aHost) == 4)
        currentList.push(this.feedCache[i].urn.replace(/\D*/, ''));
    }
    if (currentList.length) {
      var gaps = new Array();
      var targetIndex = 1;

      currentList.sort();

      for (i in currentList) {
        targetIndex = Number(i + 1);

        if (Number(currentList[i]) != targetIndex)
          gaps.push(targetIndex);
      }
      if (gaps.length)
        indexNumber = gaps[0];
      else {
        indexNumber = + (++targetIndex);
      }
    }
    return ("urn:" + aHost + ":feed-" + String(indexNumber));
  },

  exportOPMLData: function()
  {
    var nsIFilePicker = Ci.nsIFilePicker;
    var filePicker = Cc["@mozilla.org/filepicker;1"].createInstance(nsIFilePicker);
    var navigatorWindow = this.getMostRecentBrowserWindow();
    filePicker.init(navigatorWindow, this._bundle.GetStringFromName("exportOPML"), nsIFilePicker.modeSave);

    filePicker.appendFilter(this._bundle.GetStringFromName("OPMLFiles"), "*.opml");
    filePicker.appendFilter(this._bundle.GetStringFromName("XMLFiles"), "*.opml; *.xml; *.rdf; *.rss; *.html; *.htm");
    filePicker.appendFilter(this._bundle.GetStringFromName("AllFiles"), "*");

    filePicker.defaultString = "mzWebFeeds.opml";
		
    if (filePicker.show() == nsIFilePicker.returnCancel)
      return false;
    
    var names = [ "xmlUrn", "type", "xmlUrl", "title", "xmlHome", "text", "date", "lastModified", 
                  "image", "imageUrl", "imageTitle", "imageWidth", "imageHeight" ];

    var str = '<?xml version="1.0" encoding="UTF-8"?>\n' +
              '<opml version="2.0">\n' +
              '<head>\n' +
              '\t<title>MultiZilla OPML Export v0.1</title>\n' +
              '\t<dateCreated>' + new Date().toString() + '</dateCreated>\n' +
              '</head>\n' +
              '<body>\n';

    var outputStream = Cc['@mozilla.org/network/file-output-stream;1'].createInstance(Ci.nsIFileOutputStream);
    outputStream.init(filePicker.file, 0x04 | 0x08 | 0x20, 420, 0 ); 
    outputStream.write(str, str.length);

    var serializer = Cc["@mozilla.org/xmlextras/xmlserializer;1"].createInstance(Ci.nsIDOMSerializer);
    var feedCategories = this.getFeedCategories();
    var hasFeedCategories = (feedCategories.length > 0);
    var serializedFeeds = new Array();
    var _this = this;

    function _dataCollector(aFeedResource)
    {
      var feedInfo = new Array();
      feedInfo[0] = aFeedResource;
      feedInfo = _this.getFeedInfoFromRDF(feedInfo);

      if (feedInfo) {
        var outline = navigatorWindow.document.createElementNS(null, "outline");

        for (var n = 0; n < feedInfo.length; n++) {
          if (feedInfo[n] != "")
            outline.setAttribute(names[n], feedInfo[n]);
        }
        str = serializer.serializeToString(outline);
        return '\t' + str + '\n';
      }
      return '';
    }

    for (i in feedCategories) {
      var feedCategory = feedCategories[i];
      var feeds = this.getFeedsByCategory(feedCategory);

      str = '\t<outline text="' + this.getFeedCategoryName(feedCategory) + '">\n';
      outputStream.write(str, str.length);

      for (i in feeds) {
        var feed = feeds[i];
        serializedFeeds.push(feed);
        str = '\t' + _dataCollector(feed);
        outputStream.write(str, str.length);
      }

      str = '\t</outline>\n';
      outputStream.write(str, str.length);
    }

    for (i in this.feedCache) {
      if (serializedFeeds.indexOf(this.feedCache[i].urn) == -1) {
        str = _dataCollector(this.feedCache[i].urn);
        outputStream.write(str, str.length);
      }
    }
    str = '</body>\n</opml>';
	outputStream.write(str, str.length);
    outputStream.flush();
    outputStream.close();
    return true;
  },

  importOPMLData: function()
  {
    debug("Not Yet Implemented!");
  },

  getVersionInfo: function()
  {
    return VERSION_INFO;
  },

  getFeedCategories: function()
  {
    // dump("\ngetFeedCategories");
    if (this.categoryCache == null) { // is the category cache already initialized?
      var dataSource = this.initDataSource();
      var rootContainer = this.RDFCUtils.MakeSeq(dataSource, this.RDF.GetResource("urn:feeds:root")).GetElements();

      if (!rootContainer)
        return null;

      var feedGroups = new Array();

      while (rootContainer.hasMoreElements()) {
        var resource = rootContainer.getNext().QueryInterface(Ci.nsIRDFResource).Value;

        if (/^urn:feedgroup:/.test(resource))
          feedGroups.push(resource);
      }
      this.categoryCache = feedGroups; // initialize the category cache
      return (feedGroups.length) ? feedGroups : null;
    }
    return this.categoryCache; // return cached entries
  },

  getFeedsByCategory: function(aContainerResource)
  {
    return this.getResourcesFromContainer(this.RDF.GetResource(aContainerResource));
  },

  getCategoryByFeed: function(aFeed)
  {
    var feedCategories = this.getFeedCategories();
    var dataSource = this.subscriptionsDS; // initialized by 'getFeedCategoryies'
    var targetResource = this.RDF.GetResource(aFeed);

    if (feedCategories) {
      for (i in feedCategories) {
        var resourceString = feedCategories[i];
        var resource = this.RDF.GetResource(resourceString);
        var container = this.RDFCUtils.MakeSeq(dataSource, resource);

        if (container.IndexOf(targetResource) >= 0)
          return resourceString;
      }
    }
    var urn = "urn:feeds:root";
    var rootContainer = this.RDFCUtils.MakeSeq(dataSource, this.RDF.GetResource(urn));

    if (rootContainer && rootContainer.IndexOf(targetResource) >= 0)
      return urn;

    return null;
  },

  getFeedSubscriptions: function()
  {
    // dump("\ngetFeedSubscriptions-in : " + this.feedCache);
    if (this.feedCache == null) { // is the feed cache already initialized?
      this.feedCache = new Array();
      var dataSource = this.initDataSource();
      var feedList = this.getResourcesFromContainer(this.RDF.GetResource("urn:feeds:root"));
      var property = this.RDF.GetResource(this.NS + "url");

      for (i in feedList) {
        var url, urn = feedList[i];
        var feedResource = this.RDF.GetResource(urn);
        var urlNode = dataSource.GetTarget(feedResource, property, true);

        if (urlNode instanceof Ci.nsIRDFNode)
          url = urlNode.QueryInterface(Ci.nsIRDFLiteral).Value.replace(/www\./,'');
        this.feedCache.push({urn:feedList[i], url: url});
      }
    }
    // dump("\ngetFeedSubscriptions-out: " + this.feedCache.length);
    return this.feedCache;
  },

  getResourcesFromContainer: function(aContainerResource)
  {
    var _this = this;
    var resourceList = new Array();
    var dataSource = this.initDataSource();

    function _inner(aContainerResource) {
      var container = _this.RDFCUtils.MakeSeq(dataSource, aContainerResource).GetElements();

      if (!container)
        return -1;

      while (container.hasMoreElements()) {
        try {
          var resource = container.getNext().QueryInterface(Ci.nsIRDFResource);

          if (/^urn:feedgroup:/.test(resource.Value))
            _inner(resource);
          else
            resourceList.push(resource.Value);
        } catch(ex) {
          dump("\ngetResourcesFromContainer - ex: " + ex);
        }
      }
      return (resourceList.length) ? resourceList : null; // .sort();
    }
    return _inner(aContainerResource);
  },

  getMostRecentBrowserWindow: function()
  {
    return this._getMostRecentWindow("navigator:browser");
  },

  addOrChangeProperty: function(aDataSource, aResource, aProperty, aValue)
  {
   /* dump("\n\naddOrChangeProperty - aResource: " + aResource.Value +
         "\n\naddOrChangeProperty - aProperty: " + aResource.Value); */
    var currentValue = aDataSource.GetTarget(aResource, aProperty, true) || null;
    var newValue = this.RDF.GetLiteral(aValue);

    this.addOrChangeValue(aDataSource, aResource, aProperty, currentValue, newValue);
  },

  addOrChangeValue: function(aDataSource, aResource, aProperty, aCurrentValue, aValue)
  {
    if (aCurrentValue)
      aDataSource.Change(aResource, aProperty, aCurrentValue, aValue);
    else
      aDataSource.Assert(aResource, aProperty, aValue, true);
    this.flushDataSource(aDataSource);
  },

  flushDataSource: function(aDataSource)
  {
    try {
      aDataSource.QueryInterface(Ci.nsIRDFRemoteDataSource).Flush();
    } catch(ex) {
      // die silently
    }
  },

  _GetResource: function(aResource)
  {
    if (typeof aResource == "string")
      aResource = this.RDF.GetResource(aResource);
    return aResource;
  },

  _getMostRecentWindow: function(aWindowType)
  {
    return this._nsIWindowMediator.getMostRecentWindow(aWindowType);
  },

  _getPref: function(aName, aDefault) {
    var prefBranch = this.prefBranch;
    try {
      switch (prefBranch.getPrefType(aName))
      {
        case prefBranch.PREF_STRING:
             return prefBranch.getCharPref(aName);
        case prefBranch.PREF_BOOL:
             return prefBranch.getBoolPref(aName);
        case prefBranch.PREF_INT:
             return prefBranch.getIntPref(aName);
        default:
             return aDefault;
      }
    }
    catch(ex) {
      return aDefault;
    }
  },

  QueryInterface: function(aIID)
  {
    if (aIID.equals(Ci.nsIWebFeedService) ||
        aIID.equals(Ci.nsIObserver) ||
        aIID.equals(Ci.nsIProfileStartupListener) ||
        aIID.equals(Ci.nsISupports))
          return this;
    Components.returnCode = Cr.NS_ERROR_NO_INTERFACE;
    return null;
  }
};

/* :::::::: Service Registration & Initialization ::::::::::::::: */

/* ........ nsIModule .............. */

const mzWebFeedModule = 
{
  mClassName:     "MultiZilla Web Feed Service",
  mContractID:    "@multizilla.org/webfeed-service;1",
  mClassID:       Components.ID("26acb1f0-28fc-43bc-867a-a46aabc85dd4"),

  getClassObject: function(aCompMgr, aCID, aIID)
  {
    if (!aCID.equals(this.mClassID))
      throw Cr.NS_ERROR_NO_INTERFACE;
    if (!aIID.equals(Ci.nsIFactory))
      throw Cr.NS_ERROR_NOT_IMPLEMENTED;

    return this.mFactory;
  },

  registerSelf: function(aCompMgr, aFileSpec, aLocation, aType)
  {
    debug("Registering...\n");
    aCompMgr = aCompMgr.QueryInterface(Ci.nsIComponentRegistrar);
    aCompMgr.registerFactoryLocation(this.mClassID, this.mClassName, this.mContractID, aFileSpec, aLocation, aType);
    this.getCategoryManager().addCategoryEntry("profile-startup-category", this.mContractID, "", true, true);
  },

  unregisterSelf: function(aCompMgr, aFileSpec, aLocation)
  {
    debug("Unregistering...\n");
    aCompMgr = aCompMgr.QueryInterface(Ci.nsIComponentRegistrar);
    aCompMgr.unregisterFactoryLocation(this.mClassID, aFileSpec);
    this.getCategoryManager().deleteCategoryEntry("profile-startup-category", this.mContractID, true);
  },

  canUnload: function(aCompMgr)
  {
    return true;
  },

  getCategoryManager: function()
  {
    return Cc["@mozilla.org/categorymanager;1"].getService(Ci.nsICategoryManager);
  },

  /* ........ nsIFactory .............. */

  mFactory:
  {
    createInstance: function(aOuter, aIID)
    {
      if (aOuter != null)
        throw Cr.NS_ERROR_NO_AGGREGATION;
      if (!aIID.equals(Ci.nsIWebFeedService) && 
          !aIID.equals(Ci.nsIObserver) &&
          !aIID.equals(Ci.nsIProfileStartupListener) &&
          !aIID.equals(Ci.nsISupports))
        throw Cr.NS_ERROR_INVALID_ARG;

      return mzWebFeedService.QueryInterface(aIID);
    },

    lockFactory: function(aLock)
    {
      // quiten warnings
    }
  }
};

function NSGetModule(aCompMgr, aFileSpec)
{
  return mzWebFeedModule;
}

