package the.bytecode.club.bytecodeviewer.malwarescanner;

import org.apache.commons.text.WordUtils;
import org.objectweb.asm.tree.ClassNode;
import the.bytecode.club.bytecodeviewer.malwarescanner.impl.AWTRobotScanner;
import the.bytecode.club.bytecodeviewer.malwarescanner.impl.JavaIOScanner;
import the.bytecode.club.bytecodeviewer.malwarescanner.impl.JavaNetScanner;
import the.bytecode.club.bytecodeviewer.malwarescanner.impl.JavaRuntimeScanner;
import the.bytecode.club.bytecodeviewer.malwarescanner.impl.NullSecurityManagerScanner;
import the.bytecode.club.bytecodeviewer.malwarescanner.impl.ReflectionScanner;
import the.bytecode.club.bytecodeviewer.malwarescanner.impl.URLScanner;

/***************************************************************************
 * Bytecode Viewer (BCV) - Java & Android Reverse Engineering Suite        *
 * Copyright (C) 2014 Kalen 'Konloch' Kinloch - http://bytecodeviewer.com  *
 *                                                                         *
 * This program is free software: you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 ***************************************************************************/

/**
 * All of the installed malware scan modules
 *
 * @author Konloch
 * @since 6/27/2021
 */
public enum MalwareScanModule
{
	URL_SCANNER("Scan String URLs", new URLScanner(), true),
	REFLECTION_SCANNER("Scan Java Reflection", new ReflectionScanner(), false),
	JAVA_RUNTIME_SCANNER("Scan Java Runtime", new JavaRuntimeScanner(), true),
	JAVA_NET_SCANNER("Scan Java Net", new JavaNetScanner(), false),
	JAVA_IO_SCANNER("Scan Java IO", new JavaIOScanner(), false),
	AWT_ROBOT_SCANNER("Scan AWT Robot", new AWTRobotScanner(), true),
	NULL_SECURITY_MANAGER("Scan Null SecurityManager", new NullSecurityManagerScanner(), true),
	;
	
	static
	{
		for(MalwareScanModule module : values())
			module.malwareScanner.module = module;
	}
	
	private final String readableName;
	private final String optionText;
	private final MalwareCodeScanner malwareScanner;
	private final boolean toggledByDefault;
	
	MalwareScanModule(String optionText, MalwareCodeScanner malwareScanner, boolean toggledByDefault)
	{
		this.optionText = optionText;
		this.malwareScanner = malwareScanner;
		this.toggledByDefault = toggledByDefault;
		this.readableName = WordUtils.capitalizeFully(name().replace("_", " ").toLowerCase());
	}
	
	public String getOptionText()
	{
		return optionText;
	}
	
	public String getReadableName()
	{
		return readableName;
	}
	
	public CodeScanner getMalwareScanner()
	{
		return malwareScanner;
	}
	
	public boolean isToggledByDefault()
	{
		return toggledByDefault;
	}
	
	public static void performScan(MalwareScan scan)
	{
		for (ClassNode cn : scan.classNodeList)
		{
			for (MalwareScanModule module : values())
				if(scan.scanOptions.contains(module.name()))
					module.malwareScanner.scanningClass(scan, cn);
		}
	}
}
