use strict;
use warnings;
package App::Addex::Entry 0.027;
# ABSTRACT: an entry in your address book

use Mixin::ExtraFields::Param -fields => {
  driver  => 'HashGuts',
  moniker => 'field',
  id      => undef,
};

use Carp ();

#pod =method new
#pod
#pod   my $entry = App::Addex::Entry->new(\%arg);
#pod
#pod This method returns an Addex Entry object.
#pod
#pod Valid parameters (sure to change) are:
#pod
#pod   name   - a full name (required)
#pod   nick   - a nickname (optional)
#pod   emails - an arrayref of email addresses (required)
#pod
#pod =cut

sub new {
  my ($class, $arg) = @_;

  # XXX: do some validation -- rjbs, 2007-04-06
  my $self = {
    name   => $arg->{name},
    nick   => $arg->{nick},
    emails => $arg->{emails},
  };

  bless $self => $class;

  $self->field(%{ $arg->{fields} }) if $arg->{fields};

  return $self;
}

#pod =method name
#pod
#pod =method nick
#pod
#pod These methods return the value of the property they name.
#pod
#pod =cut

sub name { $_[0]->{name} }
sub nick { $_[0]->{nick} }

#pod =method emails
#pod
#pod This method returns the entry's email addresses.  In scalar context it returns
#pod the number of addresses.
#pod
#pod =cut

sub emails { @{ $_[0]->{emails} } }

#pod =method field
#pod
#pod   my $value = $entry->field($name);
#pod
#pod   $entry->field($name => $value);
#pod
#pod This method is generated by L<Mixin::ExtraFields::Param|Mixin::ExtraFields::Param>.
#pod
#pod =cut

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

App::Addex::Entry - an entry in your address book

=head1 VERSION

version 0.027

=head1 PERL VERSION SUPPORT

This module has the same support period as perl itself:  it supports the two
most recent versions of perl.  (That is, if the most recently released version
is v5.40, then this module should work on both v5.40 and v5.38.)

Although it may work on older versions of perl, no guarantee is made that the
minimum required version will not be increased.  The version may be increased
for any reason, and there is no promise that patches will be accepted to lower
the minimum required perl.

=head1 METHODS

=head2 new

  my $entry = App::Addex::Entry->new(\%arg);

This method returns an Addex Entry object.

Valid parameters (sure to change) are:

  name   - a full name (required)
  nick   - a nickname (optional)
  emails - an arrayref of email addresses (required)

=head2 name

=head2 nick

These methods return the value of the property they name.

=head2 emails

This method returns the entry's email addresses.  In scalar context it returns
the number of addresses.

=head2 field

  my $value = $entry->field($name);

  $entry->field($name => $value);

This method is generated by L<Mixin::ExtraFields::Param|Mixin::ExtraFields::Param>.

=head1 AUTHOR

Ricardo SIGNES <rjbs@semiotic.systems>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2006 by Ricardo SIGNES.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
