/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* RADIO.C --- Radio buttons. */

#include <stdlib.h>
#include <graphics.h>
#include <assert.h>
#include "window.h"
#include "key.h"

enum {

radio_border_width = 1,
radio_fg_color = cBLACK,
radio_border_color = radio_fg_color,
radio_bg_color = cWHITE,
radio_disabled_color = cLIGHTGRAY,
radio_x_margin = 4,
radio_button_diam = 11,     /* Best as an odd number. */
radio_button_spacing = 12,
radio_button_button = bLEFT,
radio_focus_color = cLIGHTRED,

};

/* Return title height or 0 if no title at all. */
LOCAL(int) title_height(RADIO r)
{
  return r->title == NULL ? 0 : textheight(r->title);
}

LOCAL(void) draw_radio_entry(RADIO_ENTRY re)
{
  int rad, center_y, circle_color, text_color, dot_color;
  RADIO r;

  r = (RADIO)re->window.parent;

  assert(re->pos == re - r->entries);

  if (!w_status_p(&re->window, wsVISIBLE))
    return;

  rad = (radio_button_diam - 1) / 2;
  center_y = re->window.height / 2;

  push_graphics_state(&re->window, 0);

  /* Figure out colors. */
  if (r_status_p(r, rENABLED)) {
    if (r->pos == re->pos) {
      dot_color = radio_fg_color;
      circle_color = (r_status_p(r, rHAVE_FOCUS)) ? radio_focus_color : radio_fg_color;
    }
    else {
      dot_color =  radio_bg_color;
      circle_color = radio_fg_color;
    }
    text_color = radio_fg_color;
  }
  else {
    dot_color = radio_bg_color;
    circle_color = radio_disabled_color;
    text_color = radio_disabled_color;
  }
  setcolor(circle_color);
  circle(rad, center_y, rad);
  setcolor(text_color);
  settextjustify(LEFT_TEXT, CENTER_TEXT);
  out_hot_textxy(radio_button_diam + radio_x_margin, center_y, &re->text);
  setfillstyle(SOLID_FILL, dot_color);
  fillellipse(rad, center_y, rad - 2, rad - 2);

  pop_graphics_state();
}

LOCAL(void) draw_radio_entry_protected(RADIO_ENTRY re)
{
  protect_cursor(&re->window);
  draw_radio_entry(re);
  unprotect_cursor();
}

LOCAL(void) draw_radio(RADIO r)
{
  int i;
  RADIO_ENTRY re;

  if (!w_status_p(&r->window, wsVISIBLE))
    return;
  push_graphics_state(&r->window, 0);
  setcolor(radio_fg_color);
  protect_cursor(&r->window);
  if (r->title != NULL) {
    settextjustify(LEFT_TEXT, TOP_TEXT);
    setfillstyle(SOLID_FILL, radio_bg_color);
    bar(radio_x_margin, 
	-radio_border_width, 
	3*radio_x_margin + textwidth(r->title) - 1, textheight(r->title) - 1);
    outtextxy(2*radio_x_margin, 0, r->title);
  }
  for (i = 0, re = r->entries; i < r->n_entries; ++i, ++re)
    draw_radio_entry(re);
  unprotect_cursor();
  pop_graphics_state();
}

/* Move radio button if radio is enabled. */
LOCAL(int) move_radio_pos(RADIO_ENTRY re)
{
  int old_pos, new_pos;
  RADIO r = (RADIO)re->window.parent;

  old_pos = r->pos;
  new_pos = re->pos;

  assert(0 <= old_pos && old_pos < r->n_entries 
      && 0 <= new_pos && new_pos < r->n_entries);

  if (!r_status_p(r, rENABLED) || re->pos == r->pos)
    return 0;

  protect_cursor(&r->window);
  r->pos = new_pos;
  draw_radio_entry(&r->entries[old_pos]);
  draw_radio_entry(re);
  unprotect_cursor();
  (*r->action.code)(r->pos, r->action.env);
  return 1;
}

LOCAL(void) handle_entry_press(EVENT e)
{
  if (e->mouse.button != radio_button_button)
    return;
  move_radio_pos((RADIO_ENTRY)e->mouse.window);
}

LOCAL(void) handle_entry_hotkey(EVENT e)
{
  RADIO_ENTRY re = (RADIO_ENTRY)e->mouse.window;
  if (key_hot_p(e->hotkey.key, &re->text))
    move_radio_pos(re);
}

BeginDefDispatch(radio_entry)
  Dispatch(eBUTTON_PRESS, handle_entry_press)
  Dispatch(eVISIBLE_HOTKEY, handle_entry_hotkey)
EndDefDispatch(radio_entry)

LOCAL(void) handle_radio_gain_focus(EVENT e)
#define r ((RADIO)e->focus.window)
{
  r->status |= bit(rHAVE_FOCUS);
  draw_radio_entry_protected(&r->entries[r->pos]);
}
#undef r

LOCAL(void) handle_radio_lose_focus(EVENT e)
#define r ((RADIO)e->focus.window)
{
  r->status &= notbit(rHAVE_FOCUS);
  draw_radio_entry_protected(&r->entries[r->pos]);
}
#undef r

LOCAL(void) handle_radio_key(EVENT e)
#define r ((RADIO)e->keystroke.window)
{
  int pos;

  switch (e->keystroke.key) {

    case UP_ARROW:
      pos = r->pos - 1;
      if (pos < 0)
	pos = r->n_entries - 1;
      move_radio_pos(&r->entries[pos]);
      break;

    case DOWN_ARROW:
      pos = r->pos + 1;
      if (pos >= r->n_entries)
	pos = 0;
      move_radio_pos(&r->entries[pos]);
      break;

    case HOME:
      pos = 0;
      move_radio_pos(&r->entries[pos]);
      break;

    case END:
      pos = r->n_entries - 1;
      move_radio_pos(&r->entries[pos]);
      break;

    case '\t':
      unset_focus(e->keystroke.window);
      break;

    default:
      refuse_keystroke(e);
  }
}
#undef r

BeginDefDispatch(radio)
  DispatchAction(eMAP, draw_radio((RADIO)e->map.window))
  Dispatch(eGAIN_FOCUS, handle_radio_gain_focus)
  Dispatch(eLOSE_FOCUS, handle_radio_lose_focus)
  Dispatch(eKEYSTROKE, handle_radio_key)
EndDefDispatch(radio)

void open_radio(RADIO r, int x, int y, WINDOW parent)
{
  int i, width, height, yb;
  RADIO_ENTRY p;

  width = 0;
  for (i = 0, p = r->entries; i < r->n_entries; ++i, ++p) 
    width = max(width, textwidth(p->text.str));
  /* margin button margin text margin */
  width += 3*radio_x_margin + radio_button_diam;
  height = title_height(r) + (r->n_entries + 1) * radio_button_spacing;
  open_window(&r->window, parent, x, y, width, height, 
	      radio_border_width, radio_border_color, radio_bg_color, 
	      bit(eMAP)|bit(eGAIN_FOCUS)|bit(eLOSE_FOCUS)|bit(eKEYSTROKE));
  SetDispatch(&r->window, radio);
  for (i = 0, p = r->entries, yb = title_height(r) + radio_button_spacing; 
       i < r->n_entries; 
       ++i, ++p, yb += radio_button_spacing) {
    open_window(&p->window, &r->window, 
		radio_x_margin, yb - (radio_button_diam - 1)/2, 
		width - 2 * radio_x_margin, radio_button_diam,
		0, TRANSPARENT, TRANSPARENT, 
		bit(eBUTTON_PRESS)|bit(eVISIBLE_HOTKEY));
    SetDispatch(&p->window, radio_entry);
    p->pos = i;
    map_window(&p->window);
  }
}

#ifdef NOT_USED

void enable_radio(RADIO r)
{
  r->status |= bit(rENABLED);
  draw_radio(r);
}

void disable_radio(RADIO r)
{
  r->status &= notbit(rENABLED);
  draw_radio(r);
}

#endif

void set_radio_pos(RADIO r, int pos)
{
  if (r->pos != pos) {
    r->pos = min(max(0, pos), r->n_entries);
    draw_radio(r);
  }
}
