// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

#include "vec/exprs/vruntimefilter_wrapper.h"

#include <fmt/format.h>

#include <cstddef>

#include "util/runtime_profile.h"
#include "vec/columns/column.h"
#include "vec/columns/column_const.h"
#include "vec/core/block.h"
#include "vec/core/column_numbers.h"
#include "vec/core/column_with_type_and_name.h"
#include "vec/core/types.h"
#include "vec/data_types/data_type.h"
#include "vec/utils/util.hpp"

namespace doris {
#include "common/compile_check_begin.h"

class RowDescriptor;
class RuntimeState;
class TExprNode;

double get_in_list_ignore_thredhold(size_t list_size) {
    return std::log2(list_size + 1) / 64;
}

double get_comparison_ignore_thredhold() {
    return 0.1;
}

double get_bloom_filter_ignore_thredhold() {
    return 0.4;
}
} // namespace doris

namespace doris::vectorized {

class VExprContext;

VRuntimeFilterWrapper::VRuntimeFilterWrapper(const TExprNode& node, VExprSPtr impl,
                                             double ignore_thredhold, bool null_aware,
                                             int filter_id)
        : VExpr(node),
          _impl(std::move(impl)),
          _ignore_thredhold(ignore_thredhold),
          _null_aware(null_aware),
          _filter_id(filter_id) {
    reset_judge_selectivity();
}

Status VRuntimeFilterWrapper::prepare(RuntimeState* state, const RowDescriptor& desc,
                                      VExprContext* context) {
    RETURN_IF_ERROR_OR_PREPARED(_impl->prepare(state, desc, context));
    _expr_name = fmt::format("VRuntimeFilterWrapper({})", _impl->expr_name());
    _prepare_finished = true;
    return Status::OK();
}

Status VRuntimeFilterWrapper::open(RuntimeState* state, VExprContext* context,
                                   FunctionContext::FunctionStateScope scope) {
    DCHECK(_prepare_finished);
    RETURN_IF_ERROR(_impl->open(state, context, scope));
    _open_finished = true;
    return Status::OK();
}

void VRuntimeFilterWrapper::close(VExprContext* context,
                                  FunctionContext::FunctionStateScope scope) {
    _impl->close(context, scope);
}

Status VRuntimeFilterWrapper::execute_column(VExprContext* context, const Block* block,
                                             size_t count, ColumnPtr& result_column) const {
    DCHECK(_open_finished || block == nullptr);
    if (_judge_counter.fetch_sub(1) == 0) {
        reset_judge_selectivity();
    }
    if (_always_true) {
        size_t size = count;
        result_column = create_always_true_column(size, _data_type->is_nullable());
        COUNTER_UPDATE(_always_true_filter_rows, size);
        return Status::OK();
    } else {
        ColumnPtr arg_column = nullptr;
        RETURN_IF_ERROR(
                _impl->execute_runtime_filter(context, block, count, result_column, &arg_column));
        // bloom filter will handle null aware inside itself
        if (_null_aware && TExprNodeType::BLOOM_PRED != node_type()) {
            DCHECK(arg_column);
            change_null_to_true(result_column->assume_mutable(), arg_column);
        }

        return Status::OK();
    }
}

const std::string& VRuntimeFilterWrapper::expr_name() const {
    return _expr_name;
}

#include "common/compile_check_end.h"
} // namespace doris::vectorized
