/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine.archiving;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.iotdb.db.concurrent.IoTDBThreadPoolFactory;
import org.apache.iotdb.db.concurrent.ThreadName;
import org.apache.iotdb.db.conf.IoTDBConfig;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.engine.StorageEngine;
import org.apache.iotdb.db.engine.archiving.ArchivingOperate;
import org.apache.iotdb.db.engine.archiving.ArchivingOperateWriter;
import org.apache.iotdb.db.engine.archiving.ArchivingRecover;
import org.apache.iotdb.db.engine.archiving.ArchivingTask;
import org.apache.iotdb.db.engine.fileSystem.SystemFileFactory;
import org.apache.iotdb.db.metadata.path.PartialPath;
import org.apache.iotdb.db.qp.utils.DateTimeUtils;
import org.apache.iotdb.tsfile.utils.FilePathUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ArchivingManager {
    private static final Logger logger = LoggerFactory.getLogger(ArchivingManager.class);
    protected static IoTDBConfig config = IoTDBDescriptor.getInstance().getConfig();
    private final ReentrantLock lock = new ReentrantLock();
    private ArchivingOperateWriter logWriter;
    private final Set<ArchivingTask> archivingTasks = new TreeSet<ArchivingTask>((task1, task2) -> {
        int statusDiff = task1.getStatus().ordinal() - task2.getStatus().ordinal();
        if (statusDiff != 0) {
            return statusDiff;
        }
        long startTimeDiff = task1.getStartTime() - task2.getStartTime();
        if (startTimeDiff != 0L) {
            return startTimeDiff > 0L ? 1 : -1;
        }
        int storageGroupDiff = task1.getStorageGroup().compareTo(task2.getStorageGroup());
        if (storageGroupDiff != 0) {
            return storageGroupDiff;
        }
        long ttlDiff = task2.getTTL() - task1.getTTL();
        return ttlDiff > 0L ? 1 : -1;
    });
    private long currentTaskId = 0L;
    private ScheduledExecutorService archivingTaskCheckThread;
    private ExecutorService archivingTaskThreadPool;
    private boolean initialized = false;
    private static final long ARCHIVING_CHECK_INTERVAL = 60000L;
    private static final File LOG_FILE = SystemFileFactory.INSTANCE.getFile(Paths.get(FilePathUtils.regularizePath((String)config.getSystemDir()), "archiving", "log.bin").toString());
    private static final File ARCHIVING_LOG_DIR = SystemFileFactory.INSTANCE.getFile(Paths.get(FilePathUtils.regularizePath((String)config.getSystemDir()), "archiving", "archiving_task").toString());

    public static ArchivingManager getInstance() {
        return ArchivingManagerHolder.INSTANCE;
    }

    public void init() {
        try {
            this.lock.lock();
            if (this.initialized) {
                return;
            }
            if (ARCHIVING_LOG_DIR == null) {
                logger.error("ARCHIVING_LOG_DIR is null");
            }
            if (!ARCHIVING_LOG_DIR.exists()) {
                if (ARCHIVING_LOG_DIR.mkdirs()) {
                    logger.info("ARCHIVING_LOG_DIR {} created successfully", (Object)ARCHIVING_LOG_DIR);
                } else {
                    logger.error("ARCHIVING_LOG_DIR {} create error", (Object)ARCHIVING_LOG_DIR);
                }
            }
            if (!ARCHIVING_LOG_DIR.isDirectory()) {
                logger.error("{} already exists but is not directory", (Object)ARCHIVING_LOG_DIR);
            }
            if (!LOG_FILE.getParentFile().exists()) {
                LOG_FILE.getParentFile().mkdirs();
            }
            if (!LOG_FILE.exists()) {
                try {
                    LOG_FILE.createNewFile();
                }
                catch (IOException e) {
                    logger.error("{} log file could not be created", (Object)LOG_FILE.getName());
                }
            }
            ArchivingRecover recover = new ArchivingRecover();
            recover.recover();
            this.archivingTasks.addAll(recover.getArchivingTasks());
            this.currentTaskId = recover.getCurrentTaskId();
            try {
                this.logWriter = new ArchivingOperateWriter(LOG_FILE);
            }
            catch (FileNotFoundException e) {
                logger.error("Cannot find/create log for archiving.");
                this.lock.unlock();
                return;
            }
            this.archivingTaskCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor(ThreadName.ARCHIVING_CHECK.getName());
            this.archivingTaskCheckThread.scheduleAtFixedRate(this::checkArchivingTasks, 60000L, 60000L, TimeUnit.MILLISECONDS);
            this.archivingTaskThreadPool = IoTDBThreadPoolFactory.newFixedThreadPool(config.getArchivingThreadNum(), ThreadName.ARCHIVING_TASK.getName());
            logger.info("start archiving check thread successfully.");
            this.initialized = true;
        }
        finally {
            this.lock.unlock();
        }
    }

    public void close() {
        this.initialized = false;
        this.archivingTaskCheckThread.shutdown();
        this.archivingTaskThreadPool.shutdown();
        try {
            this.logWriter.close();
        }
        catch (Exception e) {
            logger.error("Cannot close archiving log writer, because:", (Throwable)e);
        }
        for (ArchivingTask task : this.archivingTasks) {
            task.close();
        }
        this.archivingTasks.clear();
        this.currentTaskId = 0L;
    }

    public void clear() {
        this.close();
        LOG_FILE.delete();
    }

    public List<ArchivingTask> getArchivingTasks() {
        try {
            this.lock.lock();
            ArrayList<ArchivingTask> arrayList = new ArrayList<ArchivingTask>(this.archivingTasks);
            return arrayList;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean setArchiving(PartialPath storageGroup, File targetDir, long ttl, long startTime) {
        try {
            this.lock.lock();
            for (ArchivingTask archivingTask : this.archivingTasks) {
                if (!archivingTask.isActive() || !archivingTask.getStorageGroup().getFullPath().equals(storageGroup.getFullPath()) || !archivingTask.getTargetDir().equals(targetDir) || archivingTask.getTTL() != ttl || archivingTask.getStartTime() != startTime) continue;
                logger.warn("Fail to set archiving task, it's same as the archiving task {}", (Object)archivingTask.getTaskId());
                boolean bl = false;
                return bl;
            }
            ArchivingTask newTask = new ArchivingTask(this.currentTaskId, storageGroup, targetDir, ttl, startTime);
            try {
                this.logWriter.log(ArchivingOperate.ArchivingOperateType.SET, newTask);
            }
            catch (IOException e) {
                logger.error("write log error");
                boolean bl = false;
                this.lock.unlock();
                return bl;
            }
            this.archivingTasks.add(newTask);
            ++this.currentTaskId;
            boolean bl = true;
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    private ArchivingTask.ArchivingTaskStatus statusFromOperateType(ArchivingOperate.ArchivingOperateType archivingOperateType) {
        switch (archivingOperateType) {
            case RESUME: {
                return ArchivingTask.ArchivingTaskStatus.READY;
            }
            case CANCEL: {
                return ArchivingTask.ArchivingTaskStatus.CANCELED;
            }
            case START: {
                return ArchivingTask.ArchivingTaskStatus.RUNNING;
            }
            case PAUSE: {
                return ArchivingTask.ArchivingTaskStatus.PAUSED;
            }
            case FINISHED: {
                return ArchivingTask.ArchivingTaskStatus.FINISHED;
            }
            case ERROR: {
                return ArchivingTask.ArchivingTaskStatus.ERROR;
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean operate(ArchivingOperate.ArchivingOperateType archivingOperateType, long taskId) {
        try {
            this.lock.lock();
            ArchivingTask task = null;
            for (ArchivingTask archivingTask : this.archivingTasks) {
                if (archivingTask.getTaskId() != taskId) continue;
                task = archivingTask;
                break;
            }
            if (task == null) {
                boolean bl = false;
                return bl;
            }
            boolean bl = this.operate(archivingOperateType, task);
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean operate(ArchivingOperate.ArchivingOperateType archivingOperateType, PartialPath storageGroup) {
        try {
            this.lock.lock();
            ArchivingTask task = null;
            for (ArchivingTask archivingTask : this.archivingTasks) {
                if (!archivingTask.getStorageGroup().getFullPath().equals(storageGroup.getFullPath())) continue;
                task = archivingTask;
                break;
            }
            if (task == null) {
                boolean bl = false;
                return bl;
            }
            boolean bl = this.operate(archivingOperateType, task);
            return bl;
        }
        finally {
            this.lock.unlock();
        }
    }

    private boolean operate(ArchivingOperate.ArchivingOperateType archivingOperateType, ArchivingTask task) {
        switch (archivingOperateType) {
            case START: 
            case FINISHED: 
            case ERROR: 
            case SET: {
                return false;
            }
            case CANCEL: 
            case PAUSE: {
                if (task.getStatus() == ArchivingTask.ArchivingTaskStatus.READY || task.getStatus() == ArchivingTask.ArchivingTaskStatus.RUNNING) break;
                logger.warn("Cannot cancel or pause archiving task when it's in the {} status.", (Object)task.getStatus());
                return false;
            }
            case RESUME: {
                if (task.getStatus() == ArchivingTask.ArchivingTaskStatus.PAUSED) break;
                logger.warn("Cannot resume archiving task when it's in the {} status.", (Object)task.getStatus());
                return false;
            }
        }
        switch (archivingOperateType) {
            case RESUME: 
            case CANCEL: 
            case PAUSE: {
                try {
                    this.logWriter.log(archivingOperateType, task);
                }
                catch (IOException e) {
                    logger.error("write log error");
                    return false;
                }
                task.setStatus(this.statusFromOperateType(archivingOperateType));
                return true;
            }
        }
        return false;
    }

    public void checkArchivingTasks() {
        try {
            this.lock.lock();
            for (ArchivingTask task : this.archivingTasks) {
                if (task.getStartTime() - DateTimeUtils.currentTime() > 0L || task.getStatus() != ArchivingTask.ArchivingTaskStatus.READY) continue;
                if (!StorageEngine.getInstance().getProcessorMap().containsKey(task.getStorageGroup())) {
                    return;
                }
                task.setStatus(ArchivingTask.ArchivingTaskStatus.RUNNING);
                this.archivingTaskThreadPool.execute(() -> {
                    try {
                        this.logWriter.log(ArchivingOperate.ArchivingOperateType.START, task);
                        task.startTask();
                    }
                    catch (IOException e) {
                        logger.error("write log error");
                        task.setStatus(ArchivingTask.ArchivingTaskStatus.ERROR);
                        return;
                    }
                    StorageEngine.getInstance().getProcessorMap().get(task.getStorageGroup()).checkArchivingTask(task);
                    try {
                        this.logWriter.log(ArchivingOperate.ArchivingOperateType.FINISHED, task);
                        task.finish();
                    }
                    catch (IOException e) {
                        logger.error("write log error");
                        task.setStatus(ArchivingTask.ArchivingTaskStatus.ERROR);
                        return;
                    }
                    task.setStatus(ArchivingTask.ArchivingTaskStatus.FINISHED);
                });
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public void setCheckThreadTime(long checkThreadTime) {
        this.archivingTaskCheckThread.shutdown();
        this.archivingTaskCheckThread = IoTDBThreadPoolFactory.newSingleThreadScheduledExecutor(ThreadName.ARCHIVING_CHECK.getName());
        this.archivingTaskCheckThread.scheduleAtFixedRate(this::checkArchivingTasks, checkThreadTime, checkThreadTime, TimeUnit.MILLISECONDS);
    }

    private static class ArchivingManagerHolder {
        private static final ArchivingManager INSTANCE = new ArchivingManager();

        private ArchivingManagerHolder() {
        }
    }
}

