/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.taglib.core;

import org.apache.myfaces.shared_impl.util.ClassUtils;

import javax.faces.component.ActionSource;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.el.ValueBinding;
import javax.faces.event.ActionListener;
import javax.faces.webapp.UIComponentTag;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.tagext.Tag;
import javax.servlet.jsp.tagext.TagSupport;
/**
 * This tag creates an instance of the specified ActionListener, and
 * associates it with the nearest parent UIComponent.
 * 
 * Unless otherwise specified, all attributes accept static values
 * or EL expressions.
 * 
 * @JSFJspTag
 *   name="f:actionListener"
 *   bodyContent="empty"
 *   
 * @author Manfred Geiler (latest modification by $Author: lu4242 $)
 * @version $Revision: 659857 $ $Date: 2008-05-24 13:17:47 -0500 (Sat, 24 May 2008) $
 */
public class ActionListenerTag
        extends TagSupport
{
    //private static final Log log = LogFactory.getLog(ActionListenerTag.class);
    private static final long serialVersionUID = -2021978765020549175L;
    private String _type = null;

    public ActionListenerTag()
    {
    }

    /**
     * The fully qualified class name of the ActionListener class.
     * 
     * @JSFJspAttribute
     *   required="true"
     */
    public void setType(String type)
    {
        _type = type;
    }


    public int doStartTag() throws JspException
    {
        if (_type == null)
        {
            throw new JspException("type attribute not set");
        }

        //Find parent UIComponentTag
        UIComponentTag componentTag = UIComponentTag.getParentUIComponentTag(pageContext);
        if (componentTag == null)
        {
            throw new JspException("ActionListenerTag has no UIComponentTag ancestor");
        }

        if (componentTag.getCreated())
        {
            //Component was just created, so we add the Listener
            UIComponent component = componentTag.getComponentInstance();
            if (component instanceof ActionSource)
            {
                String className;
                if (UIComponentTag.isValueReference(_type))
                {
                    FacesContext facesContext = FacesContext.getCurrentInstance();
                    ValueBinding vb = facesContext.getApplication().createValueBinding(_type);
                    className = (String)vb.getValue(facesContext);
                }
                else
                {
                    className = _type;
                }
                ActionListener al = (ActionListener)ClassUtils.newInstance(className);
                ((ActionSource)component).addActionListener(al);
            }
            else
            {
                throw new JspException("Component " + component.getId() + " is no ActionSource");
            }
        }

        return Tag.SKIP_BODY;
    }
}
