/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.phoenix.filter;

import static org.apache.phoenix.util.TestUtil.TEST_PROPERTIES;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.util.Map;
import java.util.Properties;

import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.phoenix.compile.QueryPlan;
import org.apache.phoenix.end2end.Shadower;
import org.apache.phoenix.jdbc.PhoenixConnection;
import org.apache.phoenix.jdbc.PhoenixStatement;
import org.apache.phoenix.query.BaseConnectionlessQueryTest;
import org.apache.phoenix.query.ConnectionlessQueryServicesImpl;
import org.apache.phoenix.query.QueryConstants;
import org.apache.phoenix.query.QueryServices;
import org.apache.phoenix.schema.PTable;
import org.apache.phoenix.schema.PTableKey;
import org.apache.phoenix.schema.stats.GuidePostsInfo;
import org.apache.phoenix.schema.stats.GuidePostsInfoBuilder;
import org.apache.phoenix.schema.stats.GuidePostsKey;
import org.apache.phoenix.util.PropertiesUtil;
import org.apache.phoenix.util.ReadOnlyProps;
import org.junit.BeforeClass;
import org.junit.Test;

import org.apache.phoenix.thirdparty.com.google.common.collect.Maps;



public class SkipScanBigFilterTest extends BaseConnectionlessQueryTest {
    private static final byte[][] REGION_BOUNDARIES_MINIMAL = {
        Bytes.toBytesBinary("\\x06\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        };

    private static final byte[][] GUIDE_POSTS_MINIMAL = {
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0E\\x00\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x07#[j\\x80\\x00\\x00\\x00Y\\x08u\\xF3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0E\\x17\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\xD3U\\x88\\xFF\\x80\\x00\\x00\\x00\\x84\\xBFJ\\xE0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
    };

    
    private static final byte[][] REGION_BOUNDARIES_ALL = {
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D^\\x80\\x03t\\xC5DESKTOP\\x00\\x80\\x00\\x00\\x01y3\\xF7P\\x80\\x00\\x00\\x00B\\xE7\\xF6F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xB3\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00oI\\x17B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xE4\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\xFA._\\xE2\\x80\\x00\\x00\\x00\\x98\\xFE2\\xF5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0Da\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`1%"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x82\\xB4]\\xE7\\x80\\x00\\x00\\x00ER\\xFE#\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0D]\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x01p5R\\xD0\\x80\\x00\\x00\\x00@W\\xCC\\x12\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xB2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\xFE\\xC7U\\x80\\x00\\x00\\x00h\\xDF\"\\xBC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D_\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x82\\xC5\\x8E\\xB0\\x80\\x00\\x00\\x00yM\\xD7\\xFB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xB3\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x028\\xCA\\x85\\xFB\\x80\\x00\\x00\\x00}\\xA3*\\xE2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D^\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01y\\x17\\x8B<\\x80\\x00\\x00\\x00i'\\xE8\\xC4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xB4\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00oK\\x11_\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D]\\x80\\x07\\x15\\x12MOBILE\\x00\\x80\\x00\\x00\\x01a\\x02js\\x80\\x00\\x00\\x00@Y\\xC7\\x0C\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xB3\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0E\\x04@\\x8C\\x80\\x00\\x00\\x00o>\\xB1\\x1E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        };

    private static final byte[][] GUIDE_POSTS_ALL = {
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09J\\x80\\x00\\x0D\\xEF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01!y\\xC3\\x80\\x80\\x00\\x00\\x00+\\xB0)u\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09J\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\x90h\\xE8;\\x80\\x00\\x00\\x00\\x0E\\x9B\\xE7x\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09J\\x80\\x00\\x0E\\x1C\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\x14'_\\xF5\\x80\\x00\\x00\\x00(\\xF9\\xDD\\xB3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09J\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02+\\xF1\\xD8d\\x80\\x00\\x00\\x00\\x9B\\xC2A\\xD0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09J\\x80\\x00\\x0E.\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&9JM\\x80\\x00\\x00\\x00w\\x1A\\xF5\\x05\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09J\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02'\\xAAKT\\x80\\x00\\x00\\x00w\\x98{@\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09J\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&yD\\x10\\x80\\x00\\x00\\x00w'f\\x04\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09J\\x80\\x00\\x0Ec\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&\\x08\\x01\\xA1\\x80\\x00\\x00\\x00w\\x17W\\x0D\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09S\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&%U\\x1B\\x80\\x00\\x00\\x00w\\x19u\\x1C\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09S\\x80\\x00\\x0E\\x10\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00w\\xDBV\\x5C\\x80\\x00\\x00\\x00\\x14\\xE5\\xA4\\xCF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09S\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00\\xF2\\xE3\\xA1\\xD8\\x80\\x00\\x00\\x00\\x02\\x9DY\\x88\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09S\\x80\\x00\\x0E)\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02*\\xBA\\xDC\\xEF\\x80\\x00\\x00\\x00\\x99l\\x0D\\xD2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x09S\\x80\\x00\\x0E2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&h\\xC6\\x0C\\x80\\x00\\x00\\x00w\"\\xDE7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0Di\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\x82\\xA516\\x80\\x00\\x00\\x00EL\\xE1\\x8E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0Dr\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`1!\\xA1\\x80\\x00\\x00\\x00;\\xF4\\x8B\\xD4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`~.@\\x80\\x00\\x00\\x00<\\x03\\x85\\xA9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0Cc\\xAF\\x98\\x80\\x00\\x00\\x00o\\x17\\xB9\\x82\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x86\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`\\x80)\\xB7\\x80\\x00\\x00\\x00fo5]\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01y3a\\x7F\\x80\\x00\\x00\\x00X\\xC7\\xE3\\xC1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x97\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x81\\x0Bb\\x80\\x00\\x00\\x00<\\x04s\\xA9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA0\\x80\\x04\\x7F1DESKTOP\\x00\\x80\\x00\\x00\\x02\\x026U\\x05\\x80\\x00\\x00\\x00kF\\x16(\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CX\\xF1+\\x80\\x00\\x00\\x00~J\\x87\\x0B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xBD\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Jg&\\xF4\\x80\\x00\\x00\\x00o\\x10\\xC8\\x1F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xC6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01p5BN\\x80\\x00\\x00\\x00i\\x0El]\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xCE\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02 \\x10\\xC8t\\x80\\x00\\x00\\x00<\\x0C\\x10\\xCE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xD6\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02]\\xD1\\xBE7\\x80\\x00\\x00\\x00\\x8A\\xFA_\\xDC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xE0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\xD7\\xEE\\x19\\x80\\x00\\x00\\x00\\x89\\xEC\\xB4\\xCC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xEB\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01p3Ja\\x80\\x00\\x00\\x00tM{\\xBA\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x9F#n\\x80\\x00\\x00\\x00i\\xC9f\\xB2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xFE\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01a\\x02b\\xAA\\x80\\x00\\x00\\x00h\\xDF9\\xDA\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x08\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\xE3!\\xC8\\x80\\x00\\x00\\x00\\x89\\xFD\\x1D\\xBB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x13\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x99=\\x9F\\x80\\x00\\x00\\x00i\\xC2\\x9D\\x98\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x1A\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xDD\\xDC\\xB5\\x80\\x00\\x00\\x00\\x89\\xE5q=\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xDAW\\xCB\\x80\\x00\\x00\\x00\\x89\\xE89\\x05\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0E,\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02y\\xD1d+\\x80\\x00\\x00\\x00o\\x18\\xC7,\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`\\x7F&\\x16\\x80\\x00\\x00\\x00<\\x03\\xE5l\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0EB\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\xE2\\xAE\\x1E\\x80\\x00\\x00\\x00\\x89\\xFA\\xC8\\xED\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0EK\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x82\\xC6}\\xD7\\x80\\x00\\x00\\x00E^\\x83\\x8F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0ET\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02y\\x8B\\xCC\\x84\\x80\\x00\\x00\\x00o\\x1A\\xC6\\xA8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00b\\xB9\\x80\\x00\\x0Ea\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\x1DN#\\x80\\x00\\x00\\x00\\x8A\\x1B\\xF5\\xFB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00w\\x9C\\x80\\x00\\x0E\\x19\\x80\\x05\\xD8\\xC7TABLET\\x00\\x80\\x00\\x00\\x02Ar2q\\x80\\x00\\x00\\x00\\x98\\x9BF|\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00w\\x9C\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x86.\\xF0\\xF4\\x80\\x00\\x00\\x00\\x98\\x9B`1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00w\\xD8\\x80\\x00\\x0E_\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02\\x91\\xD3w\\xB3\\x80\\x00\\x00\\x00\\xA00\\x5C\\xF8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xE2\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x13D^:\\x80\\x00\\x00\\x00p\\x8F\\xA6\\x83\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x13\\x0Fq\\xA5\\x80\\x00\\x00\\x00p\\x84w\\x8B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x14a\\xB2\\xE6\\x80\\x00\\x00\\x00q\\x09\\x83\\x8A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x16\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x170f\\xF3\\x80\\x00\\x00\\x00q\\xD4u(\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x8Aj\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x14\\x83\\x88l\\x80\\x00\\x00\\x00q\\x11\\xAB\\xA5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x00\\x80\\x00\\x8Aj\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1BZ\\xE7\\x9E\\x80\\x00\\x00\\x00s~\\xF8\\x14\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09J\\x80\\x00\\x0D\\xEF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&~\\xE2\\xAB\\x80\\x00\\x00\\x00w(\\xD2N\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09J\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\x91W\\xCD\\xBE\\x80\\x00\\x00\\x00\\x0E\\xAD\\x0A~\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09J\\x80\\x00\\x0E\\x1C\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&4\\xE0\\x1A\\x80\\x00\\x00\\x00w\\x1A\\xA6\\x99\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09J\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02+\\xF5\\xF1\\xDD\\x80\\x00\\x00\\x00\\x99m6q\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09J\\x80\\x00\\x0E.\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&<\\xD6\\xB9\\x80\\x00\\x00\\x00w\\x1B4-\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09J\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02'\\xA7\\xA4\\xED\\x80\\x00\\x00\\x00w\\x97Lb\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09J\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&q\"]\\x80\\x00\\x00\\x00w$\\xD6\\x15\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09J\\x80\\x00\\x0Ec\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&\\x05y\\xEF\\x80\\x00\\x00\\x00w\\x17\\x19c\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09S\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00\\x8A\\xBB\\xB1\\xDE\\x80\\x00\\x00\\x00\\x0EAU\\xE5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09S\\x80\\x00\\x0E\\x10\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02&fS\\xE3\\x80\\x00\\x00\\x00w\"\\x8C\\xCE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09S\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00r\\xD5\\x99\\xF6\\x80\\x00\\x00\\x00\\x15,E\\xC7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09S\\x80\\x00\\x0E)\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&g.\\x93\\x80\\x00\\x00\\x00w\"\\xAA\\x03\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09S\\x80\\x00\\x0E2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&d\\x15?\\x80\\x00\\x00\\x00w\")\\xAF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x09S\\x80\\x00\\x0E>\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02+ \\x16\\x19\\x80\\x00\\x00\\x00x\\x95\\xE4\\x18\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x1A}\\x80\\x00\\x0E^\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x00\\xBB\\x92\\xA1\\x96\\x80\\x00\\x00\\x00\\x14J\\xAEd\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x1A}\\x80\\x00\\x0Ec\\x80\\x03\\x8B\\xF2MOBILE\\x00\\x80\\x00\\x00\\x02n\\x95\\xD0N\\x80\\x00\\x00\\x00\\x92\\x0DF\\xA4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00$\\x0F\\x80\\x00\\x0E_\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\xE1]\\x1D\\x80\\x80\\x00\\x00\\x00}\\x1A\\xA8e\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00$[\\x80\\x00\\x0D\\xDB\\x80\\x058\\x0BDESKTOP\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00n\\xE5\\xBF\\x15\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00$[\\x80\\x00\\x0EJ\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\x8A\\xC2\\x9F\\xFA\\x80\\x00\\x00\\x00\\x98c\\xD3D\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xE4\\x80\\x00\\x0D\\xDE\\x80\\x01i\\xF6DESKTOP\\x00\\x80\\x00\\x00\\x00\\xFEJ\\xDA\\x83\\x80\\x00\\x00\\x00\\x8D\\x8A\\xD1\\xA1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xE4\\x80\\x00\\x0E\\x00\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01s\\xAE\\x95\\xC7\\x80\\x00\\x00\\x00\\x86\\x04\\xF7[\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xE4\\x80\\x00\\x0E\\x17\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x06k\\xCB\\x13\\x80\\x00\\x00\\x00\\x84\\xC0N\\x1F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xE4\\x80\\x00\\x0E*\\x80\\x01i\\xF5TABLET\\x00\\x80\\x00\\x00\\x01s\\xAE\\x98\\xF9\\x80\\x00\\x00\\x00\\x9A\\xD4\\xF0\\xED\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xE4\\x80\\x00\\x0E2\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\xC2fi\\x16\\x80\\x00\\x00\\x00\\x97\\xE1:Z\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xE4\\x80\\x00\\x0E?\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01s\\xAEE\\x94\\x80\\x00\\x00\\x00\\x98\\xF4j\\x0A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xE4\\x80\\x00\\x0EL\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x002H\\x8C\\xF7\\x80\\x00\\x00\\x00\\x88\\xF6\\xC3F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xE4\\x80\\x00\\x0E\\x5C\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x01s\\xAD\\xDE1\\x80\\x00\\x00\\x00\\x9F\\xE1`\\x02\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xEF\\x80\\x00\\x0D\\xDF\\x80\\x01i\\xECOTHER\\x00\\x80\\x00\\x00\\x01\\xB7@\\x9C\\x89\\x80\\x00\\x00\\x00V\\x81\\x8E\\xC8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xEF\\x80\\x00\\x0E\\x08\\x80\\x01i\\xECOTHER\\x00\\x80\\x00\\x00\\x02\\x00!\\x9F\\xF3\\x80\\x00\\x00\\x00]9N\\x91\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xEF\\x80\\x00\\x0E\"\\x80\\x01i\\xEDOTHER\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x98\\xD1]\\x09\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xEF\\x80\\x00\\x0E,\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\x91\\xA6<J\\x80\\x00\\x00\\x00\\x98\\xD0\\xF5Y\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xEF\\x80\\x00\\x0E9\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x02\\x00)T\\x8C\\x80\\x00\\x00\\x00\\x9E[\\xB5I\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xEF\\x80\\x00\\x0EI\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\x91\\xA7\\xD7C\\x80\\x00\\x00\\x00\\x9F\\xAA\\x0E\\xCD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xEF\\x80\\x00\\x0EU\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x01\\xB7@\\x9Ee\\x80\\x00\\x00\\x00\\x9F\\xA9h\\xA7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00)\\xEF\\x80\\x00\\x0E`\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x00\\x95 o\\x5C\\x80\\x00\\x00\\x00\\xA3$t\\xEE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00=\\x96\\x80\\x00\\x0E\\x1E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01\\xF3\\xA9[>\\x80\\x00\\x00\\x00f\\x1A\\xB1\\xF7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0Dh\\x80\\x02p DESKTOP\\x00\\x80\\x00\\x00\\x01g\\x8B\\x81#\\x80\\x00\\x00\\x00?J\\xDC\\xA4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x02p\\xE4 ~\\x80\\x00\\x00\\x00\\x8F\\x05\\xDA\\x96\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0E+\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01o(\\x97\\xCA\\x80\\x00\\x00\\x00V\\xDF\\xC8\\x81\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0E3\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01Ud\\x1D\\xF2\\x80\\x00\\x00\\x00V\\xE0\\x95\\xB3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01I#\"6\\x80\\x00\\x00\\x00>\\x1E\\xDF\\x87\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0EC\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01$:\\x1BG\\x80\\x00\\x00\\x00V\\xDE\\xFD\\xBB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0EJ\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01KpC\\x07\\x80\\x00\\x00\\x006\\xCE}5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0EP\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01\\xC0\\xDC<\\x02\\x80\\x00\\x00\\x00X\\xAB\\xC6\\x1A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0EV\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01\\xC3B\\x82\\xA5\\x80\\x00\\x00\\x00\\x90\\x1B\\x8F-\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0E]\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\xC0\\xDCBU\\x80\\x00\\x00\\x00\\x93K\\x86\\xA3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00IN\\x80\\x00\\x0Eb\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01$\\xAE\\xD2\\x0A\\x80\\x00\\x00\\x00?J\\xDD\\x0B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x5C\\xBB\\x80\\x00\\x0D\\xFA\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02z*\\xDE2\\x80\\x00\\x00\\x00\\x92\\xFF\\xEEp\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0Dj\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x02.\\xD21\\x80\\x00\\x00\\x00kF\\x15b\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0Dr\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CIZ\\xF4\\x80\\x00\\x00\\x00o\\x1D<&\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x07\\x15\\x08OTHER\\x00\\x80\\x00\\x00\\x028\\xCDo\\xC5\\x80\\x00\\x00\\x00}\\xA3\\x88\\x7F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x0Cb\\xDFn\\x80\\x00\\x00\\x00~J\\xA7n\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x87\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01y\\x18K\\x92\\x80\\x00\\x00\\x00B\\xEE\\xF2?\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x8F\"Z\\x80\\x00\\x00\\x00<\\x06\\xDB!\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x97\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x90B\\x12\\x80\\x00\\x00\\x00<\\x07(@\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA1\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02J\\xD6\\x95\\xB6\\x80\\x00\\x00\\x00fo\\x84\\xB6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xB4\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x903r\\x80\\x00\\x00\\x00i\\x0Ep\\xB4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xBE\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01` \\xF0i\\x80\\x00\\x00\\x00;\\xF2=\\xBF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xC6\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x9DRS\\x80\\x00\\x00\\x00i\\xC2c\\x8A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xCE\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02Y\\x1DM\\xD8\\x80\\x00\\x00\\x00\\x8A\\x1Ak\\xD9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xD7\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\"\\x12\\x9A\\x80\\x00\\x00\\x00\\x8A\\x1B\\xD0P\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xE1\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xDDQ\\xAD\\x80\\x00\\x00\\x00\\x89\\xDF\\x8D!\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xEB\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02Y ,\\xDC\\x80\\x00\\x00\\x00\\x8A\\x1A\\x81\\xE4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x87\\x01\\x14\\x80\\x00\\x00\\x00fp<\\xDD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xFF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xD8\\xD6j\\x80\\x00\\x00\\x00\\x89\\xF6#\\x19\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x09\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE4\\x04\\xE6\\x80\\x00\\x00\\x00\\x89\\xFA\\xC8\\xDB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x13\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE3\\x06\\xBC\\x80\\x00\\x00\\x00\\x8A\\x09p<\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x1A\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`\\xED8\\xD8\\x80\\x00\\x00\\x00h\\xDE\\xD82\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0E#\\x80\\x07\\x15\\x04DESKTOP\\x00\\x80\\x00\\x00\\x01\\x82\\xC5\\x91\\xBF\\x80\\x00\\x00\\x00E]\\x98\\x96\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0E-\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xDAb\\x0A\\x80\\x00\\x00\\x00\\x89\\xDD\\xE2\\x16\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0E6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0C.z\\xDF\\x80\\x00\\x00\\x00o\\x1B\\xC6#\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0ED\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02YL\\xB0\\xE1\\x80\\x00\\x00\\x00\\x8A\\xB9X\\xF7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0EK\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`|\\x1C\\xD0\\x80\\x00\\x00\\x00<\\x03J_\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0EU\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0C?\\xA2#\\x80\\x00\\x00\\x00oKT#\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00b\\xB9\\x80\\x00\\x0Eb\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01_\\xDA\\xD6\\xF1\\x80\\x00\\x00\\x00;\\xF2\\x08f\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00w\\x9C\\x80\\x00\\x0E\\x1C\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x5C\\xB1-\\x83\\x80\\x00\\x00\\x00\\x8Ap74\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00w\\x9C\\x80\\x00\\x0E=\\x80\\x05\\xD8\\xD3DESKTOP\\x00\\x80\\x00\\x00\\x01\\xB8\\xC8\\x96\\xC1\\x80\\x00\\x00\\x00\\x9EHL\\x10\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00w\\xD8\\x80\\x00\\x0E`\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\xBCZiE\\x80\\x00\\x00\\x00u\\xD7\\xC8\\xA5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xE3\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1BtA\\xFE\\x80\\x00\\x00\\x00s\\x83\\xB5\\xC8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02!'\\xA4\\x13\\x80\\x00\\x00\\x00u\\x9E\\xD7l\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x14}~v\\x80\\x00\\x00\\x00q\\x10\\xE6\\xE9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x17\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x13\\x0AU\\xF5\\x80\\x00\\x00\\x00p\\x840\\x85\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x8Aj\\x80\\x00\\x0E&\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x17%\\x96\\x86\\x80\\x00\\x00\\x00q\\xD1v\\xC5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x01\\x80\\x00\\x8Aj\\x80\\x00\\x0E7\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x14\\x86\\xC2\"\\x80\\x00\\x00\\x00q\\x12<\\x81\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09J\\x80\\x00\\x0D\\xEF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x5CY;{\\x80\\x00\\x00\\x00\\x8A-\\x0A/\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09J\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x14'l\\xA2\\x80\\x00\\x00\\x00(\\xF9\\xF1{\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09J\\x80\\x00\\x0E\\x1D\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00:\\xFCmU\\x80\\x00\\x00\\x00\\x02\\x93l\\x11\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09J\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02,c\\xDE+\\x80\\x00\\x00\\x00\\x9B\\xC2e8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09J\\x80\\x00\\x0E.\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&]\\x9E\\x94\\x80\\x00\\x00\\x00w \\x90\\xA1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09J\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02'\\xA8\\x91\\xD7\\x80\\x00\\x00\\x00w\\x97\\x8D>\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09J\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&\\x80/\\xCD\\x80\\x00\\x00\\x00w)4\\x0B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09J\\x80\\x00\\x0Ec\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&]\\xA5\\xFC\\x80\\x00\\x00\\x00w \\xB2\\x0D\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09S\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&\\x18\\xCF\\x81\\x80\\x00\\x00\\x00w\\x18\\xBE\\xD7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09S\\x80\\x00\\x0E\\x10\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00w\\xDB>B\\x80\\x00\\x00\\x00\\x14\\xE5\\xA4\\x91\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09S\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&\\x04\\xDF\\x9B\\x80\\x00\\x00\\x00\\x99k\\xF2\\xF6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09S\\x80\\x00\\x0E)\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02*\\xC3\\x95*\\x80\\x00\\x00\\x00x\\x95\\xD5\\xD9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09S\\x80\\x00\\x0E2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02+!#!\\x80\\x00\\x00\\x00x\\x95\\xE5\\x06\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x09S\\x80\\x00\\x0E>\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02(\\xA1\\x873\\x80\\x00\\x00\\x00yf\\x12D\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x1A}\\x80\\x00\\x0E^\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02}r[`\\x80\\x00\\x00\\x00\\x92\\xBA\\xF0\\xB7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00\\x1A}\\x80\\x00\\x0Ec\\x80\\x03\\x8B\\xF2DESKTOP\\x00\\x80\\x00\\x00\\x012\\xCD{\\xD9\\x80\\x00\\x00\\x000[\\xA1u\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0Di\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01e]\\x08x\\x80\\x00\\x00\\x00=\\x92\\xF2-\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0Dq\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x02DTk\\x80\\x00\\x00\\x00kK\\x88r\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0Cc\\x9Dd\\x80\\x00\\x00\\x00~J\\xAC\\xFC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CWE\\xAC\\x80\\x00\\x00\\x00o\\x10\\xB7\\x15\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x86\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01y\\xA6\\xE9n\\x80\\x00\\x00\\x00X\\xCC\\xAF*\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x9DU\\x19\\x80\\x00\\x00\\x00i\\xC2^S\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x96\\x80\\x07\\x15\\x05OTHER\\x00\\x80\\x00\\x00\\x01\\x84&\\x91\\x1F\\x80\\x00\\x00\\x00O\\xF6\\xC7\\x89\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\x82f\\x02O\\x80\\x00\\x00\\x00E?\\xAF\\xAB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x02\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x82\\x96)[\\x80\\x00\\x00\\x00EE\\xF5\\x96\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09J\\x80\\x00\\x0D\\xEF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x5CY,O\\x80\\x00\\x00\\x00\\x8A-\\x09\\xE2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09J\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\x90m\\xAB\\xC6\\x80\\x00\\x00\\x00\\x0E\\x9B\\xE9X\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09J\\x80\\x00\\x0E\\x1D\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00:\\xFF\\xBA\\xA2\\x80\\x00\\x00\\x00\\x02\\x93F\\xC1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09J\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02,g\\x89+\\x80\\x00\\x00\\x00\\x9B\\xC2n'\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09J\\x80\\x00\\x0E.\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&`\\xABX\\x80\\x00\\x00\\x00w!\\x98m\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09J\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02,f\\xF9\\x7F\\x80\\x00\\x00\\x00yL\\x83_\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09J\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02+\\xE2\\xC9\\xE6\\x80\\x00\\x00\\x00yL5\\x18\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09J\\x80\\x00\\x0Ec\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&\\x08\\x01\\x96\\x80\\x00\\x00\\x00w\\x17W\\x0D\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09S\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&Q\\x143\\x80\\x00\\x00\\x00w\\x1E\\x0F\\x05\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09S\\x80\\x00\\x0E\\x10\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00r\\xD6\\x15\\xCB\\x80\\x00\\x00\\x00\\x0C\\xDB\\x92\\xFE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09S\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\xBE\\x83 9\\x80\\x00\\x00\\x00\\x0C\\xDB\\x93\\x80\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x09S\\x80\\x00\\x0E)\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02*\\xC4\\xD3\\xA7\\x80\\x00\\x00\\x00x\\x95\\xD6+\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0Di\\x80\\x03t\\xC5OTHER\\x00\\x80\\x00\\x00\\x01p+I\\xE4\\x80\\x00\\x00\\x00h\\xDE\\xDA\\xF6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0Dr\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`{\\x1D\\xF9\\x80\\x00\\x00\\x00<\\x03\\x17}\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x82\\xC7\\x85\\x80\\x00\\x00\\x00i\\xA97#\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CbU\\xD0\\x80\\x00\\x00\\x00~J\\xA3\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x86\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02\\x0CXM\\x87\\x80\\x00\\x00\\x00o\\x11TY\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01p7\\xDC\\x83\\x80\\x00\\x00\\x00@\\x5C`\\xE4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x97\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`{%\\x14\\x80\\x00\\x00\\x00<\\x03\\x1F\\x87\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\x82\\xC0\\xDE\\x1E\\x80\\x00\\x00\\x00EY\\xB2\\xF4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x82\\xC6\\x07\\xE8\\x80\\x00\\x00\\x00O\\xF6\\xE8\\xA6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xBD\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02 \\x10\\xC8\\x9F\\x80\\x00\\x00\\x00<\\x0C\\x08\\xC1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xC6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\xFF\\xB6/\\x80\\x00\\x00\\x00h\\xDF\"\\xEB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xCE\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02 \\x10\\xC9O\\x80\\x00\\x00\\x00<\\x0C\\x13\\xCC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xD6\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02]\\xBFT*\\x80\\x00\\x00\\x00\\x8A\\xFAA\\x8C\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xE0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02+h\\xBD\\xBF\\x80\\x00\\x00\\x00y\"N\\x97\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xEB\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01p4:\\xDA\\x80\\x00\\x00\\x00u\\xAE\\x95q\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x9F*\\xA8\\x80\\x00\\x00\\x00i\\xC9F\\xCE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xFE\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01e]\\x0D\\xAF\\x80\\x00\\x00\\x00=\\x94\\xC4)\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x08\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\xE02\\xE6\\x80\\x00\\x00\\x00\\x8A\\x0F6(\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x13\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\xFF\\xD19\\x80\\x00\\x00\\x00h\\xDE\\x9Df\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x1A\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xDB\\xB2\\xB5\\x80\\x00\\x00\\x00\\x89\\xE36\\xDB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xDAW\\xC3\\x80\\x00\\x00\\x00\\x89\\xE13\\x0B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0E,\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02`u&J\\x80\\x00\\x00\\x00\\x8B\\xB7)\\x8E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02y\\x8B\\xDE\\x0D\\x80\\x00\\x00\\x00o\\x11@\\x1C\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0EB\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\x1E\\x9C1\\x80\\x00\\x00\\x00\\x8A\\x1B\\xF6?\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0EK\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01e]\\xF8q\\x80\\x00\\x00\\x00i\\x0E\\xBE\\xD0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0ET\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02y\\xC7\\xE8\\xB1\\x80\\x00\\x00\\x00o\\x12\\x1B\\xA3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00b\\xB9\\x80\\x00\\x0Ea\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01h\\x13I\\x9E\\x80\\x00\\x00\\x00i\\x0D\\xD6\\xD6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00w\\x9C\\x80\\x00\\x0E\\x18\\x80\\x08]LDESKTOP\\x00\\x80\\x00\\x00\\x02m\\x07\\x90\\xC0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00w\\x9C\\x80\\x00\\x0E:\\x80\\x08]LDESKTOP\\x00\\x80\\x00\\x00\\x02\\x86/\\xEE\\x1C\\x80\\x00\\x00\\x00\\x98\\x9B\\x5C\\x88\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00w\\xD8\\x80\\x00\\x0E_\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02E\\xEC@\\x05\\x80\\x00\\x00\\x00Y\\x12\\xC4\\x16\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xE2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02!\\x18T\\xAC\\x80\\x00\\x00\\x00u\\x9C\\xDA\\xB6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xF4\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x14`\\x9C6\\x80\\x00\\x00\\x00q\\x09Hp\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x05\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x1B\\x14o\"\\x80\\x00\\x00\\x00sZ\\xFFN\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x16\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x15\\x82t{\\x80\\x00\\x00\\x00qRS\\x1B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x8Aj\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02\\x16\\xBC\\x8C|\\x80\\x00\\x00\\x00q\\xAA\\x14\\xA5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x03\\x80\\x00\\x8Aj\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x17\\x02O\\xE9\\x80\\x00\\x00\\x00q\\xC9E\\x10\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0Di\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01y41[\\x80\\x00\\x00\\x00B\\xE7\\xEDE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0Dq\\x80\\x03t\\xC5DESKTOP\\x00\\x80\\x00\\x00\\x01y6&\\x83\\x80\\x00\\x00\\x00B\\xE8\\x96\\xCB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`|\\x15<\\x80\\x00\\x00\\x00<\\x03D\\x13\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CX\\x8C\\xE1\\x80\\x00\\x00\\x00o\\x1C(\\xA7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x86\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01\\x82\\x8E'\\x99\\x80\\x00\\x00\\x00b\\xF3\\xABM\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01a\\x01\\x1E\\xB7\\x80\\x00\\x00\\x00h\\xDE\\x93\\xC5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x96\\x80\\x07\\x15\\x08OTHER\\x00\\x80\\x00\\x00\\x01\\x84!\\xA5\\x8F\\x80\\x00\\x00\\x00E\\xD7\\xA6)\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x0C+!\\x0E\\x80\\x00\\x00\\x00oI\\xEC\\xEC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x026l\\xA0\\x80\\x00\\x00\\x00EJ\\x18\\xB3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xBD\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Jg$=\\x80\\x00\\x00\\x00;\\xF2d.\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xC6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE2\\xA7\\xE7\\x80\\x00\\x00\\x00\\x8A\\x07\\x07\\x05\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xCE\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01e\\x5CF^\\x80\\x00\\x00\\x00=\\x92\\x87:\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xD7\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02J\\xD4Rm\\x80\\x00\\x00\\x00o\\x1C8\\x09\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xE1\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xD9\\xF2\\xCF\\x80\\x00\\x00\\x00\\x89\\xEE\"\\xAB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xEB\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02YH1\\xB0\\x80\\x00\\x00\\x00\\x8A\\xB8\\xE3\\x8E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02y\\x8B\\xDE\\xF0\\x80\\x00\\x00\\x00\\x903\\xAC\\xEC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xFF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02YN/5\\x80\\x00\\x00\\x00\\x8A\\x1E)\\x1F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x09\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE1\\xCBo\\x80\\x00\\x00\\x00\\x89\\xEE&M\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x13\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE0J\\x0B\\x80\\x00\\x00\\x00\\x8A\\x0An\\xA8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x1A\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x02Y\\x1D\\xFD\\xD0\\x80\\x00\\x00\\x00\\x8A\\x1B\\xF4\\x9B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02s\\x16\\xF1\\xF6\\x80\\x00\\x00\\x00\\x95R\\x03\\xD8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0E-\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y,\\xF0\\xE1\\x80\\x00\\x00\\x00\\x8A\\x1D,\\x83\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0E6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01e\\x5C8\\xE1\\x80\\x00\\x00\\x00i\\x0E\\x9D\\xE4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0EC\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\xDD\\x08\\x8B\\x80\\x00\\x00\\x00\\x89\\xEA\\x05k\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0EK\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02a\\xC9\\x1D\\x10\\x80\\x00\\x00\\x00\\x96\\x06\\x18\\xCD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0EU\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01h\\x13T\\x04\\x80\\x00\\x00\\x00i\\x0D\\xE8\\xB6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00b\\xB9\\x80\\x00\\x0Eb\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE0x\\xA4\\x80\\x00\\x00\\x00\\x89\\xDFT\\xE2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00w\\x9C\\x80\\x00\\x0E\\x1B\\x80\\x05\\xD8\\xD0DESKTOP\\x00\\x80\\x00\\x00\\x01\\xC9M\\xF4b\\x80\\x00\\x00\\x00\\x98\\x9AJy\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00w\\x9C\\x80\\x00\\x0E=\\x80\\x05\\xD8\\xC9DESKTOP\\x00\\x80\\x00\\x00\\x01\\xB8\\xC8\\x96\\xC5\\x80\\x00\\x00\\x00m\\x12\\x9DA\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00w\\xD8\\x80\\x00\\x0E`\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\xBCX8\"\\x80\\x00\\x00\\x00Wy\\x11\\xB3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xE3\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1B\\x1A\\xD6\\x99\\x80\\x00\\x00\\x00s]\\x01U\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1Bg`\\xFF\\x80\\x00\\x00\\x00s\\x82B>\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02\\x14\\x8B\\xA5\\x90\\x80\\x00\\x00\\x00q\\x13\\xF5c\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x16\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x15\\x81\\xB7m\\x80\\x00\\x00\\x00qR\\x02\\xF1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00\\x8Aj\\x80\\x00\\x0E&\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x16}In\\x80\\x00\\x00\\x00q\\x9F\\xCF\\xDF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x04\\x80\\x00\\x8Aj\\x80\\x00\\x0E6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1Bq\\xC2\\xCB\\x80\\x00\\x00\\x00s\\x83s\\x1E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09J\\x80\\x00\\x0D\\xEF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01$\\xD6\\x5Cm\\x80\\x00\\x00\\x00,\\xBE\\x1A\\xC0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09J\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\x9D\\xE5\\x15\\xA1\\x80\\x00\\x00\\x00a\\xE9D\\xD6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09J\\x80\\x00\\x0E\\x1D\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00:\\xF9^%\\x80\\x00\\x00\\x00\\x02\\x93n\\xBC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09J\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&\\x83\\xBD\\x95\\x80\\x00\\x00\\x00\\x99ki_\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09J\\x80\\x00\\x0E.\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02%\\xF8\\xF5\\xDB\\x80\\x00\\x00\\x00w\\x16Fr\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09J\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02'x\\x1D\\xA9\\x80\\x00\\x00\\x00x\\x00Fb\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09J\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&Z1\\xCB\\x80\\x00\\x00\\x00w\\x1F\\xB4\\xF2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09J\\x80\\x00\\x0Ec\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02+\\xE3b*\\x80\\x00\\x00\\x00yL8p\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09S\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02&b\\xA9H\\x80\\x00\\x00\\x00w!\\xA20\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09S\\x80\\x00\\x0E\\x10\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02&\\x00.\\xC9\\x80\\x00\\x00\\x00w\\x16\\x8F\\xB8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09S\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02*\\xCEzM\\x80\\x00\\x00\\x00\\x99l\\x16e\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09S\\x80\\x00\\x0E)\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&\\x5C\\x0B\\xAA\\x80\\x00\\x00\\x00w \\x08\\xA8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x09S\\x80\\x00\\x0E2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&\\x18\\xDD\\xFC\\x80\\x00\\x00\\x00w\\x18\\xBE\\xE9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0Di\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01e\\x5C-Y\\x80\\x00\\x00\\x00=\\x94\\xD7\\x8A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0Dq\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01zjc\\x9A\\x80\\x00\\x00\\x00i\\x0Eg$\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CX\\x87\\x01\\x80\\x00\\x00\\x00oI\\xA1D\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x82\\xC6~\\x06\\x80\\x00\\x00\\x00O\\xF6\\xA9\\xF3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x86\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01y1\\xD2\\xE8\\x80\\x00\\x00\\x00X\\xC6]1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x803}\\x80\\x00\\x00\\x00<\\x04*\\xC0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x96\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\x82\\x88\\xCF\\x83\\x80\\x00\\x00\\x00EA\\x0D\\xFB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA0\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02Jg$o\\x80\\x00\\x00\\x00;\\xF2VF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\xFE\\xA4\\x16\\x80\\x00\\x00\\x00h\\xDF-\\xB4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xBD\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0Cd\\x099\\x80\\x00\\x00\\x00~J\\xAF\\x7F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xC6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x9DM\\xDB\\x80\\x00\\x00\\x00i\\xCAK\\x0F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xCE\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01z\\x5C\\x1C\\x1D\\x80\\x00\\x00\\x00\\x7FX\\x85\\x07\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xD6\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02Y\\xD4E\\xF4\\x80\\x00\\x00\\x00\\x89\\xE1\\xA5\\xB4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xE0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`\\x85\\x05\\x94\\x80\\x00\\x00\\x00i\\xA6\\x82C\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xEB\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\xAB\\xC0\\x11\\x80\\x00\\x00\\x00<\\x0B\\xBE\\xAE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xF4\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\xDE\\x03\\xDD\\x80\\x00\\x00\\x00\\x89\\xF71o\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xFE\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02Y\\xDB\\xB4\\x19\\x80\\x00\\x00\\x00\\x89\\xE1_{\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x08\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`\\x9Dc\\xDA\\x80\\x00\\x00\\x00i\\xC9\\xEB\\x0A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x12\\x80\\x03t\\xC5DESKTOP\\x00\\x80\\x00\\x00\\x02y\\x8B\\xD9<\\x80\\x00\\x00\\x00oIQ\\x11\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x1A\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02YH;\\x0B\\x80\\x00\\x00\\x00\\x8A\\xB9\\x0B\\x01\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02J\\xD6\\xA4;\\x80\\x00\\x00\\x00fvo\\xC3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0E,\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01a$\\xD3\\xB9\\x80\\x00\\x00\\x00<&\\x98\\xCC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02X\\xB8J\\x18\\x80\\x00\\x00\\x00\\x8A\\x1B.2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0EB\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02YH\\xBB\\x10\\x80\\x00\\x00\\x00\\x8A\\x1CK5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0EK\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\xFE\\xAC\\xCF\\x80\\x00\\x00\\x00h\\xDF&V\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0ET\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xD6!^\\x80\\x00\\x00\\x00\\x8A\\x0E\\x08\\xD6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00b\\xB9\\x80\\x00\\x0Ea\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02Y\\xD7;\\x09\\x80\\x00\\x00\\x00\\x89\\xDC\\x868\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00w\\x9C\\x80\\x00\\x0E\\x18\\x80\\x05\\xD8\\xCBTABLET\\x00\\x80\\x00\\x00\\x02m\\x05\\x10\\xDC\\x80\\x00\\x00\\x00\\x98\\x9AJ\\x97\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00w\\x9C\\x80\\x00\\x0E:\\x80\\x05\\xD8\\xD0MOBILE\\x00\\x80\\x00\\x00\\x01\\xC0\\x93\\xD4\\xB4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00w\\xD8\\x80\\x00\\x0E_\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01\\xBCcx\\x86\\x80\\x00\\x00\\x00Wz\\xADo\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xE2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x15\\x83\\xA7\\x97\\x80\\x00\\x00\\x00qR\\x7Fd\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xF4\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1B_\\xB4S\\x80\\x00\\x00\\x00s\\x81%\\x04\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x05\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02\\x15\\x81\\xCD\\xAD\\x80\\x00\\x00\\x00qR\\x03\\x16\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x16\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x14hQ\\xF8\\x80\\x00\\x00\\x00q\\x0CN\\x89\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x8Aj\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1BYk\"\\x80\\x00\\x00\\x00s|\\xE9U\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x05\\x80\\x00\\x8Aj\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x14\\x88\\xEE\\xC5\\x80\\x00\\x00\\x00q\\x12\\xDD\\xDF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09J\\x80\\x00\\x0D\\xEF\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x00$\\xB5u\\x0D\\x80\\x00\\x00\\x00\\x02\\x97\\xAA\\xC5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09J\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00x{\\xB2\\xDB\\x80\\x00\\x00\\x00\\x0C\\xE3\\xB8\\x92\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09J\\x80\\x00\\x0E\\x1C\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00xn\\xAFo\\x80\\x00\\x00\\x00\\x0C\\xE3\\x87\\xD7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09J\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&~\\x14b\\x80\\x00\\x00\\x00\\x99kw\\xD1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09J\\x80\\x00\\x0E.\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&+i\\xF9\\x80\\x00\\x00\\x00w\\x19\\xD6:\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09J\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02'\\xA3\\xF1\\xBA\\x80\\x00\\x00\\x00w\\x96S\\x1D\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09J\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&h\"z\\x80\\x00\\x00\\x00w\"\\xDF\\xCD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09J\\x80\\x00\\x0Ec\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02,s.}\\x80\\x00\\x00\\x00yL\\xBBn\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09S\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00w\\xDE6\\xB2\\x80\\x00\\x00\\x00\\x0C\\xDB\\x99\\x19\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09S\\x80\\x00\\x0E\\x10\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02&`\\xBF7\\x80\\x00\\x00\\x00w!}\\xC9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09S\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00r\\xD5\\xB2\\xB7\\x80\\x00\\x00\\x00\\x0C\\xDB\\x8E\\x9B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00\\x09S\\x80\\x00\\x0E)\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02*\\xC0{G\\x80\\x00\\x00\\x00\\x99l\\x11\\x9F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0Di\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01p8kT\\x80\\x00\\x00\\x00sQ\\x05F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0Dq\\x80\\x07\\x15\\x05OTHER\\x00\\x80\\x00\\x00\\x01\\x82\\xEEN\\x97\\x80\\x00\\x00\\x00E{\\x11\\x08\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`1\\x15\\xB3\\x80\\x00\\x00\\x00;\\xF4\\x82\\xCB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CX\\xE4m\\x80\\x00\\x00\\x00~J\\x9C~\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x86\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01\\x82\\x8B9\\xA3\\x80\\x00\\x00\\x00b\\xF6R\\x9E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01o\\xBCKV\\x80\\x00\\x00\\x00fo8$\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x97\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`y\\xE2q\\x80\\x00\\x00\\x00<\\x02\\xDA\\xAC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02+h\\xB9\\xA7\\x80\\x00\\x00\\x00y\"N\\x12\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x02 \\x0E\\xEB\\x80\\x00\\x00\\x00kF\\x12\\xCA\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xBC\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x82\\xC9qi\\x80\\x00\\x00\\x00E`\\xAE_\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xC5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01zd\\xF0\\x1C\\x80\\x00\\x00\\x00B\\xED\\xFBn\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xCD\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02YKjG\\x80\\x00\\x00\\x00\\x8A\\xA07w\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xD5\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02]\\xC5{\\xD4\\x80\\x00\\x00\\x00\\x8A\\xFAMq\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xDF\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02`wo\\xC8\\x80\\x00\\x00\\x00\\x8B\\xB7\\xA6\\xCF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xEA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x85\\x13\\xC1\\x80\\x00\\x00\\x00i\\xA97\\x15\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xF3\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`\\x9F\\x1F<\\x80\\x00\\x00\\x00i\\xC9u\\x17\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xFD\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01zd]\\x0B\\x80\\x00\\x00\\x00i\\xA97\\x89\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x06\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x07\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02Y\\xE2\\x97\\x08\\x80\\x00\\x00\\x00\\x89\\xDAU\\xE0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09J\\x80\\x00\\x0D\\xEF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01/\"6\\xC4\\x80\\x00\\x00\\x00+\\xB0(\\xE7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09J\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\xF3\\x100+\\x80\\x00\\x00\\x00\\x02\\x92\\xF5[\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09J\\x80\\x00\\x0E\\x1C\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00xs\\xFFB\\x80\\x00\\x00\\x00\\x0C\\xE3\\x97\\xE8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09J\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&i\\x0C\\x00\\x80\\x00\\x00\\x00\\x9B\\xBFNN\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09J\\x80\\x00\\x0E.\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&\\x1B=L\\x80\\x00\\x00\\x00w\\x18\\xE4\\xE7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09J\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02'\\x8A\\xF7\\xC9\\x80\\x00\\x00\\x00w\\x92\\xA5R\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09J\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&x+\\x85\\x80\\x00\\x00\\x00w'\\x1C\\xAE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09J\\x80\\x00\\x0Ec\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&sI\\x84\\x80\\x00\\x00\\x00w%\\xA0U\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09S\\x80\\x00\\x0D\\xF6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00r\\xD5\\xEE\\xBE\\x80\\x00\\x00\\x00\\x0C\\xDB\\x92R\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09S\\x80\\x00\\x0E\\x10\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00\\xB9\\xF1\\xB3K\\x80\\x00\\x00\\x00\\x0C\\xDB\\x959\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09S\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&6\\x9C\\xA4\\x80\\x00\\x00\\x00\\x99k\\x8C\\xF9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09S\\x80\\x00\\x0E)\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02*\\xCE\\x90J\\x80\\x00\\x00\\x00\\x99l\\x16w\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x09S\\x80\\x00\\x0E2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02*\\xC6\\xCAN\\x80\\x00\\x00\\x00x\\x95\\xD7\\x13\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0Di\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01e]~\\xD6\\x80\\x00\\x00\\x00=\\x94\\xAC\\xD2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0Dq\\x80\\x03t\\xC5DESKTOP\\x00\\x80\\x00\\x00\\x01a\\x03\\xB3\\x1E\\x80\\x00\\x00\\x00h\\xDF2~\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x028\\xC9\\x88\\xE5\\x80\\x00\\x00\\x00}\\xA1\\xD1\\x9A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CX\\xE6\\x0D\\x80\\x00\\x00\\x00~J\\x86\\xA5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x86\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01\\x82\\x87\\x7F@\\x80\\x00\\x00\\x00b\\xF6T\\xBA\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x90D\\x80\\x80\\x00\\x00\\x00<\\x07)H\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x96\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x02C\\xE6\\x8E\\x80\\x00\\x00\\x00kKk\\xA1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`\\x8B\\xF7\\x93\\x80\\x00\\x00\\x00<\\x06\"\\xAD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01e\\x11dD\\x80\\x00\\x00\\x00=l\\x98y\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xBD\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02J\\xD4R'\\x80\\x00\\x00\\x00o\\x1A\\x0E:\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xC6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01p5I-\\x80\\x00\\x00\\x00@\\x5C`k\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xCE\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02W\\xE0l\\x84\\x80\\x00\\x00\\x00\\x8A\\xB36E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xD6\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`\\xA1Q\\xDB\\x80\\x00\\x00\\x00i\\xCA\\xEF\\x15\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xE0\\x80\\x07\\x15\\x05OTHER\\x00\\x80\\x00\\x00\\x02YJ\\x93B\\x80\\x00\\x00\\x00\\x89*\\xE4\\xE8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xEB\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02J\\xD5\\xB7\\xB6\\x80\\x00\\x00\\x00fol\\x13\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01a\\x03\\xB2\\xD2\\x80\\x00\\x00\\x00h\\xDE\\xBE5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xFE\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\xDD\\x8Dv\\x80\\x00\\x00\\x00\\x89\\xDFBB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x08\\x80\\x08~\\xD0DESKTOP\\x00\\x80\\x00\\x00\\x02&\\xBD\\x82\\x85\\x80\\x00\\x00\\x00we\\xBF\\x1A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x13\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01h\\x14\\x06.\\x80\\x00\\x00\\x00i\\x0D\\xE9=\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x1A\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02`v\\xD9g\\x80\\x00\\x00\\x00\\x8B\\xB8Pg\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE1\\xDD\\x84\\x80\\x00\\x00\\x00\\x8A\\x070\\xA3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0E,\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02b\\xC7N\\xF7\\x80\\x00\\x00\\x00\\x96\\x06\\x1BW\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02y\\x8B\\xD2!\\x80\\x00\\x00\\x00oJ\\x14\\xF1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0EB\\x80\\x04\\x7F1DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xDE/\\xEA\\x80\\x00\\x00\\x00\\x89\\xF0F0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0EK\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01p,2b\\x80\\x00\\x00\\x00h\\xDE\\xEA\\xFA\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0ET\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01a\\x01%\\xB7\\x80\\x00\\x00\\x00hm-=\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00b\\xB9\\x80\\x00\\x0Eb\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x81\\x1E\\xE1\\x80\\x00\\x00\\x00fo \\xB8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00w\\x9C\\x80\\x00\\x0E\\x19\\x80\\x05\\xD8\\xD0DESKTOP\\x00\\x80\\x00\\x00\\x02ArO\\xB3\\x80\\x00\\x00\\x00\\x80\\x0C\\x8F\\xDD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00w\\x9C\\x80\\x00\\x0E;\\x80\\x05\\xD8\\xCCDESKTOP\\x00\\x80\\x00\\x00\\x01\\xB8\\xC8\\x96\\xA0\\x80\\x00\\x00\\x00\\x9EHL\\x14\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00w\\xD8\\x80\\x00\\x0E_\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\xBC`=T\\x80\\x00\\x00\\x00WzW\\xB1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xE2\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x17c{$\\x80\\x00\\x00\\x00q\\xE2\\x0C\\x82\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x14e\\xF8z\\x80\\x00\\x00\\x00q\\x0B;I\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x15oZ\\x10\\x80\\x00\\x00\\x00qM\\xDA\\xCD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x16\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1B$1\\x08\\x80\\x00\\x00\\x00s`\\xFB\\xD9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x8Aj\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x16\\xF5\\xCB\\xE5\\x80\\x00\\x00\\x00q\\xC6\\x94\\x1A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x07\\x80\\x00\\x8Aj\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02!e\\xBF\\xBC\\x80\\x00\\x00\\x00u\\xAC\\x9E(\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09J\\x80\\x00\\x0D\\xEF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x5CX\\xE7\\xAA\\x80\\x00\\x00\\x00\\x8A,\\xA1w\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09J\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\xF3\\x0D\\xB7V\\x80\\x00\\x00\\x00\\x02\\x92\\xF5a\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09J\\x80\\x00\\x0E\\x1C\\x80\\x02\\xC0\\x0CMOBILE\\x00\\x80\\x00\\x00\\x00xn\\xF7\\xCE\\x80\\x00\\x00\\x00\\x0C\\xE3\\x89\\x03\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09J\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02,cs\\xD9\\x80\\x00\\x00\\x00\\x9B\\xC2_\\xFF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09J\\x80\\x00\\x0E.\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&iqb\\x80\\x00\\x00\\x00w#\\x17f\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09J\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02+\\xE3\\xB6\\xC5\\x80\\x00\\x00\\x00yL9\\xCF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09J\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02+\\xD2\\xD1\\x93\\x80\\x00\\x00\\x00yL }\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09J\\x80\\x00\\x0Ec\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&z*\\xE8\\x80\\x00\\x00\\x00w'\\x9EV\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09S\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&`\\xC0G\\x80\\x00\\x00\\x00w!}\\xCE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09S\\x80\\x00\\x0E\\x10\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00w\\xD9\\x86\\x03\\x80\\x00\\x00\\x00\\x0C\\xDB\\x93\\xE5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09S\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&\\x02\\xE2\\x06\\x80\\x00\\x00\\x00\\x99k\\xF2\\xCE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09S\\x80\\x00\\x0E)\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&h\\x0B~\\x80\\x00\\x00\\x00w\"\\xDE\\x07\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09S\\x80\\x00\\x0E2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&g*\\xD2\\x80\\x00\\x00\\x00w\"\\xA9\\xFD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x09S\\x80\\x00\\x0E>\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02(\\x97\\x5C$\\x80\\x00\\x00\\x00y\\x0B\\xDD0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x1A}\\x80\\x00\\x0E^\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02%\\xDA\\x9DM\\x80\\x00\\x00\\x00,\\x8F\\xAE\\xD9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x1A}\\x80\\x00\\x0Ec\\x80\\x03\\x8B\\xF3TABLET\\x00\\x80\\x00\\x00\\x00\\xF9v\\x14\\x07\\x80\\x00\\x00\\x00#g+\\xB9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00$\\x0F\\x80\\x00\\x0E_\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\xE1U\\xA3Y\\x80\\x00\\x00\\x00\\x80\\x04\\xCC\\xE6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00$[\\x80\\x00\\x0D\\xE4\\x80\\x04\\xF7\\x13DESKTOP\\x00\\x80\\x00\\x00\\x01\\x8A\\xC2o\\x08\\x80\\x00\\x00\\x00N\\xFE\\xBA\\x9F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00$[\\x80\\x00\\x0EK\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x8A\\xC2\\xED\\x08\\x80\\x00\\x00\\x00\\x98c\\x90\\xC1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0D\\xDE\\x80\\x01i\\xF6DESKTOP\\x00\\x80\\x00\\x00\\x01\\xC2g\\x19l\\x80\\x00\\x00\\x00\\x86\\x06\\xB4\\xC6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0E\\x00\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\x07#[j\\x80\\x00\\x00\\x00Y\\x08u\\xF3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0E\\x17\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\xD3U\\x88\\xFF\\x80\\x00\\x00\\x00\\x84\\xBFJ\\xE0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x00\\xCF3%\\x7F\\x80\\x00\\x00\\x00\\x98\\xFE\\xAA\\x16\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0E+\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\xCB\\xA6K\\x9C\\x80\\x00\\x00\\x00`\\xBAL\\xF8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0E2\\x80\\x01i\\xF6MOBILE\\x00\\x80\\x00\\x00\\x00\\xD4\\x7Fu*\\x80\\x00\\x00\\x00a\\x0Er1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0E?\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x8B\\x85\\x18E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x03\\x8B\\x0D\\xEC"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0EL\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x00\\xE1\\x7F\\xFD\\x03\\x80\\x00\\x00\\x00K\\x89\\xC74\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xE4\\x80\\x00\\x0E]\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\xD6\\xC5\\x06\\xA9\\x80\\x00\\x00\\x00\\x9F\\xE2\\x84\\xF3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xEF\\x80\\x00\\x0D\\xE0\\x80\\x01i\\xECTABLET\\x00\\x80\\x00\\x00\\x01\\xC3\\x1D\\xE7~\\x80\\x00\\x00\\x00\\x87\\xD8\\x06\\xFD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xEF\\x80\\x00\\x0E\\x09\\x80\\x01i\\xECDESKTOP\\x00\\x80\\x00\\x00\\x00\\x94\\xE2i2\\x80\\x00\\x00\\x00]:b\\xE0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xEF\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x028\\xC62Z\\x80\\x00\\x00\\x00\\x98\\xD1\\x5C\\x06\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xEF\\x80\\x00\\x0E,\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\xB5 r.\\x80\\x00\\x00\\x00\\x98\\xD0\\xCE\\x98\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xEF\\x80\\x00\\x0E9\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x17\\xAE\\x05\\xA0\\x80\\x00\\x00\\x00\\x9E[n~\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xEF\\x80\\x00\\x0EI\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02\\x007~\\x0A\\x80\\x00\\x00\\x00\\x9F\\xA8\\xE5\\xF0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xEF\\x80\\x00\\x0EU\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00\\x94\\xD9k\\x13\\x80\\x00\\x00\\x00\\x9F\\xA8\\xF7N\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00)\\xEF\\x80\\x00\\x0E`\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x02\\x00,\\xDC\\xC9\\x80\\x00\\x00\\x00\\x9EE\\xE3_\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00=\\x96\\x80\\x00\\x0E!\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\xF3\\xA9T\\x13\\x80\\x00\\x00\\x00\\x91\\x0F\\xD7\\xAF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00IN\\x80\\x00\\x0Du\\x80\\x02p DESKTOP\\x00\\x80\\x00\\x00\\x01yF\\xB19\\x80\\x00\\x00\\x00,\\xE6\\x8CZ\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00IN\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01$:\\x15\\x05\\x80\\x00\\x00\\x00\\x8B\\xED\\xB2\\xFD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00IN\\x80\\x00\\x0E+\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x01\\xD77n\\xFC\\x80\\x00\\x00\\x00\\x8B\\xEBP\\xEF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00IN\\x80\\x00\\x0E3\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x01\\xD77t.\\x80\\x00\\x00\\x00Y\\x0EJp\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00IN\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02;F\\xC9\\xE8\\x80\\x00\\x00\\x00\\x89\\xEA\\xF1\\xC6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00IN\\x80\\x00\\x0EC\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01f4\\x03F\\x80\\x00\\x00\\x00\\x89\\xEA\\xD7\\xF8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00IN\\x80\\x00\\x0EJ\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01\\xC3B\\x98\\xBC\\x80\\x00\\x00\\x00X\\xB04[\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00IN\\x80\\x00\\x0EP\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02CX\\xAA\\x80\\x80\\x00\\x00\\x00\\x90=3j\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0Di\\x80\\x07\\x15\\x04TABLET\\x00\\x80\\x00\\x00\\x01\\x82\\xC6\\x092\\x80\\x00\\x00\\x00E^\\x134\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0Dr\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x80\\xBA\\xB9\\x80\\x00\\x00\\x00fp\\x17\\xE9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x8D\\x1C?\\x80\\x00\\x00\\x00i\\x0Ea3\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02 \\x10\\xC9\\x0F\\x80\\x00\\x00\\x00<\\x0C\\x1E\\x1D\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x86\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`\\x99J_\\x80\\x00\\x00\\x00i\\xCAu\\xD7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01y41\\xB9\\x80\\x00\\x00\\x00X\\xCC~\\xE8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x97\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x8B\\xEB\\x08\\x80\\x00\\x00\\x00i\\x0Er\\xFB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA1\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`1&\\x81\\x80\\x00\\x00\\x00;\\xF4\\x8Ea\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CX\\xFB\\x98\\x80\\x00\\x00\\x00o\\x1Dg\\xF2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xBD\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`{\\x1C\\x8C\\x80\\x00\\x00\\x00<\\x03\\x14d\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xC5\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x02J\\xD5\\xAE\\xDA\\x80\\x00\\x00\\x00o\\x13\\xCE\\xF8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xCD\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x0CXl\\x8D\\x80\\x00\\x00\\x00o\\x14\\xBBx\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xD6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xDD\\xC3_\\x80\\x00\\x00\\x00\\x8A\\x07\\x061\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xE0\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02`v\\xEB\\x9B\\x80\\x00\\x00\\x00\\x8B\\xB8R\\xF7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xEA\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02y\\x19\\x00\\xF9\\x80\\x00\\x00\\x00oIJ\\xF9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xF4\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02K.\\xEB\\x8E\\x80\\x00\\x00\\x00h\\xDF$\\x03\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xFE\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE0\\xCC\\xF7\\x80\\x00\\x00\\x00\\x89\\xDF\\xA0N\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x08\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01e\\x5C-g\\x80\\x00\\x00\\x00\\x8C\\xC5\\x10Z\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x12\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02Y\\xE0\\xDC\\xEB\\x80\\x00\\x00\\x00\\x89\\xE1ch\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x1A\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01h\\x14\\x1A\\xE2\\x80\\x00\\x00\\x00i\\x0D\\xD3\\xFB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02s\\x14m\\xAD\\x80\\x00\\x00\\x00\\x95Q\\xEE\\x9E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0E,\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x88\\xF7_\\x80\\x00\\x00\\x00i\\xA6\\x82#\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE0D\\x94\\x80\\x00\\x00\\x00\\x89\\xE9\\xB9*\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0EG\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE1\\xF6+\\x80\\x00\\x00\\x00\\x89\\xEA\\x96X\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02YM\\x9C\\x0A\\x80\\x00\\x00\\x00\\x8A\\xB9k\\x8B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00b\\xB9\\x80\\x00\\x0EY\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02YFK4\\x80\\x00\\x00\\x00\\x8A\\x1Ed\\xFC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00w\\x9C\\x80\\x00\\x0D\\xE0\\x80\\x05\\xD8\\xCATABLET\\x00\\x80\\x00\\x00\\x01\\xB8\\xE93y\\x80\\x00\\x00\\x00V\\xB7)\\x1A\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00w\\x9C\\x80\\x00\\x0E&\\x80\\x05\\xD8\\xD2DESKTOP\\x00\\x80\\x00\\x00\\x02Ar3\\x10\\x80\\x00\\x00\\x00\\x9B\\xF3\\xE5\\xCE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00w\\x9C\\x80\\x00\\x0EN\\x80\\x05\\xD3<DESKTOP\\x00\\x80\\x00\\x00\\x01\\xB8\\xD0\\x81\\xB4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00w\\xD8\\x80\\x00\\x0Eb\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01\\xBCY\\x8E\\xCD\\x80\\x00\\x00\\x00WyB\\xE6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xE9\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x1B\\x14\\xFFf\\x80\\x00\\x00\\x00s[\\x1C\\xB4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xFB\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x15\\x5C\\xD4\\xF3\\x80\\x00\\x00\\x00qLl\\xD9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x0C\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x13FX\\x01\\x80\\x00\\x00\\x00p\\x90<\\xEA\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x1C\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1B\\x80\\xF66\\x80\\x00\\x00\\x00s\\x85V\\xC5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x8Aj\\x80\\x00\\x0E+\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x16\\x82\\xCB\\xE2\\x80\\x00\\x00\\x00q\\xA0\\x8Fr\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x08\\x80\\x00\\x8Aj\\x80\\x00\\x0ES\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x15\\x82 \\xE8\\x80\\x00\\x00\\x00qR\"E\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09J\\x80\\x00\\x0D\\xEF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01!z\\xD4\\x11\\x80\\x00\\x00\\x00+\\xB0*t\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09J\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00xy\\x0C\\xAA\\x80\\x00\\x00\\x00\\x0C\\xE3\\xADC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09J\\x80\\x00\\x0E\\x1D\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00:\\xFF\\xDB\\xBF\\x80\\x00\\x00\\x00\\x02\\x93H1\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09J\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02+\\xF2%\\xAC\\x80\\x00\\x00\\x00\\x9B\\xC1\\x9A;\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09J\\x80\\x00\\x0E.\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&_\\x95\\xAD\\x80\\x00\\x00\\x00w!W\\xD6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09J\\x80\\x00\\x0E;\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02,d\\xC8\\x18\\x80\\x00\\x00\\x00yLz.\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09J\\x80\\x00\\x0EN\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02+\\xD4\\xD6\\xAF\\x80\\x00\\x00\\x00yL#D\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09J\\x80\\x00\\x0Ec\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&\\x16*\\xC4\\x80\\x00\\x00\\x00w\\x18\\xA9\\xE9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09S\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02&d\\xBD\\xD5\\x80\\x00\\x00\\x00w\")\\xBF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09S\\x80\\x00\\x0E\\x10\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00w\\xD9w\\xD9\\x80\\x00\\x00\\x00\\x0C\\xDB\\x94\\x06\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09S\\x80\\x00\\x0E\"\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00r\\xD5\\xC4Q\\x80\\x00\\x00\\x00\\x0C\\xDB\\x93\\x9B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09S\\x80\\x00\\x0E)\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02*\\xBE}\\x14\\x80\\x00\\x00\\x00x\\x95\\xD3\\x13\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09S\\x80\\x00\\x0E2\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02&i-l\\x80\\x00\\x00\\x00w#\\x05\\xE2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x09S\\x80\\x00\\x0E>\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x02(\\xA6\\xFFl\\x80\\x00\\x00\\x00x\\xEC\\xED\\xB7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x1A}\\x80\\x00\\x0E^\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01\\x05\\xBDY\\xC4\\x80\\x00\\x00\\x00%\\xC7\\x96d\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x1A}\\x80\\x00\\x0Ec\\x80\\x03\\x8B\\xFATABLET\\x00\\x80\\x00\\x00\\x02+\\xD0Dg\\x80\\x00\\x00\\x00bn\\xEB\\x03\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00$\\x0F\\x80\\x00\\x0E_\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01}\\xA1\\x89\\x11\\x80\\x00\\x00\\x00C\\xCC\\xDB\\x1C\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00$[\\x80\\x00\\x0D\\xEB\\x80\\x04\\xF7\\x19TABLET\\x00\\x80\\x00\\x00\\x01\\x1E\\xF9\\xEE\\x12\\x80\\x00\\x00\\x00n\\xE5\\xCF\\xFD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00$[\\x80\\x00\\x0EK\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02t\\xEE\\xA8y\\x80\\x00\\x00\\x00\\x98\\x19X7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00)\\xE4\\x80\\x00\\x0D\\xDF\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x00\\xD3UgN\\x80\\x00\\x00\\x00\\x8D\\x8A\\xBC\\x90\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00)\\xE4\\x80\\x00\\x0E\\x00\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x00\\xC3?\\xAC\\x13\\x80\\x00\\x00\\x00\\x8D]\\xE3\\xE2\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00)\\xE4\\x80\\x00\\x0E\\x17\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x00\\xCE\\xEE\\x03\\xB0\\x80\\x00\\x00\\x00\\x97\\xE2W\\xC0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0Di\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\x82\\xB3B\\xB6\\x80\\x00\\x00\\x00ER?\\xE0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0Dr\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x7F*\\xFC\\x80\\x00\\x00\\x00<\\x03\\xE9\\x8F\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0Dx\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01`\\x7F\\xFBo\\x80\\x00\\x00\\x00<\\x03\\xFA\\xFE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x7F\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0Cd\\x04\\xA9\\x80\\x00\\x00\\x00~J\\xAE]\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x86\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x028\\xCC:w\\x80\\x00\\x00\\x00}\\xA2$\\xB7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x8E\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01y\\x15\\x1E{\\x80\\x00\\x00\\x00PS\\xF9\\xB9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\x97\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`|\\x0A\\xE8\\x80\\x00\\x00\\x00<\\x03:\\x02\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xA0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02J\\xD4FC\\x80\\x00\\x00\\x00o\\x15k\\xFB\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xAA\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x02C\\xAD\\x86\\x80\\x00\\x00\\x00k\\x1E(t\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xBD\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x0CX][\\x80\\x00\\x00\\x00~J\\x83=\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xC6\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\xB0\\x84>\\x80\\x00\\x00\\x00<\\x0C1\\xE7\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xCE\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02J\\xD4D\\x5C\\x80\\x00\\x00\\x00o\\x18n \\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xD6\\x80\\x00\\x00\\x01OTHER\\x00\\x80\\x00\\x00\\x02YJ\\xF25\\x80\\x00\\x00\\x00\\x89*\\xE8|\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xE0\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02Y\\xD6\\xDBa\\x80\\x00\\x00\\x00\\x89\\xE9\\xC4\\xE6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xEB\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01p*\\x8E\\xE3\\x80\\x00\\x00\\x00@Y\\x9D\\xCA\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x87\\x09x\\x80\\x00\\x00\\x00fo\\x83\\xD5\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0D\\xFE\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01e\\x5C<\\x14\\x80\\x00\\x00\\x00=\\x92\\xA6@\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x08\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02YP\\x13\\xDF\\x80\\x00\\x00\\x00\\x89*\\xD0t\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x13\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x01`\\x9F\"\\xE3\\x80\\x00\\x00\\x00i\\xC9\\x85c\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0E\\x1A\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE2\\x16&\\x80\\x00\\x00\\x00\\x89\\xE0\\xE7\\xB9\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0E#\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02Y\\xE0\\xBC\\xDC\\x80\\x00\\x00\\x00\\x8A\\x0F6\\xAF\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0E,\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02y\\x8B\\xD7\\x83\\x80\\x00\\x00\\x00o\\x1B\\xCE\\xC0\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01`y\\xFE*\\x80\\x00\\x00\\x00<\\x03\\x048\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0EB\\x80\\x07\\x15\\x08DESKTOP\\x00\\x80\\x00\\x00\\x02Y\"\\x1B%\\x80\\x00\\x00\\x00\\x8A\\x1B\\xD0s\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0EK\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02@\\xD9\\x9FC\\x80\\x00\\x00\\x00<\\x03\\x9CC\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0ET\\x80\\x00\\x00\\x01MOBILE\\x00\\x80\\x00\\x00\\x01d\\xE0#\\xD0\\x80\\x00\\x00\\x00h\\xDF\\x19\\x18\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00b\\xB9\\x80\\x00\\x0Ea\\x80\\x07\\x15\\x08DESKTOP\\x00\\x80\\x00\\x00\\x01\\x82\\xC5\\x8D\\x9E\\x80\\x00\\x00\\x00yM\\xD8\\x1B\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00w\\x9C\\x80\\x00\\x0E\\x19\\x80\\x05\\xD8\\xD1TABLET\\x00\\x80\\x00\\x00\\x01\\xC9@\\xE7\\x99\\x80\\x00\\x00\\x00\\x98\\x9AJ\\xAA\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00w\\x9C\\x80\\x00\\x0E;\\x80\\x05\\xD8\\xCADESKTOP\\x00\\x80\\x00\\x00\\x01\\xB8\\xC8\\x96\\xB2\\x80\\x00\\x00\\x00\\x9EHL\\x14\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00w\\xD8\\x80\\x00\\x0E_\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x01\\xBC[\\xC7\\x1E\\x80\\x00\\x00\\x00u\\xD7\\xC8\\xC6\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xE2\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x15lSk\\x80\\x00\\x00\\x00qM\\xC2\\x95\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x8Aj\\x80\\x00\\x0D\\xF5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x13\\x10\\x14!\\x80\\x00\\x00\\x00p\\x84\\x82\\xBD\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x06\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x15ZF\\xD4\\x80\\x00\\x00\\x00qLA\\xFE\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x8Aj\\x80\\x00\\x0E\\x16\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02\\x1B`\\x9Fw\\x80\\x00\\x00\\x00s\\x81d\\x06\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x8Aj\\x80\\x00\\x0E%\\x80\\x00\\x00\\x01TABLET\\x00\\x80\\x00\\x00\\x02\\x1B\\x16m\\xF0\\x80\\x00\\x00\\x00s[\\xB0\\xF8\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
        Bytes.toBytesBinary("\\x09\\x80\\x00\\x8Aj\\x80\\x00\\x0E5\\x80\\x00\\x00\\x01DESKTOP\\x00\\x80\\x00\\x00\\x02!e\\xD4\\x14\\x80\\x00\\x00\\x00u\\xAC\\xA1\\xF4\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x80\\x00\\x00\\x00\\x00\\x00\\x00\\x00"),
    };

    
    @BeforeClass
    @Shadower(classBeingShadowed = BaseConnectionlessQueryTest.class)
    public static void doSetup() throws Exception {
        Map<String,String> props = Maps.newHashMapWithExpectedSize(1);
        // enables manual splitting on salted tables
        props.put(QueryServices.FORCE_ROW_KEY_ORDER_ATTRIB, Boolean.toString(false));
        initDriver(new ReadOnlyProps(props.entrySet().iterator()));
    }
    
    @Test
    public void testRangeIntersect() throws Exception {
        testIntersect(REGION_BOUNDARIES_MINIMAL, GUIDE_POSTS_MINIMAL);
    }
    
    @Test
    public void testAllIntersect() throws Exception {
        testIntersect(REGION_BOUNDARIES_ALL, GUIDE_POSTS_ALL);
    }
    
    private void testIntersect(byte[][] regionBoundaries, byte[][] guidePosts) throws Exception {
        String ddl = "create table PERF.BIG_OLAP_DOC (\n" + 
        "client_id                           integer not null\n" + 
        ",customer_id                        integer\n" + 
        ",time_id                            integer not null\n" + 
        ",conversion_type_id                 integer not null\n" + 
        ",device_type                        varchar(16)\n" + 
        ",keyword_id                         bigint not null\n" + 
        ",creative_id                        bigint not null\n" + 
        ",placement_id                       bigint not null\n" + 
        ",product_target_id                  bigint not null\n" + 
        ",network                            varchar(7)\n" + 
        ",impressions                        decimal(18, 4)\n" + 
        ",publisher_clicks                   decimal(18, 4)\n" + 
        ",publisher_cost                     decimal(18, 4)\n" + 
        ",conversions                        decimal(18, 4)\n" + 
        ",revenue                            decimal(18, 4)\n" + 
        "    constraint perf_fact_pk primary key (client_id, time_id, conversion_type_id, device_type, keyword_id, creative_id, placement_id, product_target_id))SALT_BUCKETS=10";
        Properties props = PropertiesUtil.deepCopy(TEST_PROPERTIES);
        Connection conn = DriverManager.getConnection(getUrl(), props);
        StringBuilder ddlBuf = new StringBuilder(ddl + " SPLIT ON (");
        for (int i = 0; i < regionBoundaries.length; i++) {
            ddlBuf.append("?,");
        }
        ddlBuf.setCharAt(ddlBuf.length()-1, ')');;
        PreparedStatement stmt = conn.prepareStatement(ddlBuf.toString());
        int i = 1;
        for (byte[] boundary : regionBoundaries) {
            stmt.setBytes(i++, boundary);
        }
        stmt.execute();
        
        final PTable table = conn.unwrap(PhoenixConnection.class).getTable(new PTableKey(null, "PERF.BIG_OLAP_DOC"));
        GuidePostsInfoBuilder gpWriter = new GuidePostsInfoBuilder();
        for (byte[] gp : guidePosts) {
            gpWriter.trackGuidePost(new ImmutableBytesWritable(gp), 1000, 0, 0);
        }
        GuidePostsInfo info = gpWriter.build();
        PhoenixConnection pConn = conn.unwrap(PhoenixConnection.class);
        pConn.addTable(table, System.currentTimeMillis());
        ((ConnectionlessQueryServicesImpl) pConn.getQueryServices())
                .addTableStats(new GuidePostsKey(table.getName().getBytes(), QueryConstants.DEFAULT_COLUMN_FAMILY_BYTES), info);

        String query = "SELECT count(1) cnt,\n" + 
                "       coalesce(SUM(impressions), 0.0) AS \"impressions\",\n" + 
                "       coalesce(SUM(publisher_clicks), 0.0) AS \"pub_clicks\",\n" + 
                "       coalesce(SUM(publisher_cost), 0.0) AS \"pub_cost\",\n" + 
                "       coalesce(SUM(conversions), 0.0) AS \"conversions\",\n" + 
                "       coalesce(SUM(revenue), 0.0) AS \"revenue\" \n" + 
                "  FROM perf.big_olap_doc\n" + 
                " WHERE time_id between 3000 and 3700\n" + 
                "   AND network in ('SEARCH')\n" + 
                "   AND conversion_type_id = 1\n" + 
                "   AND client_id = 10724\n" + 
                "   AND device_type in ('MOBILE','DESKTOP','OTHER','TABLET')\n" + 
                "   AND keyword_id in (\n" + 
                "613214369, 613217307, 613247509, 613248897, 613250382, 613250387, 613252322, 613260252, 613261753, 613261754, 613261759, \n" + 
                "613261770, 613261873, 613261884, 613261885, 613261888, 613261889, 613261892, 613261897, 613261913, 613261919, 613261927, \n" + 
                "614496021, 843606367, 843606967, 843607021, 843607033, 843607089, 1038731600, 1038731672, 1038731673, 1038731675, \n" + 
                "1038731684, 1038731693, 1046990487, 1046990488, 1046990499, 1046990505, 1046990506, 1049724722, 1051109548, 1051311275, \n" + 
                "1051311904, 1060574377, 1060574395, 1060574506, 1060574562, 1115915938, 1115915939, 1115915941, 1116310571, 1367495544, \n" + 
                "1367495545, 1367497297, 1367497298, 1367497299, 1367497300, 1367497303, 1367497313, 1367497813, 1367497816, 1367497818, \n" + 
                "1367497821, 1367497822, 1367497823, 1624976423, 1624976451, 1624976457, 3275636061, 3275640505, 3275645765, 3275645807, \n" + 
                "3275649138, 3275651456, 3275651460, 3275651478, 3275651479, 3275654566, 3275654568, 3275654570, 3275654575, 3275659612, \n" + 
                "3275659616, 3275659620, 3275668880, 3275669693, 3275675627, 3275675634, 3275677479, 3275677504, 3275678855, 3275679524, \n" + 
                "3275679532, 3275680014, 3275682307, 3275682308, 3275682309, 3275682310, 3275682420, 3275682423, 3275682436, 3275682448, \n" + 
                "3275682460, 3275682462, 3275682474, 3275684831, 3275688903, 3275694023, 3275694025, 3275694027, 3275695054, 3275695056,\n" + 
                "3275695062, 3275699512, 3275699514, 3275699518, 3275701682, 3275701683, 3275701685, 3275701688, 3275703633, 3275703634, \n" + 
                "3275703635, 3275703636, 3275703638, 3275703639, 3275704860, 3275704861, 3275764577, 3275797149, 3275798566, 3275798567, \n" + 
                "3275798568, 3275798592, 3275931147, 3275942728, 3275945337, 3275945338, 3275945339, 3275945340, 3275945342, 3275945344, \n" + 
                "3275946319, 3275946322, 3275946324, 3275946643, 3275949495, 3275949498, 3275949500, 3275950250, 3275955128, 3275955129, \n" + 
                "3275955130, 3427017435, 3427017450, 3438304254, 3438304257, 3447068169, 3505227849, 3505227890, 3505556908, 3506351285, \n" + 
                "3506351389, 3506351398, 3506351468, 3510037138, 3510038610, 3545590644, 3545594378, 3545595073, 3545595318, 3545595506, \n" + 
                "3545597841, 3545598818, 3545599658, 3545599663, 3545601215, 3556080898, 3556080980, 3556080999, 3556081323, 3565122663, \n" + 
                "3565122679, 3565122801, 3565122858, 3565122908, 3565122929, 3565122952, 3565122984, 3565123028, 3565123047, 3565123048, \n" + 
                "3565123203, 3565123230, 3949988054, 3949988056, 3949988070, 3972992248, 3972992252, 3972992254, 3972992257, 3972992263, \n" + 
                "3972992267, 3972992268, 3972992269, 3972992270, 3972992274, 3972992275, 3972992277, 3972992281, 3972992293, 3972992298, \n" + 
                "3972992299, 3972992305, 3972992307, 3972992313, 3972992316, 3972992322, 3972992338, 3978471261, 3978471272, 4266318185, \n" + 
                "4298107404, 4308853119, 4308853123, 4308853500, 4451174646, 4451174656, 4451174701, 4569827278, 4569827284, 4569827287, \n" + 
                "4569827379, 4569827523, 4569827524, 4896589676, 4979049725, 5054587609, 5136433884, 5362640372, 5393109964, 5393405364, \n" + 
                "5393405365, 5393405620, 5393405625, 5393405675, 5393405677, 5393405858, 5393405970)";
        QueryPlan plan = conn.createStatement().unwrap(PhoenixStatement.class).compileQuery(query);
        plan.iterator();
    }

}
