/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import org.eclipse.ebam.emf.interfaces.IMessageTransaction;
import org.eclipse.ebam.model.entities.DataAttribute;
import org.eclipse.ebam.model.entities.Message;
import org.eclipse.ebam.model.entities.MessageEngine;
import org.eclipse.ebam.model.entities.SbiDomain;
import org.eclipse.ebam.model.entities.SbiThresholdValue;
import org.eclipse.ebam.model.entities.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class DataAttributeTransactions extends EMF {

	/**
	 * Logger.
	 */
	private static Logger logger = LoggerFactory
	.getLogger(DataAttributeTransactions.class);


	@SuppressWarnings("finally")
	public final DataAttribute getDataAttributeById(final Integer id) throws Exception {
		DataAttribute dataAttribute = null;
		EntityManager em = getEMF().createEntityManager();
		try {

			Query query = em
			.createQuery("select e from DataAttribute "
					+ "e where e.attributeId = :id");

			query.setParameter("id", id);

			dataAttribute = (DataAttribute) query.getSingleResult();

		} catch (NoResultException e) {
			//logger.debug(e.getMessage(), e);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw e;
		} finally {
			em.close();
			return dataAttribute;
		}
	}
	
	/**
	 *  delete a DataAttribute
	 * @param message
	 * @return
	 * @throws Exception
	 */

	public final boolean deleteDataAttribute(final DataAttribute dataAttribute) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			em.setFlushMode(FlushModeType.COMMIT);

			DataAttribute toRemove = em.find(DataAttribute.class, dataAttribute.getAttributeId());
			//toRemove.setMessages(null);
			em.merge(toRemove);
			em.remove(toRemove);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}
	
	
	/**
	 *  delete all DataAttributes associated to a message
	 * @param message
	 * @return
	 * @throws Exception
	 */

	public final void deleteDataAttributesByMessageId(Integer id) throws Exception {
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();
			em.setFlushMode(FlushModeType.COMMIT);

			Query query = em.createQuery("delete from DataAttribute da "
					+ "where da.message.messageId = :messageID");
			query.setParameter("messageID", id);
			int i = query.executeUpdate();
			em.getTransaction().commit();

			
		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
		} finally {
			em.close();
		}
	}
	

	

	/**Method that persists <code>DataAttribute</code> on database.
	 * @param j <code>DataAttribute</code> to save
	 * @return operation result
	 * @throws Exception error on saving 
	 */
	public final boolean createNewDataAttribute(final DataAttribute dataAttribute) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {

			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			em.persist(dataAttribute);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

	
	/** merge DataAttribute
	 * 
	 */
	public final boolean mergeDataAttribute(final DataAttribute dataAttribute) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			//take all contacts

			em.merge(dataAttribute);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}
	
}

