/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Marco Cortella (Engineering Group) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebam.emf;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.FlushModeType;
import javax.persistence.NoResultException;
import javax.persistence.Query;

import org.eclipse.ebam.emf.interfaces.IMessageTransaction;
import org.eclipse.ebam.emf.utilities.CacheSingleton;
import org.eclipse.ebam.model.entities.DataAttribute;
import org.eclipse.ebam.model.entities.Message;
import org.eclipse.ebam.model.entities.MessageEngine;
import org.eclipse.ebam.model.entities.SbiAlarmContact;
import org.eclipse.ebam.model.entities.SbiDomain;
import org.eclipse.ebam.model.entities.SbiResources;
import org.eclipse.ebam.model.entities.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;



public class MessageTransactions extends EMF implements IMessageTransaction{

	/**
	 * Logger.
	 */
	private static Logger logger = LoggerFactory
	.getLogger(MessageTransactions.class);



	/** Get all messages.
	 * 
	 * @return list of Messages
	 * @throws Exception
	 */

	public final List getMessageList() throws Exception {
		List messages = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			// remove from cache because could have been changed outside of program
			getEMF().getCache().evict(Message.class);	
			Query query = em.createQuery("select s from "
					+ "Message s");
			//query.setParameter("code", code);
			messages = query.getResultList();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {			
			em.close();
		}
		return messages;
	}



	/**
	 * Used to retrieve the name if the engine to fwd the message type
	 */
	// Da verificare
	public List<SbiDomain> getSbiDomainsbyMessageId(Integer idType) throws Exception {
		List<SbiDomain> result = new ArrayList<SbiDomain>();

		EntityManager em = getEMF().createEntityManager();
		try {
			Query query = em.createQuery("select m from MessageEngine m "
					+ "where m.messageId = :messageID");
			query.setParameter("messageID", idType);
			List<MessageEngine> mengine = query.getResultList();
			if (mengine != null){
				for (MessageEngine me : mengine){
					result.add(me.getSbiDomain()); 
					logger.debug("Sbi Domain value: "+ me.getSbiDomain().getValueNm());
				}
			}
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {
			em.close();
			return result;
		}
	}


	/**
	 * delete all message engine associated to a message
	 */
	// Da verificare
	public void deleteSbiDomainsByMessageId(Integer idType) throws Exception {

		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();
			em.setFlushMode(FlushModeType.COMMIT);

			Query query = em.createQuery("delete from MessageEngine m "
					+ "where m.messageId = :messageID");
			query.setParameter("messageID", idType);
			int i = query.executeUpdate();
			em.getTransaction().commit();

			
		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
		} finally {
			em.close();
		}
	}
	


	/**
	 * Method that executes database query to extract <code>DataAttribute</code>
	 * with xpath expression for specific message type.
	 * 
	 * @param msgType
	 *            messageId
	 * @return <code>DataAttribute</code>
	 * @throws Exception
	 *             error on select
	 */
	@SuppressWarnings( { "finally", "unchecked" })
	public final List<DataAttribute> getDataAttributeByMsgId(
			final Integer msgType) throws Exception {
		
		List<DataAttribute> dataAttr = null;
		if (CacheSingleton.getInstance().get(msgType.toString(), "DataAttribute") != null) {

			dataAttr = (List<DataAttribute>) CacheSingleton.getInstance().get(msgType.toString(), "DataAttribute");
			logger.info("Take DataAttribute from cache...");
		} else {

			EntityManager em = getEMF().createEntityManager();
			try {
				Query query = em.createQuery("select e from DataAttribute e "
						+ "where e.message.messageId = :messageID");
				query.setParameter("messageID", msgType);
				dataAttr = (List<DataAttribute>) query.getResultList();
				CacheSingleton.getInstance().put(msgType.toString(), dataAttr,"DataAttribute");
				logger.info("Put SbiResources from cache...");
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
			} finally {
				em.close();

			}
		}
		return dataAttr;
	}

	public final List<DataAttribute> getDataAttributeByMsgIdNoCache(
			final Integer msgType) throws Exception {

		List<DataAttribute> dataAttr = null;

		EntityManager em = getEMF().createEntityManager();
		try {
			Query query = em.createQuery("select e from DataAttribute e "
					+ "where e.message.messageId = :messageID");
			query.setParameter("messageID", msgType);
			dataAttr = (List<DataAttribute>) query.getResultList();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {
			em.close();

		}

		return dataAttr;
	}
	/**
	 * Method that executes database query to extract message id by message
	 * name.
	 * 
	 * @param msgTypeName
	 *            name
	 * @return <code>Integer</code> messageId
	 * @throws Exception
	 *             error on select
	 */
	@SuppressWarnings("finally")
	public final Integer getMsgIdByName(final String msgTypeName)
	throws Exception {
		Integer msgId = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			Query query = em.createQuery("select m.messageId from "
					+ "Message m where m.name = :name");
			query.setParameter("name", msgTypeName);
			msgId = (Integer) query.getSingleResult();

		} catch (NoResultException e) {
			//logger.debug(e.getMessage(), e);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {
			em.close();
			return msgId;
		}
	}

	/**
	 * Method that executes database query to extract <code>Message</code> by
	 * message type.
	 * 
	 * @param msgType
	 *            message type
	 * @param serviceName
	 *            service name
	 * @return <code>Message</code>
	 * @throws Exception
	 *             error on select
	 */

	@SuppressWarnings("finally")
	public final Message getMessageByType(final String msgType,
			final String serviceName) throws Exception {
		Message msg = null;

		if (CacheSingleton.getInstance().get(msgType+"_"+serviceName, "Message") != null) {

			msg = (Message) CacheSingleton.getInstance().get(msgType+"_"+serviceName, "Message");
			logger.info("Take Message from cache...");
		} else {

			EntityManager em = getEMF().createEntityManager();
			try {

				Query query = em
						.createQuery("select e from Message "
								+ "e where e.name = :name and e.service.name = :serviceName ");

				query.setParameter("name", msgType);
				query.setParameter("serviceName", serviceName);

				msg = (Message) query.getSingleResult();
				CacheSingleton.getInstance().put(msgType+"_"+serviceName, msg,"Message");
				logger.info("Put Message from cache...");
			} catch (NoResultException e) {
				// logger.debug(e.getMessage(), e);
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
				throw e;
			} finally {
				em.close();

			}
		}
		return msg;
	}





	/**
	 * Method that executes database query to the id of the typeMessage 
	 * using the xml root of the message
	 * 
	 * 
	 * @param label
	 *            to search
	 * @param serviceID
	 *            service id
	 * @return <code>ServiceInstance</code>
	 * @throws Exception
	 *             error
	 */
	public Message getMessageByXmlRoot(String root) throws Exception {

		Message message = null;
		EntityManager em = getEMF().createEntityManager();
		try {
			Query query = em
			.createQuery("select e from Message e "
					+ "where e.rootElement = :root");
			query.setParameter("root", root);
			message = (Message) query.getSingleResult();
		} catch (NoResultException e) {
			//logger.debug(e.getMessage(), e);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw e;
		} finally {
			em.close();
			return message;
		}
	}
	
	
	@SuppressWarnings("finally")
	public final Message getMessageById(final Integer id) throws Exception {
		Message msg = null;
		EntityManager em = getEMF().createEntityManager();
		try {

			Query query = em
			.createQuery("select e from Message "
					+ "e where e.messageId = :id");

			query.setParameter("id", id);

			msg = (Message) query.getSingleResult();

		} catch (NoResultException e) {
			//logger.debug(e.getMessage(), e);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw e;
		} finally {
			em.close();
			return msg;
		}
	}


	

	/**
	 *  delete a message
	 * @param message
	 * @return
	 * @throws Exception
	 */

	public final boolean deleteMessage(final Message message) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			em.setFlushMode(FlushModeType.COMMIT);

			Message toRemove = em.find(Message.class, message.getMessageId());
			//toRemove.setMessages(null);
			em.merge(toRemove);
			em.remove(toRemove);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}


	/**Method that persists <code>Message</code> on database.
	 * @param j <code>Message</code> to save
	 * @return operation result
	 * @throws Exception error on saving 
	 */
	public final boolean createNewMessage(final Message message) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			em.persist(message);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}


	/** merge an existing message
	 * 
	 * @param Message
	 * @return
	 * @throws Exception
	 */

	public final boolean mergeMessage(final Message message) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();
			// remove from cache

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			//take all contacts

			em.merge(message);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

	
	/** merge an existing message without using the cache, (used by designer)
	 * 
	 * @param Message
	 * @return
	 * @throws Exception
	 */

	public final boolean mergeMessageNoCache(final Message message) throws Exception {
		boolean isSaved = false;
		EntityManager em = getEMF().createEntityManager();
		try {
			em.getTransaction().begin();
			// remove from cache
			getEMF().getCache().evict(MessageEngine.class);	
			getEMF().getCache().evict(Message.class);	

			// Set the flush-mode so the manager query does not force a 
			// write before the read.
			em.setFlushMode(FlushModeType.COMMIT);

			//take all contacts

			em.merge(message);

			em.getTransaction().commit();

			isSaved = true;

		} catch (Exception e) {
			if (em.getTransaction() != null && em.getTransaction().isActive()) {
				em.getTransaction().rollback();
			}			
			logger.error(e.getMessage(), e);
			throw e;
		} finally {			
			em.close();			
		}
		return isSaved;
	}

	
}
