package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;
import org.mandas.docker.client.messages.Healthcheck;
import org.mandas.docker.client.messages.mount.Mount;

/**
 * Immutable implementation of {@link ContainerSpec}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableContainerSpec.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableContainerSpec implements ContainerSpec {
  private final String image;
  private final @Nullable String hostname;
  private final Map<String, String> labels;
  private final @Nullable List<String> command;
  private final @Nullable List<String> args;
  private final @Nullable List<String> env;
  private final @Nullable String dir;
  private final @Nullable String user;
  private final @Nullable List<String> groups;
  private final @Nullable Boolean tty;
  private final @Nullable List<Mount> mounts;
  private final @Nullable Long stopGracePeriod;
  private final @Nullable Healthcheck healthcheck;
  private final @Nullable List<String> hosts;
  private final @Nullable List<SecretBind> secrets;
  private final @Nullable List<ConfigBind> configs;
  private final @Nullable DnsConfig dnsConfig;
  private final Map<String, String> sysctls;
  private final @Nullable Boolean init;

  private ImmutableContainerSpec(
      String image,
      @Nullable String hostname,
      Map<String, String> labels,
      @Nullable List<String> command,
      @Nullable List<String> args,
      @Nullable List<String> env,
      @Nullable String dir,
      @Nullable String user,
      @Nullable List<String> groups,
      @Nullable Boolean tty,
      @Nullable List<Mount> mounts,
      @Nullable Long stopGracePeriod,
      @Nullable Healthcheck healthcheck,
      @Nullable List<String> hosts,
      @Nullable List<SecretBind> secrets,
      @Nullable List<ConfigBind> configs,
      @Nullable DnsConfig dnsConfig,
      Map<String, String> sysctls,
      @Nullable Boolean init) {
    this.image = image;
    this.hostname = hostname;
    this.labels = labels;
    this.command = command;
    this.args = args;
    this.env = env;
    this.dir = dir;
    this.user = user;
    this.groups = groups;
    this.tty = tty;
    this.mounts = mounts;
    this.stopGracePeriod = stopGracePeriod;
    this.healthcheck = healthcheck;
    this.hosts = hosts;
    this.secrets = secrets;
    this.configs = configs;
    this.dnsConfig = dnsConfig;
    this.sysctls = sysctls;
    this.init = init;
  }

  /**
   * @return The value of the {@code image} attribute
   */
  @JsonProperty("Image")
  @Override
  public String image() {
    return image;
  }

  /**
   * @return The value of the {@code hostname} attribute
   */
  @JsonProperty("Hostname")
  @Override
  public @Nullable String hostname() {
    return hostname;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public Map<String, String> labels() {
    return labels;
  }

  /**
   * @return The value of the {@code command} attribute
   */
  @JsonProperty("Command")
  @Override
  public @Nullable List<String> command() {
    return command;
  }

  /**
   * @return The value of the {@code args} attribute
   */
  @JsonProperty("Args")
  @Override
  public @Nullable List<String> args() {
    return args;
  }

  /**
   * @return The value of the {@code env} attribute
   */
  @JsonProperty("Env")
  @Override
  public @Nullable List<String> env() {
    return env;
  }

  /**
   * @return The value of the {@code dir} attribute
   */
  @JsonProperty("Dir")
  @Override
  public @Nullable String dir() {
    return dir;
  }

  /**
   * @return The value of the {@code user} attribute
   */
  @JsonProperty("User")
  @Override
  public @Nullable String user() {
    return user;
  }

  /**
   * @return The value of the {@code groups} attribute
   */
  @JsonProperty("Groups")
  @Override
  public @Nullable List<String> groups() {
    return groups;
  }

  /**
   * @return The value of the {@code tty} attribute
   */
  @JsonProperty("TTY")
  @Override
  public @Nullable Boolean tty() {
    return tty;
  }

  /**
   * @return The value of the {@code mounts} attribute
   */
  @JsonProperty("Mounts")
  @Override
  public @Nullable List<Mount> mounts() {
    return mounts;
  }

  /**
   * @return The value of the {@code stopGracePeriod} attribute
   */
  @JsonProperty("StopGracePeriod")
  @Override
  public @Nullable Long stopGracePeriod() {
    return stopGracePeriod;
  }

  /**
   * @return The value of the {@code healthcheck} attribute
   */
  @JsonProperty("Healthcheck")
  @Override
  public @Nullable Healthcheck healthcheck() {
    return healthcheck;
  }

  /**
   * @return The value of the {@code hosts} attribute
   */
  @JsonProperty("Hosts")
  @Override
  public @Nullable List<String> hosts() {
    return hosts;
  }

  /**
   * @return The value of the {@code secrets} attribute
   */
  @JsonProperty("Secrets")
  @Override
  public @Nullable List<SecretBind> secrets() {
    return secrets;
  }

  /**
   * @return The value of the {@code configs} attribute
   */
  @JsonProperty("Configs")
  @Override
  public @Nullable List<ConfigBind> configs() {
    return configs;
  }

  /**
   * @return The value of the {@code dnsConfig} attribute
   */
  @JsonProperty("DNSConfig")
  @Override
  public @Nullable DnsConfig dnsConfig() {
    return dnsConfig;
  }

  /**
   * @return The value of the {@code sysctls} attribute
   */
  @JsonProperty("Sysctls")
  @Override
  public Map<String, String> sysctls() {
    return sysctls;
  }

  /**
   * @return The value of the {@code init} attribute
   */
  @JsonProperty("Init")
  @Override
  public @Nullable Boolean init() {
    return init;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerSpec#image() image} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for image
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableContainerSpec withImage(String value) {
    String newValue = Objects.requireNonNull(value, "image");
    if (this.image.equals(newValue)) return this;
    return new ImmutableContainerSpec(
        newValue,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerSpec#hostname() hostname} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for hostname (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableContainerSpec withHostname(@Nullable String value) {
    if (Objects.equals(this.hostname, value)) return this;
    return new ImmutableContainerSpec(
        this.image,
        value,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object by replacing the {@link ContainerSpec#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the labels map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withLabels(Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    Map<String, String> newValue = createUnmodifiableMap(true, false, entries);
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        newValue,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#command() command}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withCommand(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableContainerSpec(
          this.image,
          this.hostname,
          this.labels,
          null,
          this.args,
          this.env,
          this.dir,
          this.user,
          this.groups,
          this.tty,
          this.mounts,
          this.stopGracePeriod,
          this.healthcheck,
          this.hosts,
          this.secrets,
          this.configs,
          this.dnsConfig,
          this.sysctls,
          this.init);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        newValue,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#command() command}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of command elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withCommand(@Nullable Iterable<String> elements) {
    if (this.command == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        newValue,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#args() args}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withArgs(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableContainerSpec(
          this.image,
          this.hostname,
          this.labels,
          this.command,
          null,
          this.env,
          this.dir,
          this.user,
          this.groups,
          this.tty,
          this.mounts,
          this.stopGracePeriod,
          this.healthcheck,
          this.hosts,
          this.secrets,
          this.configs,
          this.dnsConfig,
          this.sysctls,
          this.init);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        newValue,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#args() args}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of args elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withArgs(@Nullable Iterable<String> elements) {
    if (this.args == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        newValue,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#env() env}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withEnv(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableContainerSpec(
          this.image,
          this.hostname,
          this.labels,
          this.command,
          this.args,
          null,
          this.dir,
          this.user,
          this.groups,
          this.tty,
          this.mounts,
          this.stopGracePeriod,
          this.healthcheck,
          this.hosts,
          this.secrets,
          this.configs,
          this.dnsConfig,
          this.sysctls,
          this.init);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        newValue,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#env() env}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of env elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withEnv(@Nullable Iterable<String> elements) {
    if (this.env == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        newValue,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerSpec#dir() dir} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for dir (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableContainerSpec withDir(@Nullable String value) {
    if (Objects.equals(this.dir, value)) return this;
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        value,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerSpec#user() user} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for user (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableContainerSpec withUser(@Nullable String value) {
    if (Objects.equals(this.user, value)) return this;
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        value,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#groups() groups}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withGroups(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableContainerSpec(
          this.image,
          this.hostname,
          this.labels,
          this.command,
          this.args,
          this.env,
          this.dir,
          this.user,
          null,
          this.tty,
          this.mounts,
          this.stopGracePeriod,
          this.healthcheck,
          this.hosts,
          this.secrets,
          this.configs,
          this.dnsConfig,
          this.sysctls,
          this.init);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        newValue,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#groups() groups}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of groups elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withGroups(@Nullable Iterable<String> elements) {
    if (this.groups == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        newValue,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerSpec#tty() tty} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for tty (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableContainerSpec withTty(@Nullable Boolean value) {
    if (Objects.equals(this.tty, value)) return this;
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        value,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#mounts() mounts}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withMounts(@Nullable Mount... elements) {
    if (elements == null) {
      return new ImmutableContainerSpec(
          this.image,
          this.hostname,
          this.labels,
          this.command,
          this.args,
          this.env,
          this.dir,
          this.user,
          this.groups,
          this.tty,
          null,
          this.stopGracePeriod,
          this.healthcheck,
          this.hosts,
          this.secrets,
          this.configs,
          this.dnsConfig,
          this.sysctls,
          this.init);
    }
    @Nullable List<Mount> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        newValue,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#mounts() mounts}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of mounts elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withMounts(@Nullable Iterable<? extends Mount> elements) {
    if (this.mounts == elements) return this;
    @Nullable List<Mount> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        newValue,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerSpec#stopGracePeriod() stopGracePeriod} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for stopGracePeriod (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableContainerSpec withStopGracePeriod(@Nullable Long value) {
    if (Objects.equals(this.stopGracePeriod, value)) return this;
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        value,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerSpec#healthcheck() healthcheck} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for healthcheck (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableContainerSpec withHealthcheck(@Nullable Healthcheck value) {
    if (this.healthcheck == value) return this;
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        value,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#hosts() hosts}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withHosts(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableContainerSpec(
          this.image,
          this.hostname,
          this.labels,
          this.command,
          this.args,
          this.env,
          this.dir,
          this.user,
          this.groups,
          this.tty,
          this.mounts,
          this.stopGracePeriod,
          this.healthcheck,
          null,
          this.secrets,
          this.configs,
          this.dnsConfig,
          this.sysctls,
          this.init);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        newValue,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#hosts() hosts}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of hosts elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withHosts(@Nullable Iterable<String> elements) {
    if (this.hosts == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        newValue,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#secrets() secrets}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withSecrets(@Nullable SecretBind... elements) {
    if (elements == null) {
      return new ImmutableContainerSpec(
          this.image,
          this.hostname,
          this.labels,
          this.command,
          this.args,
          this.env,
          this.dir,
          this.user,
          this.groups,
          this.tty,
          this.mounts,
          this.stopGracePeriod,
          this.healthcheck,
          this.hosts,
          null,
          this.configs,
          this.dnsConfig,
          this.sysctls,
          this.init);
    }
    @Nullable List<SecretBind> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        newValue,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#secrets() secrets}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of secrets elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withSecrets(@Nullable Iterable<? extends SecretBind> elements) {
    if (this.secrets == elements) return this;
    @Nullable List<SecretBind> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        newValue,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#configs() configs}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withConfigs(@Nullable ConfigBind... elements) {
    if (elements == null) {
      return new ImmutableContainerSpec(
          this.image,
          this.hostname,
          this.labels,
          this.command,
          this.args,
          this.env,
          this.dir,
          this.user,
          this.groups,
          this.tty,
          this.mounts,
          this.stopGracePeriod,
          this.healthcheck,
          this.hosts,
          this.secrets,
          null,
          this.dnsConfig,
          this.sysctls,
          this.init);
    }
    @Nullable List<ConfigBind> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        newValue,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerSpec#configs() configs}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of configs elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withConfigs(@Nullable Iterable<? extends ConfigBind> elements) {
    if (this.configs == elements) return this;
    @Nullable List<ConfigBind> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        newValue,
        this.dnsConfig,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerSpec#dnsConfig() dnsConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for dnsConfig (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableContainerSpec withDnsConfig(@Nullable DnsConfig value) {
    if (this.dnsConfig == value) return this;
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        value,
        this.sysctls,
        this.init);
  }

  /**
   * Copy the current immutable object by replacing the {@link ContainerSpec#sysctls() sysctls} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the sysctls map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerSpec withSysctls(Map<String, ? extends String> entries) {
    if (this.sysctls == entries) return this;
    Map<String, String> newValue = createUnmodifiableMap(true, false, entries);
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        newValue,
        this.init);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerSpec#init() init} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for init (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableContainerSpec withInit(@Nullable Boolean value) {
    if (Objects.equals(this.init, value)) return this;
    return new ImmutableContainerSpec(
        this.image,
        this.hostname,
        this.labels,
        this.command,
        this.args,
        this.env,
        this.dir,
        this.user,
        this.groups,
        this.tty,
        this.mounts,
        this.stopGracePeriod,
        this.healthcheck,
        this.hosts,
        this.secrets,
        this.configs,
        this.dnsConfig,
        this.sysctls,
        value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableContainerSpec} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableContainerSpec
        && equalTo(0, (ImmutableContainerSpec) another);
  }

  private boolean equalTo(int synthetic, ImmutableContainerSpec another) {
    return image.equals(another.image)
        && Objects.equals(hostname, another.hostname)
        && labels.equals(another.labels)
        && Objects.equals(command, another.command)
        && Objects.equals(args, another.args)
        && Objects.equals(env, another.env)
        && Objects.equals(dir, another.dir)
        && Objects.equals(user, another.user)
        && Objects.equals(groups, another.groups)
        && Objects.equals(tty, another.tty)
        && Objects.equals(mounts, another.mounts)
        && Objects.equals(stopGracePeriod, another.stopGracePeriod)
        && Objects.equals(healthcheck, another.healthcheck)
        && Objects.equals(hosts, another.hosts)
        && Objects.equals(secrets, another.secrets)
        && Objects.equals(configs, another.configs)
        && Objects.equals(dnsConfig, another.dnsConfig)
        && sysctls.equals(another.sysctls)
        && Objects.equals(init, another.init);
  }

  /**
   * Computes a hash code from attributes: {@code image}, {@code hostname}, {@code labels}, {@code command}, {@code args}, {@code env}, {@code dir}, {@code user}, {@code groups}, {@code tty}, {@code mounts}, {@code stopGracePeriod}, {@code healthcheck}, {@code hosts}, {@code secrets}, {@code configs}, {@code dnsConfig}, {@code sysctls}, {@code init}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + image.hashCode();
    h += (h << 5) + Objects.hashCode(hostname);
    h += (h << 5) + labels.hashCode();
    h += (h << 5) + Objects.hashCode(command);
    h += (h << 5) + Objects.hashCode(args);
    h += (h << 5) + Objects.hashCode(env);
    h += (h << 5) + Objects.hashCode(dir);
    h += (h << 5) + Objects.hashCode(user);
    h += (h << 5) + Objects.hashCode(groups);
    h += (h << 5) + Objects.hashCode(tty);
    h += (h << 5) + Objects.hashCode(mounts);
    h += (h << 5) + Objects.hashCode(stopGracePeriod);
    h += (h << 5) + Objects.hashCode(healthcheck);
    h += (h << 5) + Objects.hashCode(hosts);
    h += (h << 5) + Objects.hashCode(secrets);
    h += (h << 5) + Objects.hashCode(configs);
    h += (h << 5) + Objects.hashCode(dnsConfig);
    h += (h << 5) + sysctls.hashCode();
    h += (h << 5) + Objects.hashCode(init);
    return h;
  }

  /**
   * Prints the immutable value {@code ContainerSpec} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ContainerSpec{"
        + "image=" + image
        + ", hostname=" + hostname
        + ", labels=" + labels
        + ", command=" + command
        + ", args=" + args
        + ", env=" + env
        + ", dir=" + dir
        + ", user=" + user
        + ", groups=" + groups
        + ", tty=" + tty
        + ", mounts=" + mounts
        + ", stopGracePeriod=" + stopGracePeriod
        + ", healthcheck=" + healthcheck
        + ", hosts=" + hosts
        + ", secrets=" + secrets
        + ", configs=" + configs
        + ", dnsConfig=" + dnsConfig
        + ", sysctls=" + sysctls
        + ", init=" + init
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ContainerSpec} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ContainerSpec instance
   */
  public static ImmutableContainerSpec copyOf(ContainerSpec instance) {
    if (instance instanceof ImmutableContainerSpec) {
      return (ImmutableContainerSpec) instance;
    }
    return ImmutableContainerSpec.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableContainerSpec ImmutableContainerSpec}.
   * <pre>
   * ImmutableContainerSpec.builder()
   *    .image(String) // required {@link ContainerSpec#image() image}
   *    .hostname(String | null) // nullable {@link ContainerSpec#hostname() hostname}
   *    .addLabel|putAllLabels(String =&gt; String) // {@link ContainerSpec#labels() labels} mappings
   *    .command(List&amp;lt;String&amp;gt; | null) // nullable {@link ContainerSpec#command() command}
   *    .args(List&amp;lt;String&amp;gt; | null) // nullable {@link ContainerSpec#args() args}
   *    .env(List&amp;lt;String&amp;gt; | null) // nullable {@link ContainerSpec#env() env}
   *    .dir(String | null) // nullable {@link ContainerSpec#dir() dir}
   *    .user(String | null) // nullable {@link ContainerSpec#user() user}
   *    .groups(List&amp;lt;String&amp;gt; | null) // nullable {@link ContainerSpec#groups() groups}
   *    .tty(Boolean | null) // nullable {@link ContainerSpec#tty() tty}
   *    .mounts(List&amp;lt;org.mandas.docker.client.messages.mount.Mount&amp;gt; | null) // nullable {@link ContainerSpec#mounts() mounts}
   *    .stopGracePeriod(Long | null) // nullable {@link ContainerSpec#stopGracePeriod() stopGracePeriod}
   *    .healthcheck(org.mandas.docker.client.messages.Healthcheck | null) // nullable {@link ContainerSpec#healthcheck() healthcheck}
   *    .hosts(List&amp;lt;String&amp;gt; | null) // nullable {@link ContainerSpec#hosts() hosts}
   *    .secrets(List&amp;lt;org.mandas.docker.client.messages.swarm.SecretBind&amp;gt; | null) // nullable {@link ContainerSpec#secrets() secrets}
   *    .configs(List&amp;lt;org.mandas.docker.client.messages.swarm.ConfigBind&amp;gt; | null) // nullable {@link ContainerSpec#configs() configs}
   *    .dnsConfig(org.mandas.docker.client.messages.swarm.DnsConfig | null) // nullable {@link ContainerSpec#dnsConfig() dnsConfig}
   *    .addSysctl|putAllSysctls(String =&gt; String) // {@link ContainerSpec#sysctls() sysctls} mappings
   *    .init(Boolean | null) // nullable {@link ContainerSpec#init() init}
   *    .build();
   * </pre>
   * @return A new ImmutableContainerSpec builder
   */
  public static ImmutableContainerSpec.Builder builder() {
    return new ImmutableContainerSpec.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableContainerSpec ImmutableContainerSpec}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ContainerSpec.Builder {
    private static final long INIT_BIT_IMAGE = 0x1L;
    private long initBits = 0x1L;

    private String image;
    private String hostname;
    private Map<String, String> labels = new LinkedHashMap<String, String>();
    private List<String> command = null;
    private List<String> args = null;
    private List<String> env = null;
    private String dir;
    private String user;
    private List<String> groups = null;
    private Boolean tty;
    private List<Mount> mounts = null;
    private Long stopGracePeriod;
    private Healthcheck healthcheck;
    private List<String> hosts = null;
    private List<SecretBind> secrets = null;
    private List<ConfigBind> configs = null;
    private DnsConfig dnsConfig;
    private Map<String, String> sysctls = new LinkedHashMap<String, String>();
    private Boolean init;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ContainerSpec} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ContainerSpec instance) {
      Objects.requireNonNull(instance, "instance");
      this.image(instance.image());
      @Nullable String hostnameValue = instance.hostname();
      if (hostnameValue != null) {
        hostname(hostnameValue);
      }
      putAllLabels(instance.labels());
      @Nullable List<String> commandValue = instance.command();
      if (commandValue != null) {
        addAllCommand(commandValue);
      }
      @Nullable List<String> argsValue = instance.args();
      if (argsValue != null) {
        addAllArgs(argsValue);
      }
      @Nullable List<String> envValue = instance.env();
      if (envValue != null) {
        addAllEnv(envValue);
      }
      @Nullable String dirValue = instance.dir();
      if (dirValue != null) {
        dir(dirValue);
      }
      @Nullable String userValue = instance.user();
      if (userValue != null) {
        user(userValue);
      }
      @Nullable List<String> groupsValue = instance.groups();
      if (groupsValue != null) {
        addAllGroups(groupsValue);
      }
      @Nullable Boolean ttyValue = instance.tty();
      if (ttyValue != null) {
        tty(ttyValue);
      }
      @Nullable List<Mount> mountsValue = instance.mounts();
      if (mountsValue != null) {
        addAllMounts(mountsValue);
      }
      @Nullable Long stopGracePeriodValue = instance.stopGracePeriod();
      if (stopGracePeriodValue != null) {
        stopGracePeriod(stopGracePeriodValue);
      }
      @Nullable Healthcheck healthcheckValue = instance.healthcheck();
      if (healthcheckValue != null) {
        healthcheck(healthcheckValue);
      }
      @Nullable List<String> hostsValue = instance.hosts();
      if (hostsValue != null) {
        addAllHosts(hostsValue);
      }
      @Nullable List<SecretBind> secretsValue = instance.secrets();
      if (secretsValue != null) {
        addAllSecrets(secretsValue);
      }
      @Nullable List<ConfigBind> configsValue = instance.configs();
      if (configsValue != null) {
        addAllConfigs(configsValue);
      }
      @Nullable DnsConfig dnsConfigValue = instance.dnsConfig();
      if (dnsConfigValue != null) {
        dnsConfig(dnsConfigValue);
      }
      putAllSysctls(instance.sysctls());
      @Nullable Boolean initValue = instance.init();
      if (initValue != null) {
        init(initValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerSpec#image() image} attribute.
     * @param image The value for image 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Image")
    public final Builder image(String image) {
      this.image = Objects.requireNonNull(image, "image");
      initBits &= ~INIT_BIT_IMAGE;
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerSpec#hostname() hostname} attribute.
     * @param hostname The value for hostname (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Hostname")
    public final Builder hostname(@Nullable String hostname) {
      this.hostname = hostname;
      return this;
    }

    /**
     * Put one entry to the {@link ContainerSpec#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          Objects.requireNonNull(value, value == null ? "labels value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link ContainerSpec#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link ContainerSpec#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(Map<String, ? extends String> entries) {
      this.labels.clear();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link ContainerSpec#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            Objects.requireNonNull(v, v == null ? "labels value for key: " + k : null));
      }
      return this;
    }

    /**
     * Adds one element to {@link ContainerSpec#command() command} list.
     * @param element A command element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder command(String element) {
      if (this.command == null) {
        this.command = new ArrayList<String>();
      }
      this.command.add(Objects.requireNonNull(element, "command element"));
      return this;
    }

    /**
     * Adds elements to {@link ContainerSpec#command() command} list.
     * @param elements An array of command elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder command(String... elements) {
      if (this.command == null) {
        this.command = new ArrayList<String>();
      }
      for (String element : elements) {
        this.command.add(Objects.requireNonNull(element, "command element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ContainerSpec#command() command} list.
     * @param elements An iterable of command elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Command")
    public final Builder command(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.command = null;
        return this;
      }
      this.command = new ArrayList<String>();
      return addAllCommand(elements);
    }

    /**
     * Adds elements to {@link ContainerSpec#command() command} list.
     * @param elements An iterable of command elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllCommand(Iterable<String> elements) {
      Objects.requireNonNull(elements, "command element");
      if (this.command == null) {
        this.command = new ArrayList<String>();
      }
      for (String element : elements) {
        this.command.add(Objects.requireNonNull(element, "command element"));
      }
      return this;
    }

    /**
     * Adds one element to {@link ContainerSpec#args() args} list.
     * @param element A args element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder arg(String element) {
      if (this.args == null) {
        this.args = new ArrayList<String>();
      }
      this.args.add(Objects.requireNonNull(element, "args element"));
      return this;
    }

    /**
     * Adds elements to {@link ContainerSpec#args() args} list.
     * @param elements An array of args elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder args(String... elements) {
      if (this.args == null) {
        this.args = new ArrayList<String>();
      }
      for (String element : elements) {
        this.args.add(Objects.requireNonNull(element, "args element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ContainerSpec#args() args} list.
     * @param elements An iterable of args elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Args")
    public final Builder args(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.args = null;
        return this;
      }
      this.args = new ArrayList<String>();
      return addAllArgs(elements);
    }

    /**
     * Adds elements to {@link ContainerSpec#args() args} list.
     * @param elements An iterable of args elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllArgs(Iterable<String> elements) {
      Objects.requireNonNull(elements, "args element");
      if (this.args == null) {
        this.args = new ArrayList<String>();
      }
      for (String element : elements) {
        this.args.add(Objects.requireNonNull(element, "args element"));
      }
      return this;
    }

    /**
     * Adds one element to {@link ContainerSpec#env() env} list.
     * @param element A env element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder env(String element) {
      if (this.env == null) {
        this.env = new ArrayList<String>();
      }
      this.env.add(Objects.requireNonNull(element, "env element"));
      return this;
    }

    /**
     * Adds elements to {@link ContainerSpec#env() env} list.
     * @param elements An array of env elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder env(String... elements) {
      if (this.env == null) {
        this.env = new ArrayList<String>();
      }
      for (String element : elements) {
        this.env.add(Objects.requireNonNull(element, "env element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ContainerSpec#env() env} list.
     * @param elements An iterable of env elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Env")
    public final Builder env(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.env = null;
        return this;
      }
      this.env = new ArrayList<String>();
      return addAllEnv(elements);
    }

    /**
     * Adds elements to {@link ContainerSpec#env() env} list.
     * @param elements An iterable of env elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllEnv(Iterable<String> elements) {
      Objects.requireNonNull(elements, "env element");
      if (this.env == null) {
        this.env = new ArrayList<String>();
      }
      for (String element : elements) {
        this.env.add(Objects.requireNonNull(element, "env element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerSpec#dir() dir} attribute.
     * @param dir The value for dir (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Dir")
    public final Builder dir(@Nullable String dir) {
      this.dir = dir;
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerSpec#user() user} attribute.
     * @param user The value for user (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("User")
    public final Builder user(@Nullable String user) {
      this.user = user;
      return this;
    }

    /**
     * Adds one element to {@link ContainerSpec#groups() groups} list.
     * @param element A groups element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder group(String element) {
      if (this.groups == null) {
        this.groups = new ArrayList<String>();
      }
      this.groups.add(Objects.requireNonNull(element, "groups element"));
      return this;
    }

    /**
     * Adds elements to {@link ContainerSpec#groups() groups} list.
     * @param elements An array of groups elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder groups(String... elements) {
      if (this.groups == null) {
        this.groups = new ArrayList<String>();
      }
      for (String element : elements) {
        this.groups.add(Objects.requireNonNull(element, "groups element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ContainerSpec#groups() groups} list.
     * @param elements An iterable of groups elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Groups")
    public final Builder groups(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.groups = null;
        return this;
      }
      this.groups = new ArrayList<String>();
      return addAllGroups(elements);
    }

    /**
     * Adds elements to {@link ContainerSpec#groups() groups} list.
     * @param elements An iterable of groups elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllGroups(Iterable<String> elements) {
      Objects.requireNonNull(elements, "groups element");
      if (this.groups == null) {
        this.groups = new ArrayList<String>();
      }
      for (String element : elements) {
        this.groups.add(Objects.requireNonNull(element, "groups element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerSpec#tty() tty} attribute.
     * @param tty The value for tty (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("TTY")
    public final Builder tty(@Nullable Boolean tty) {
      this.tty = tty;
      return this;
    }

    /**
     * Adds one element to {@link ContainerSpec#mounts() mounts} list.
     * @param element A mounts element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder mount(Mount element) {
      if (this.mounts == null) {
        this.mounts = new ArrayList<Mount>();
      }
      this.mounts.add(Objects.requireNonNull(element, "mounts element"));
      return this;
    }

    /**
     * Adds elements to {@link ContainerSpec#mounts() mounts} list.
     * @param elements An array of mounts elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder mounts(Mount... elements) {
      if (this.mounts == null) {
        this.mounts = new ArrayList<Mount>();
      }
      for (Mount element : elements) {
        this.mounts.add(Objects.requireNonNull(element, "mounts element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ContainerSpec#mounts() mounts} list.
     * @param elements An iterable of mounts elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Mounts")
    public final Builder mounts(@Nullable Iterable<? extends Mount> elements) {
      if (elements == null) {
        this.mounts = null;
        return this;
      }
      this.mounts = new ArrayList<Mount>();
      return addAllMounts(elements);
    }

    /**
     * Adds elements to {@link ContainerSpec#mounts() mounts} list.
     * @param elements An iterable of mounts elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllMounts(Iterable<? extends Mount> elements) {
      Objects.requireNonNull(elements, "mounts element");
      if (this.mounts == null) {
        this.mounts = new ArrayList<Mount>();
      }
      for (Mount element : elements) {
        this.mounts.add(Objects.requireNonNull(element, "mounts element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerSpec#stopGracePeriod() stopGracePeriod} attribute.
     * @param stopGracePeriod The value for stopGracePeriod (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("StopGracePeriod")
    public final Builder stopGracePeriod(@Nullable Long stopGracePeriod) {
      this.stopGracePeriod = stopGracePeriod;
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerSpec#healthcheck() healthcheck} attribute.
     * @param healthcheck The value for healthcheck (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Healthcheck")
    public final Builder healthcheck(@Nullable Healthcheck healthcheck) {
      this.healthcheck = healthcheck;
      return this;
    }

    /**
     * Adds one element to {@link ContainerSpec#hosts() hosts} list.
     * @param element A hosts element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder host(String element) {
      if (this.hosts == null) {
        this.hosts = new ArrayList<String>();
      }
      this.hosts.add(Objects.requireNonNull(element, "hosts element"));
      return this;
    }

    /**
     * Adds elements to {@link ContainerSpec#hosts() hosts} list.
     * @param elements An array of hosts elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder hosts(String... elements) {
      if (this.hosts == null) {
        this.hosts = new ArrayList<String>();
      }
      for (String element : elements) {
        this.hosts.add(Objects.requireNonNull(element, "hosts element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ContainerSpec#hosts() hosts} list.
     * @param elements An iterable of hosts elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Hosts")
    public final Builder hosts(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.hosts = null;
        return this;
      }
      this.hosts = new ArrayList<String>();
      return addAllHosts(elements);
    }

    /**
     * Adds elements to {@link ContainerSpec#hosts() hosts} list.
     * @param elements An iterable of hosts elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllHosts(Iterable<String> elements) {
      Objects.requireNonNull(elements, "hosts element");
      if (this.hosts == null) {
        this.hosts = new ArrayList<String>();
      }
      for (String element : elements) {
        this.hosts.add(Objects.requireNonNull(element, "hosts element"));
      }
      return this;
    }

    /**
     * Adds one element to {@link ContainerSpec#secrets() secrets} list.
     * @param element A secrets element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder secret(SecretBind element) {
      if (this.secrets == null) {
        this.secrets = new ArrayList<SecretBind>();
      }
      this.secrets.add(Objects.requireNonNull(element, "secrets element"));
      return this;
    }

    /**
     * Adds elements to {@link ContainerSpec#secrets() secrets} list.
     * @param elements An array of secrets elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder secrets(SecretBind... elements) {
      if (this.secrets == null) {
        this.secrets = new ArrayList<SecretBind>();
      }
      for (SecretBind element : elements) {
        this.secrets.add(Objects.requireNonNull(element, "secrets element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ContainerSpec#secrets() secrets} list.
     * @param elements An iterable of secrets elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Secrets")
    public final Builder secrets(@Nullable Iterable<? extends SecretBind> elements) {
      if (elements == null) {
        this.secrets = null;
        return this;
      }
      this.secrets = new ArrayList<SecretBind>();
      return addAllSecrets(elements);
    }

    /**
     * Adds elements to {@link ContainerSpec#secrets() secrets} list.
     * @param elements An iterable of secrets elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllSecrets(Iterable<? extends SecretBind> elements) {
      Objects.requireNonNull(elements, "secrets element");
      if (this.secrets == null) {
        this.secrets = new ArrayList<SecretBind>();
      }
      for (SecretBind element : elements) {
        this.secrets.add(Objects.requireNonNull(element, "secrets element"));
      }
      return this;
    }

    /**
     * Adds one element to {@link ContainerSpec#configs() configs} list.
     * @param element A configs element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder config(ConfigBind element) {
      if (this.configs == null) {
        this.configs = new ArrayList<ConfigBind>();
      }
      this.configs.add(Objects.requireNonNull(element, "configs element"));
      return this;
    }

    /**
     * Adds elements to {@link ContainerSpec#configs() configs} list.
     * @param elements An array of configs elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder configs(ConfigBind... elements) {
      if (this.configs == null) {
        this.configs = new ArrayList<ConfigBind>();
      }
      for (ConfigBind element : elements) {
        this.configs.add(Objects.requireNonNull(element, "configs element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ContainerSpec#configs() configs} list.
     * @param elements An iterable of configs elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Configs")
    public final Builder configs(@Nullable Iterable<? extends ConfigBind> elements) {
      if (elements == null) {
        this.configs = null;
        return this;
      }
      this.configs = new ArrayList<ConfigBind>();
      return addAllConfigs(elements);
    }

    /**
     * Adds elements to {@link ContainerSpec#configs() configs} list.
     * @param elements An iterable of configs elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllConfigs(Iterable<? extends ConfigBind> elements) {
      Objects.requireNonNull(elements, "configs element");
      if (this.configs == null) {
        this.configs = new ArrayList<ConfigBind>();
      }
      for (ConfigBind element : elements) {
        this.configs.add(Objects.requireNonNull(element, "configs element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerSpec#dnsConfig() dnsConfig} attribute.
     * @param dnsConfig The value for dnsConfig (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("DNSConfig")
    public final Builder dnsConfig(@Nullable DnsConfig dnsConfig) {
      this.dnsConfig = dnsConfig;
      return this;
    }

    /**
     * Put one entry to the {@link ContainerSpec#sysctls() sysctls} map.
     * @param key The key in the sysctls map
     * @param value The associated value in the sysctls map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addSysctl(String key, String value) {
      this.sysctls.put(
          Objects.requireNonNull(key, "sysctls key"),
          Objects.requireNonNull(value, value == null ? "sysctls value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link ContainerSpec#sysctls() sysctls} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addSysctl(Map.Entry<String, ? extends String> entry) {
      String k = entry.getKey();
      String v = entry.getValue();
      this.sysctls.put(
          Objects.requireNonNull(k, "sysctls key"),
          Objects.requireNonNull(v, v == null ? "sysctls value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link ContainerSpec#sysctls() sysctls} map. Nulls are not permitted
     * @param entries The entries that will be added to the sysctls map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Sysctls")
    public final Builder sysctls(Map<String, ? extends String> entries) {
      this.sysctls.clear();
      return putAllSysctls(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link ContainerSpec#sysctls() sysctls} map. Nulls are not permitted
     * @param entries The entries that will be added to the sysctls map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllSysctls(Map<String, ? extends String> entries) {
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.sysctls.put(
            Objects.requireNonNull(k, "sysctls key"),
            Objects.requireNonNull(v, v == null ? "sysctls value for key: " + k : null));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerSpec#init() init} attribute.
     * @param init The value for init (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Init")
    public final Builder init(@Nullable Boolean init) {
      this.init = init;
      return this;
    }

    /**
     * Builds a new {@link ImmutableContainerSpec ImmutableContainerSpec}.
     * @return An immutable instance of ContainerSpec
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableContainerSpec build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableContainerSpec(
          image,
          hostname,
          createUnmodifiableMap(false, false, labels),
          command == null ? null : createUnmodifiableList(true, command),
          args == null ? null : createUnmodifiableList(true, args),
          env == null ? null : createUnmodifiableList(true, env),
          dir,
          user,
          groups == null ? null : createUnmodifiableList(true, groups),
          tty,
          mounts == null ? null : createUnmodifiableList(true, mounts),
          stopGracePeriod,
          healthcheck,
          hosts == null ? null : createUnmodifiableList(true, hosts),
          secrets == null ? null : createUnmodifiableList(true, secrets),
          configs == null ? null : createUnmodifiableList(true, configs),
          dnsConfig,
          createUnmodifiableMap(false, false, sysctls),
          init);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_IMAGE) != 0) attributes.add("image");
      return "Cannot build ContainerSpec, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
